#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: ugroup.pm 390 2007-03-20 14:06:06Z hikarin $
#

package Zeromin2::App::ugroup;

use strict;

sub load {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value( 1, [] );
    return $zApp->return_value( 0,
        $zApp->user()->get_groups_with_page( $zApp->page_param() ) );
}

sub save {
    my ($zApp) = @_;
    $zApp->check_csrf() or return $zApp->return_value(1);
    $zApp->privilege('can_edit_ugroup')
        or $zApp->privilege('can_create_ugroup')
        or return $zApp->return_value(1);

    my $zUser    = $zApp->user();
    my $iRequest = $zApp->request();
    my $error    = {};
    my $param    = {};

    _validate_ugroup_param( $zApp, $zUser, $param, $error )
        or return $zApp->return_value( $error->{code} );
    my $id           = $iRequest->param_int('id');
    my $ugroup       = $zUser->get_group($id);
    my $name         = $param->{name};
    my $bbs          = $param->{bbs};
    my $user_priv    = $param->{user_priv};
    my $cap_priv     = $param->{cap_priv};
    my $bbs_priv     = $param->{bbs_priv};
    my $subject_priv = $param->{subject_priv};
    my $setting_priv = $param->{setting_priv};
    my $thread_priv  = $param->{thread_priv};
    my $pool_priv    = $param->{pool_priv};
    my $archive_priv = $param->{archive_priv};
    my $res_priv     = $param->{res_priv};
    my $meta_priv    = $param->{meta_priv};
    my $cntl_priv    = $param->{cntl_priv};
    my $cat_priv     = $param->{cat_priv};
    my $view_priv    = $param->{view_priv};
    my $plugin_priv  = $param->{plugin_priv};

    if ( $ugroup->{id} ) {
        $zApp->privilege('can_edit_ugroup')
            or return $zApp->return_value(1);
        $zUser->edit_group(
            $id,
            {   name      => $name,
                bbs       => $bbs,
                privilege => {
                    user     => $user_priv,
                    cap      => $cap_priv,
                    bbs      => $bbs_priv,
                    subject  => $subject_priv,
                    setting  => $setting_priv,
                    thread   => $thread_priv,
                    pool     => $pool_priv,
                    archive  => $archive_priv,
                    res      => $res_priv,
                    meta     => $meta_priv,
                    control  => $cntl_priv,
                    category => $cat_priv,
                    view     => $view_priv,
                    plugin   => $plugin_priv,
                },
            }
        );
        $zUser->save();
        $zApp->logger( 1, 'EDIT USER GROUP: %s belongs %d', [ $name, $bbs ],
        );
        $zApp->add_template_param( { Done_ugroup_edit => 1 } );
        $zApp->return_value(0);
    }
    else {
        $zApp->privilege('can_create_ugroup')
            or return $zApp->return_value(1);
        $zUser->add_group(
            {   name      => $name,
                bbs       => $bbs,
                privilege => {
                    user     => $user_priv,
                    cap      => $cap_priv,
                    bbs      => $bbs_priv,
                    subject  => $subject_priv,
                    setting  => $setting_priv,
                    thread   => $thread_priv,
                    pool     => $pool_priv,
                    archive  => $archive_priv,
                    res      => $res_priv,
                    meta     => $meta_priv,
                    control  => $cntl_priv,
                    category => $cat_priv,
                    view     => $view_priv,
                    plugin   => $plugin_priv,
                },
            }
        );
        $zUser->save();
        $zApp->logger(
            1,
            'CREATE USER GROUP: %s belongs %d',
            [ $name, $bbs ],
        );
        $zApp->add_template_param( { Done_ugroup_create => 1 } );
        $zApp->return_value( 0, $zUser->get_group_id($name) );
    }
}

sub remove {
    my ($zApp) = @_;
    $zApp->check_csrf() or return $zApp->return_value(1);
    $zApp->privilege('can_remove_ugroup')
        or return $zApp->return_value(1);

    my $zUser  = $zApp->user();
    my $status = {};
    my @ids    = $zApp->request()->param('id');
    scalar @ids or return $zApp->return_value(2);

    for my $id (@ids) {
        $id = Img0ch::Kernel::intval($id);
        $id == 1 and next;
        if ( $zUser->remove_group($id) ) {
            $zApp->logger( 1, 'REMOVE USER GROUP: %s', [$id] );
            $status->{$id} = 1;
        }
        else {
            $zApp->logger( 0, 'TRIED REMOVING INEXIST USER GROUP: \%s',
                [$id] );
            $status->{$id} = 0;
        }
    }

    $zUser->save();
    return $zApp->return_value( 0, $status );
}

sub create {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value(1);

    require Zeromin::BBS;
    my $zBBS = Zeromin::BBS->new( $zApp->kernel(), { bbs => '' } );
    my ( $unijp, $encoding ) = $zApp->unijp();

    $zApp->add_template_param( { BBSList => $zBBS->all($unijp) } );
    return $zApp->return_value(0);
}

sub edit {
    my ($zApp) = @_;
    $zApp->privilege('can_enter_user_section')
        or return $zApp->return_value(1);

    require Zeromin::BBS;
    my $zUser = $zApp->user();
    my $group = $zUser->get_group( $zApp->request()->param('id') );
    my $zBBS  = Zeromin::BBS->new( $zApp->kernel(), { bbs => '' } );

    my ( $unijp, $encoding ) = $zApp->unijp();
    my $allBBS     = $zBBS->all($unijp);
    my $allBelongs = $group->{bbs};
    for my $bbs (@$allBBS) {
        for my $belong (@$allBelongs) {
            $bbs->{id} == $belong and $bbs->{selected} = 1;
        }
    }

    my $privilege = {};
    my $priv      = $group->{privilege};
    while ( my ( $name, $value ) = each %$priv ) {
        for my $privilege_bit ( 1, 2, 4, 8, 16, 32, 64 ) {
            $privilege->{"Privilege_${name}_${privilege_bit}"}
                = $value & $privilege_bit ? 1 : 0;
        }
    }

    $zApp->add_template_param(
        {   BBSList  => $allBBS,
            IsGlobal => ( $allBelongs->[0] == 0 ? 1 : 0 ),
            %$privilege,
            %$group
        }
    );
    return $zApp->return_value(0);
}

sub _validate_ugroup_param {
    my ( $zApp, $zUser, $param, $error ) = @_;
    my $iRequest = $zApp->request();

    my $name = $iRequest->param('name');
    if ( $name eq '' ) {
        $zApp->logger( 0, 'NO USER GROUP NAME WAS SET' );
        %$error = ( code => 2 );
        return 0;
    }
    if ( $name eq 'Administrators' ) {
        $zApp->logger( 0, 'CANNOT EDIT "Administrators"' );
        %$error = ( code => 3 );
        return 0;
    }

    my @bbs = $iRequest->param('bbs');
    $iRequest->param('all') and @bbs = (0);
    $param->{name}         = $name;
    $param->{bbs}          = \@bbs;
    $param->{user_priv}    = _privilege( $iRequest, 'user_priv' );
    $param->{cap_priv}     = _privilege( $iRequest, 'cap_priv' );
    $param->{bbs_priv}     = _privilege( $iRequest, 'bbs_priv' );
    $param->{subject_priv} = _privilege( $iRequest, 'subject_priv' );
    $param->{setting_priv} = _privilege( $iRequest, 'setting_priv' );
    $param->{thread_priv}  = _privilege( $iRequest, 'thread_priv' );
    $param->{pool_priv}    = _privilege( $iRequest, 'pool_priv' );
    $param->{archive_priv} = _privilege( $iRequest, 'archive_priv' );
    $param->{res_priv}     = _privilege( $iRequest, 'res_priv' );
    $param->{meta_priv}    = _privilege( $iRequest, 'meta_priv' );
    $param->{cntl_priv}    = _privilege( $iRequest, 'control_priv' );
    $param->{cat_priv}     = _privilege( $iRequest, 'category_priv' );
    $param->{view_priv}    = _privilege( $iRequest, 'view_priv' );
    $param->{plugin_priv}  = _privilege( $iRequest, 'plugin_priv' );
    1;
}

sub _privilege {
    my ( $iRequest, $name ) = @_;
    my @param = $iRequest->param( $name, 1 );
    my $ret = 0;
    map { $ret += Img0ch::Kernel::intval($_) } @param;
    return $ret;
}

1;
__END__
