/*
 * $Id: Builder.java,v 1.5 2003/04/06 01:50:50 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.converter;

import java.io.*;
import java.util.Properties;
import java.util.List;
import java.util.ArrayList;
import java.util.zip.Deflater;
import jp.sourceforge.imodoki.util.ExecUtils;
import jp.sourceforge.imodoki.util.FileUtils;
import jp.sourceforge.imodoki.util.StringUtils;
import jp.sourceforge.imodoki.util.JavaCompiler;
import jp.sourceforge.imodoki.util.OptionParser;
import jp.sourceforge.imodoki.util.OptionParserException;
import jp.sourceforge.imodoki.shrinker.Shrinker;

/**
 * ӥɤԤʤ
 */
public class Builder {
    /* ޥǽ */
    private boolean m_compilerVerbose  = false;
    private boolean m_compilerDebug    = true;
    private boolean m_compilerOptimize = true;
    private String  m_compilerTarget   = "1.1";
    private String  m_compilerEncoding = "EUC-JP";
    private String  m_manifestEncoding = "UTF-8";
    private String  m_jadFileEncoding  = "UTF-8";
    private boolean m_shrinkerEnabled  = true;

    /* ץꥱͭΥץѥƥ */
    private File m_baseDir;
    private File m_binDir;
    private File m_iappliDir;
    private File m_srcDir;
    private File m_classesDir;
    private File m_unjarredDir;
    private File m_resDir;
    private File m_editedDir;
    private File m_preverifiedDir;
    private File m_manifestFile;
    private File m_imodokiHome;
    private String m_enabledOptions;
    private String m_disabledOptions;
    private String m_iappliAppClass;
    private String m_appBaseName;
    private File m_appJarFile;
    private File m_appJadFile;
    private File m_outputDir;

    private File m_j2mewtkHome;
    private String m_bootClassPath;
    private String m_preverifier;

    /* verbose ٥ */
    private static final int QUIET   = 0;
    private static final int TERSE   = 1;
    private static final int VERBOSE = 2;
    private static final int DEBUG   = 3;

    private int m_verbosity;

    /**
     * 󥹥ȥ饯
     *
     * @param  baseDir    ȥǥ쥯ȥꡣ
     * @param  verbosity  verbose ٥롣
     */
    public Builder(File baseDir, int verbosity) {
        m_baseDir = baseDir;
        m_verbosity = verbosity;

        m_binDir         = new File(baseDir, "bin");
        m_iappliDir      = new File(baseDir, "iappli");
        m_srcDir         = new File(baseDir, "src");
        m_classesDir     = new File(baseDir, "classes");
        m_unjarredDir    = new File(baseDir, "unjarred");
        m_resDir         = new File(baseDir, "res");
        m_editedDir      = new File(baseDir, "edited");
        m_preverifiedDir = new File(baseDir, "preverified");

        m_manifestFile = new File(m_binDir, "MANIFEST.MF");

        if (m_verbosity >= DEBUG) {
            System.setProperty("jp.sourceforge.imodoki.debug", "true");
        }
    }

    /**
     * ӥɤԤʤ
     *
     * @return  ӥɤ true
     */
    public boolean build() {
        try {

            System.out.println();
            System.out.println("[Compilation phase]");
            System.out.println();

            /* ץѥƥɤ߹ */
            loadBuildProperties();
            loadSystemProperties();

            /* ȥǥ쥯ȥκ */
            mkdirs(m_srcDir);
            mkdirs(m_classesDir);
            mkdirs(m_unjarredDir);
            mkdirs(m_resDir);
            mkdirs(m_editedDir);
            mkdirs(m_preverifiedDir);

            /* 󥿥饤֥Υѥ */
            compileRuntime();

            /* iץ jar Ÿ */
            File iappliJar = new File(m_iappliDir, m_appBaseName + ".jar");
            try {
                FileUtils.expandZip(iappliJar, m_unjarredDir);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to extract `" + iappliJar + "'", ioe);
            }

            /* 饹եԽ */
            editClasses();

            /* ץ٥ե */
            preverify();

            /* ꥽եѴ */
            convertResources();

            /* jar ե */
            makeJar();

            if (m_shrinkerEnabled) {
                System.out.println();
                System.out.println("[Shrinking phase]");
                System.out.println();

                /* Shrinker ư */
                shrink();
            }

            /* jad ե */
            makeJad();

        } catch (BuildException be) {
            System.err.println(be.getMessage());
            return false;
        }

        return true;
    }

    /** ӥɸͭΥץѥƥɤ߹ࡣ */
    private void loadBuildProperties() throws BuildException {
        Properties props = new Properties();

        File buildPropsFile = new File(m_baseDir, "build.properties");
        try {
            loadProperties(props, buildPropsFile);
        } catch (IOException ioe) {
            throw new BuildException(
                "Failed to read `" + buildPropsFile + "'", ioe);
        }

        String str = requireProperty(props, "imodoki.home", buildPropsFile);
        m_imodokiHome = FileUtils.decodeAntLikePath(str);
        m_enabledOptions =
            requireProperty(props, "enabled.options", buildPropsFile);
        m_disabledOptions =
            requireProperty(props, "disabled.options", buildPropsFile);
        m_iappliAppClass =
            requireProperty(props, "iappli.app.class", buildPropsFile);
        m_appBaseName =
            requireProperty(props, "app.basename", buildPropsFile);
        m_outputDir =
            new File(requireProperty(props, "output.dir", buildPropsFile));

        m_appJarFile = new File(m_binDir, m_appBaseName + ".jar");
        m_appJadFile = new File(m_binDir, m_appBaseName + ".jad");
    }

    /** ƥ(= iɤ) ϢΥץѥƥɤ߹ࡣ */
    private void loadSystemProperties() throws BuildException {
        Properties props = new Properties();

        File systemPropsFile = new File(m_imodokiHome, "env.properties");
        if (m_verbosity >= DEBUG) {
            System.out.println("System property file: " + systemPropsFile);
        }
        if (systemPropsFile.exists()) {
            try {
                loadProperties(props, systemPropsFile);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to read `" + systemPropsFile + "'", ioe);
            }
        }

        /* 桼Τۤͥ褹 (顢ɤ) */
        File userPropsFile = new File(System.getProperty("user.home"),
                                      ".imodoki-env.properties");
        if (m_verbosity >= DEBUG) {
            System.out.println("User property file: " + userPropsFile);
        }
        if (userPropsFile.exists()) {
            try {
                loadProperties(props, userPropsFile);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to read `" + userPropsFile + "'", ioe);
            }
        }

        String str = requireProperty(props, "j2mewtk.home", systemPropsFile);
        m_j2mewtkHome = FileUtils.decodeAntLikePath(str);

        File midpapiZip =
            FileUtils.newFile3(m_j2mewtkHome, "lib", "midpapi.zip");
        if (!midpapiZip.isFile()) {
            throw new BuildException("File `" + midpapiZip +
                                     "' does not exist. Please check " +
                                     "the property `j2mewtk.home' in " +
                                     systemPropsFile);
        }

        m_bootClassPath = midpapiZip.getPath();
        m_preverifier =
            FileUtils.newFile3(m_j2mewtkHome, "bin", "preverify").getPath();
    }

    private static String requireProperty(Properties props, String key,
                                          File file) throws BuildException {
        String value = props.getProperty(key);
        if (value == null) {
            throw new BuildException(
                "Property `" + key + "' is not set in file `" + file + "'");
        }

        return value;
    }

    /** 󥿥饤֥򥳥ѥ뤹롣 */
    // TODO: ॹפѤƤʤ٤ɬפʥѥ򤷤ʤ
    //       褦ˤ
    private void compileRuntime() throws BuildException {
        /* եΥԡ */
        System.out.println("Copying runtime source...");
        try {
            File runtimeSrcDir =
                new File(new File(m_imodokiHome, "runtime"), "src");
            FileUtils.copyDirectory(runtimeSrcDir, m_srcDir);
        } catch (IOException ioe) {
            throw new BuildException("Failed to copy runtime source", ioe);
        }

        /* JEnable ƤӽФ */
        System.out.println("Preprocessing runtime source...");
        String dir;
        try {
            dir = m_srcDir.getCanonicalPath();
        } catch (IOException ioe) {
            throw new BuildException("Failed to get canonical path", ioe);
        }
        String[] jenableArgs = new String[] {
            "-e", m_enabledOptions,
            "-d", m_disabledOptions,
            "-c", m_compilerEncoding,
            dir.replace(File.separatorChar, '/') + "/**/*",
        };
        // TODO: 顼
        JEnable.main(jenableArgs);

        /* Java ѥƤӽФ */
        System.out.println("Compiling runtime source...");
        boolean success;
        try {
            JavaCompiler compiler = new JavaCompiler();
            compiler.setSrcDir(m_srcDir);
            compiler.setDestDir(m_classesDir);
            compiler.setBootClassPath(m_bootClassPath);
            compiler.setClassPath(m_classesDir.getPath());
            compiler.setVerbose(m_compilerVerbose);
            compiler.setOptimize(m_compilerOptimize);
            compiler.setDebug(m_compilerDebug);
            compiler.setTarget(m_compilerTarget);
            compiler.setEncoding(m_compilerEncoding);
            success = compiler.compile();
        } catch (IOException ioe) {
            throw new BuildException("Failed to invoke compiler", ioe);
        }

        if (!success) {
            throw new BuildException("Compilation failed");
        }
    }

    /** 饹եԽ롣 */
    private void editClasses() throws BuildException {
        System.out.println("Editing class files...");

        /*
         * 󥿥饤֥Խ
         *
         * 󥿥饤֥ classDir  editedDir ˥ԡʤ顢
         * MainMIDlet ˴ޤޤ IAppStub 饹ؤλȤ
         * iץΥ饹֤롣
         */
        String[] fnames =
            FileUtils.listRecursive(m_classesDir,
                                    FileUtils.CLASSFILENAME_FILTER);
        for (int i = 0; i < fnames.length; i++) {
            String fname = fnames[i];
            File srcFile = new File(m_classesDir, fname);
            File destFile = new File(m_editedDir, fname);
            mkdirs(destFile.getParentFile());
            String canonName = fname.replace(File.separatorChar, '/');
            try {
                if (canonName.equals("imodoki/IAppStub.class")) {
                    /* do nothing */
                } else if (canonName.equals("imodoki/MainMIDlet.class")) {
                    ClassNameReplacer replacer =
                        new ClassNameReplacer(srcFile);
                    replacer.replace("imodoki.IAppStub", m_iappliAppClass);
                    replacer.dump(destFile);
                } else {
                    FileUtils.copyFile(srcFile, destFile);
                }
            } catch (IOException ioe) {
                throw new BuildException("Failed to copy/edit file from `" +
                                         srcFile + "' to `" +
                                         destFile + "'", ioe);
            }
        }

        /*
         * iץꥯ饹Խ
         *
         * iץꥯ饹 unjarredDir  editedDir ˥ԡʤ顢
         * Connector 饹ȼΤΤ֤롣
         */
        fnames = FileUtils.listRecursive(m_unjarredDir,
                                         FileUtils.CLASSFILENAME_FILTER);
        for (int i = 0; i < fnames.length; i++) {
            String fname = fnames[i];
            File srcFile = new File(m_unjarredDir, fname);
            File destFile = new File(m_editedDir, fname);
            mkdirs(destFile.getParentFile());
            try {
                ClassNameReplacer replacer = new ClassNameReplacer(srcFile);
                replacer.replace("javax.microedition.io.Connector",
                                 "imodoki.wrapper.Connector");
                replacer.dump(destFile);
            } catch (IOException ioe) {
                throw new BuildException("Failed to edit file from `" +
                                         srcFile + "' to `" +
                                         destFile + "'", ioe);
            }
        }
    }

    /** ץ٥ե */
    private void preverify() throws BuildException {
        System.out.println("Preverifying class files...");

        String[] cmdarray = new String[] {
            m_preverifier,
            "-classpath",
            m_bootClassPath,
            "-d",
            m_preverifiedDir.getPath(),
            m_editedDir.getPath(),
        };

        try {
            int exitCode = ExecUtils.execCommand(cmdarray);
            if (exitCode != 0) {
                throw new BuildException("Preverify failed (exit code " +
                                         exitCode + ")");
            }
        } catch (IOException ioe) {
            throw new BuildException("Preverify failed", ioe);
        }
    }

    /** ꥽Ѵ롣 */
    private void convertResources() throws BuildException {
        System.out.println("Converting resource files...");

        String[] fnames =
            FileUtils.listRecursive(m_unjarredDir,
                                    FileUtils.RESOURCE_FILENAME_FILTER);
        for (int i = 0; i < fnames.length; i++) {
            String fname = fnames[i];
            File srcFile = new File(m_unjarredDir, fname);
            File destFile = new File(m_resDir, fname);
            mkdirs(destFile.getParentFile());
            try {
                if (StringUtils.endsWithIgnoreCase(fname, ".gif")) {
                    ImageConverter converter = ImageConverter.getInstance();
                    converter.setOutputFormat("PNG");
                    converter.convert(srcFile, destFile);
                } else {
                    FileUtils.copyFile(srcFile, destFile);
                }
            } catch (Exception e) {
                throw new BuildException("Failed to copy/convert file from `" +
                                         srcFile + "' to `" +
                                         destFile + "'", e);
            }
        }
    }

    /** jar ե롣 */
    private void makeJar() throws BuildException {
        System.out.println("Making jar file...");

        try {
            FileUtils.createJar(m_appJarFile, m_preverifiedDir, null,
                                m_manifestFile, Deflater.DEFAULT_COMPRESSION);
            FileUtils.updateJar(m_appJarFile, m_resDir, null);
        } catch (IOException ioe) {
            throw new BuildException("Failed to make jar file `" +
                                     m_appJarFile + "'", ioe);
        }
    }

    /** jad ե롣 */
    private void makeJad() throws BuildException {
        System.out.println("Making jad file...");

        long fileSize = m_appJarFile.length();

        /*
         * Manifest եƤ򥳥ԡ
         * MIDlet-Jar-Size  MIDlet-Jar-URL եɤɲä롣
         */
        try {
            FileInputStream fis = new FileInputStream(m_manifestFile);
            InputStreamReader isr =
                new InputStreamReader(fis, m_manifestEncoding);
            BufferedReader reader = new BufferedReader(isr);

            FileOutputStream fos = new FileOutputStream(m_appJadFile);
            OutputStreamWriter osw =
                new OutputStreamWriter(fos, m_jadFileEncoding);
            PrintWriter writer = new PrintWriter(osw);

            String line;
            while ((line = reader.readLine()) != null) {
                writer.println(line);
            }
            reader.close();

            writer.println("MIDlet-Jar-URL: " + m_appBaseName + ".jar");
            writer.println("MIDlet-Jar-Size: " + fileSize);

            writer.close();

        } catch (IOException ioe) {
            throw new BuildException("Failed to create jad file `" +
                                     m_appJadFile + "'", ioe);
        }
    }

    /** Shrinker ƤӽФ */
    private void shrink() throws BuildException {
        System.out.println("Shrinking jar file...");

        File orgFile = new File(m_appJarFile.getPath() + ".org");
        orgFile.delete();
        if (m_appJarFile.renameTo(orgFile) == false) {
            throw new BuildException("Failed to rename from `" +
                                     m_appJarFile + "' to `" + orgFile + "'");
        }

        System.setProperty("imodoki.exttools.dir",
                           m_imodokiHome.getPath() + File.separatorChar +
                           "ext-tools");

        List args = new ArrayList();
        args.add("-classpath");
        args.add(m_bootClassPath);
        args.add("-injar");
        args.add(orgFile.getPath());
        args.add("-outjar");
        args.add(m_appJarFile.getPath());
        args.add("-preservemidlet");
        args.add("-preverifier");
        args.add(m_preverifier);
        if (m_verbosity >= DEBUG) {
            args.add("-debug");
        } else if (m_verbosity >= VERBOSE) {
            args.add("-verbose");
        }

        String[] args_ary = (String[])args.toArray(new String[0]);

        if (Shrinker.internalMain(args_ary) == false) {
            throw new BuildException("Shrinking failed");
        }

        System.out.println("Shrinking jar file...done");
    }

    /**
     * ǥХåѥӥɤԤʤ
     *
     * @return  ӥɤ true
     */
    public boolean debugBuild() {
        m_shrinkerEnabled = false;

        return build();
    }

    /**
     * ӥɤˤä줿ե롣
     *
     * @return   true
     */
    public boolean clean() {
        System.out.println("Cleaning work directory...");

        try {

            /* ץѥƥɤ߹ */
            loadBuildProperties();

        } catch (BuildException be) {
            System.err.println(be.getMessage());
            return false;
        }

        /*  */

        FileUtils.deleteRecursive(m_srcDir);
        FileUtils.deleteRecursive(m_classesDir);
        FileUtils.deleteRecursive(m_unjarredDir);
        FileUtils.deleteRecursive(m_resDir);
        FileUtils.deleteRecursive(m_editedDir);
        FileUtils.deleteRecursive(m_preverifiedDir);

        m_appJarFile.delete();
        File orgFile = new File(m_appJarFile.getPath() + ".org");
        orgFile.delete();
        m_appJadFile.delete();

        return true;
    }

    /**
     * ӥɤˤä줿եϥǥ쥯ȥ˥ԡ롣
     *
     * @return   true
     */
    public boolean output() {
        System.out.println("Copying output files...");

        File srcFile = null, destFile = null;
        try {

            /* ץѥƥɤ߹ */
            loadBuildProperties();

            /* ԡ */

            mkdirs(m_outputDir);

            srcFile = m_appJarFile;
            destFile = new File(m_outputDir, m_appBaseName + ".jar");
            FileUtils.copyFile(srcFile, destFile);

            srcFile = m_appJadFile;
            destFile = new File(m_outputDir, m_appBaseName + ".jad");
            FileUtils.copyFile(srcFile, destFile);

        } catch (IOException ioe) {
            BuildException be =
                new BuildException("Failed to copy file from `" +
                                   srcFile + "' to `" +
                                   destFile + "'", ioe);
            System.err.println(be.getMessage());
            return false;
        } catch (BuildException be) {
            System.err.println(be.getMessage());
            return false;
        }

        return true;
    }

    /**
     * ȥǥ쥯ȥ꼫Τ롣
     *
     * @return   true
     */
    public boolean removeWorkDir() {
        System.out.println("Removing work directory...");

        /*  */
        boolean success = FileUtils.deleteRecursive(m_baseDir);
        if (!success) {
            BuildException be =
                new BuildException("Failed to remove directory `" +
                                   m_baseDir + "'");
            System.err.println(be.getMessage());
            return false;
        }

        return true;
    }

    /**
     * 顼դ File#mkdirs()
     * ǥ쥯ȥ꤬¸ξˤϥ顼Фʤ
     */
    private static void mkdirs(File dir) throws BuildException {
        if (dir.isDirectory())
            return;

        if (dir.mkdirs() == false) {
            throw new BuildException(
                "Failed to create directory `" + dir + "'");
        }
    }

    /** ץѥƥեɤࡣ */
    private static void loadProperties(Properties props, File file)
        throws IOException {
        InputStream is = new FileInputStream(file);
        try {
            props.load(is);
        } finally {
            is.close();
        }
    }

    /** ¼Υᥤ */
    public static boolean internalMain(String[] args) {
        File baseDir = new File(".");
        boolean verbose = false, debug = false;

        OptionParser optParser = new OptionParser();
        optParser.addOption("-basedir", "<dir>",
                            "Specify base directory (default: .)");
//         optParser.addOption("-quiet",
//                             "Silent execution");
        optParser.addOption("-verbose",
                            "Verbose execution");
        optParser.addOption("-debug",
                            "Print debugging information");
        optParser.addOption("-help",
                            "Show this help");

        try {
            String opt;
            while ((opt = optParser.parse(args)) != null) {
                if (opt.equals("-basedir")) {
                    baseDir = new File(optParser.getOptionArg());
                } else if (opt.equals("-verbose")) {
                    verbose = true;
                } else if (opt.equals("-debug")) {
                    debug = true;
                } else if (opt.equals("-help")) {
                    usage(optParser);
                    return true;
                } else {
                    throw new RuntimeException("Unreachable");
                }
            }
        } catch (OptionParserException ope) {
            System.err.println(ope.getMessage());
            return false;
        }

        int verbosity = TERSE;
        if (debug) {
            verbosity = DEBUG;
        } else if (verbose) {
            verbosity = VERBOSE;
        }

        Builder builder =
            new Builder(baseDir, verbosity);

        boolean success = false;
        String[] restArgs = optParser.getArgs();
        if (restArgs.length == 0) {
            /* default action */
            success = builder.build();
        } else {
            for (int i = 0; i < restArgs.length; i++) {
                String target = restArgs[i];

                if (target.equals("build")) {
                    success = builder.build();
                } else if (target.equals("debugbuild")) {
                    success = builder.debugBuild();
                } else if (target.equals("clean")) {
                    success = builder.clean();
                } else if (target.equals("output")) {
                    success = builder.output();
                } else if (target.equals("remove-workdir")) {
                    success = builder.removeWorkDir();
                } else {
                    System.err.println("Unknown target: " + target);
                    success = false;
                }

                if (!success) {
                    break;
                }
            }
        }

        System.out.println();
        if (success) {
            System.out.println("BUILD SUCCEESSFUL");
        } else {
            System.out.println("BUILD FAILED");
        }
        return success;
    }

    /** ᥤ */
    public static void main(String[] args) {
        boolean success = internalMain(args);
        System.exit(success ? 0 : 1);
    }

    private static void usage(OptionParser optParser) {
        System.out.println("usage: builder [options...] [targets...]");
        System.out.println();
        System.out.println("options:");
        optParser.printUsage(System.out, 2, 20);
        System.out.println();
        System.out.println("targets:");
        System.out.println("  build             Build MIDlet (default)");
        System.out.println("  debugbuild        Debug build (not perform shrinking)");
        System.out.println("  clean             Clean working directory");
        System.out.println();
        System.out.println("  output            Copy resulting files to output directory");
        System.out.println("  remove-workdir    Remove working directory itself");
    }
}
