/*
 * $Id: Converter.java,v 1.10 2003/04/06 01:50:50 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.converter;

import java.io.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import jp.sourceforge.imodoki.util.FileUtils;
import jp.sourceforge.imodoki.converter.Builder;

/**
 * Ѵʬᥤ
 * ȥǥ쥯ȥiץ jam, jar ե
 * Configurator Ƥӡɬפʥե롣
 */
public class Converter {
    private URL m_jamURL;
    private ConverterOption m_converterOption;
    private JAMFile m_jamContent;
    private JarFileInspector m_jarInspector;
    private RuntimeCompileOption m_compileOption;
    private String m_sourceURL;
    private File m_imodokiHome;

    /**
     * 󥹥ȥ饯
     */
    public Converter(URL jamURL, ConverterOption converterOption,
                     RuntimeCompileOption compileOoption) {
        m_jamURL = jamURL;
        m_converterOption = converterOption;
        m_compileOption = compileOoption;

        m_imodokiHome = new File(System.getProperty("imodoki.home"));
        try {
            m_imodokiHome = m_imodokiHome.getCanonicalFile();
        } catch (IOException ioe) {
            System.err.println("Warning: cannot canonicalize IMODOKI_HOME (" +
                               ioe + ")");
        }
    }

    /**
     * ѴԤʤ
     *
     * @return  Ѵ true
     */
    public boolean convert() {
        File workDir = m_converterOption.getWorkDir();

        File iappDir = new File(workDir, "iappli");
        File binDir = new File(workDir, "bin");

        try {

            /* ȥǥ쥯ȥ */
            FileUtils.deleteRecursive(workDir);
            mkdirs(workDir);
            mkdirs(iappDir);
            mkdirs(binDir);

            /* jam ե iappDir ¸ */

            String baseName = getURLBaseName(m_jamURL);
            int idx = baseName.lastIndexOf('.');
            if (idx != -1) {
                baseName = baseName.substring(0, idx);
            }

            File jamFile = new File(iappDir, baseName + ".jam");
            log("Getting iappli jam file: " + m_jamURL);
            try {
                saveURL(m_jamURL, jamFile);
            } catch (FileNotFoundException fnfe) {
                throw new BuildException(
                    "jam file not found: `" + m_jamURL + "'");
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to get jam file `" + m_jamURL + "'", ioe);
            }

            /* jam եȤĴ٤ */
            try {
                m_jamContent = new JAMFile(jamFile);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to parse jam file `" + jamFile + "'", ioe);
            }

            /* jar ե iappDir ¸ */

            String packageURL = m_jamContent.getPackageURL();
            URL jarURL;
            try {
                jarURL = new URL(m_jamURL, packageURL);
            } catch (MalformedURLException urle) {
                throw new BuildException(
                    "Failed to parse PackageURL `" + packageURL + "'" + urle);
            }

            log("Getting iappli jar file: " + jarURL);
            File jarFile = new File(iappDir, baseName + ".jar");
            try {
                saveURL(jarURL, jarFile);
            } catch (FileNotFoundException fnfe) {
                throw new BuildException(
                    "jar file not found: `" + jarURL + "'");
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to save jar file `" + jarURL + "'", ioe);
            }

            System.out.println();
            System.out.println("[Configuration phase]");
            System.out.println();

            m_sourceURL = getParentURL(jarURL).toString();

            /* jam, jar Ĵɬפʾ */
            try {
                m_jarInspector = new JarFileInspector(jarFile);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to inspect jar file `" + jarFile + "'", ioe);
            }
            m_compileOption.configure(m_jamContent, m_jarInspector);

            /* MIDlet Manifest ե */
            log("Generating manifest file");
            File manifestFile = new File(binDir, "MANIFEST.MF");
            try {
                String manifestEncoding = "UTF-8";
                createManifest(manifestFile, manifestEncoding);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to create manifest file `" + manifestFile + "'",
                    ioe);
            }

            /* build.properties ե */
            log("Generating build.properties");
            File propFile = new File(workDir, "build.properties");
            try {
                createBuildProperties(propFile, baseName);
            } catch (IOException ioe) {
                throw new BuildException(
                    "Failed to create property file `" + propFile + "'", ioe);
            }

            try {
                m_jarInspector.close();
            } catch (IOException ioe) {
                throw new BuildException(
                    "Warning: failed to close jar file `" + jarFile + "'",
                    ioe);
            }

        } catch (BuildException be) {
            System.err.println(be.getMessage());
            return false;
        }

        if (!m_converterOption.isKeepWorkDir()) {
            log("");
            log("Launching Builder");

            int verbosity = m_converterOption.getVerbosity();

            List args = new ArrayList();
            args.add("-basedir");
            args.add(workDir.getPath());
            if (verbosity >= ConverterOption.DEBUG) {
                args.add("-debug");
            } else if (verbosity >= ConverterOption.DEBUG) {
                args.add("-verbose");
            }
            if (m_converterOption.isDebugBuild()) {
                args.add("debugbuild");
            } else {
                args.add("build");
            }
            args.add("output");
            args.add("remove-workdir");

            String[] args_ary = (String[])args.toArray(new String[0]);

            boolean success = Builder.internalMain(args_ary);
            return success;
        } else {
            return true;
        }
    }

    /** MIDlet Manifest ե롣 */
    private void createManifest(File file, String encoding)
        throws IOException {
        Map content = new TreeMap();

        content.put("MIDlet-Name", m_jamContent.getAppName());
        content.put("MIDlet-Version", "1.0");
        content.put("MIDlet-Vendor", System.getProperty("user.name"));
        content.put("MIDlet-1", m_jamContent.getAppName() + ",," +
                                "imodoki.MainMIDlet");
        content.put("MicroEdition-Configuration", "CLDC-1.0");
        content.put("MicroEdition-Profile", "MIDP-1.0");

        int spSize = m_jamContent.getSPsize();
        if (spSize > 0) {
            // FIXME: εư ezplus ȼΤΡMIDPѡJ-Phone
            //        ˤ̤н򤹤٤
            spSize += 30 + 1;
            if (spSize > 10240) {
                log("Warning: RecordStore will not enough: " + spSize);
                spSize = 10240;
            }

            content.put("MIDlet-Data-Size", String.valueOf(spSize));
        }

        if (m_jarInspector.callsMethod("com.nttdocomo.ui.IApplication",
                                       "getSourceURL",
                                       "()Ljava/lang/String;")) {
            content.put("IMDK-SourceURL", m_sourceURL);
        }
        String appParam = m_jamContent.getAppParam();
        if (appParam != null && appParam.length() > 0) {
            content.put("IMDK-AppParam", appParam);
        }

        writeManifest(file, encoding, content);
    }

    /** MIDlet Manifest ե񤭽Ф */
    private static void writeManifest(File file, String encoding, Map content)
        throws IOException {
        String eol = System.getProperty("line.separator");

        Writer out =
            new OutputStreamWriter(new FileOutputStream(file), encoding);
        try {
            Iterator itr = content.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry entry = (Map.Entry)itr.next();
                out.write((String)entry.getKey());
                out.write(": ");
                out.write((String)entry.getValue());
                out.write(eol);
            }
        } finally {
            out.close();
        }
    }

    /** build.properties ե롣 */
    private void createBuildProperties(File file, String baseName)
        throws IOException {
        Properties props = new Properties();

        props.setProperty("imodoki.home",
                          m_imodokiHome.getPath());
        props.setProperty("app.basename", baseName);
        props.setProperty("iappli.app.class", m_jamContent.getAppClass());
        props.setProperty("output.dir",
                          m_converterOption.getOutputDir().getPath());
        props.setProperty("enabled.options",
                          m_compileOption.getBooleanList(true));
        props.setProperty("disabled.options",
                          m_compileOption.getBooleanList(false));

        OutputStream out = new FileOutputStream(file);
        try {
            props.store(out, null);
        } finally {
            out.close();
        }
    }

    /** Ϥ롣 */
    private void log(String mesg) {
        System.out.println(mesg);
    }

    /** 顼դ File#mkdirs() */
    private static void mkdirs(File dir) throws BuildException {
        if (dir.isDirectory())
            return;

        if (dir.mkdirs() == false) {
            throw new BuildException(
                "Failed to create directory '" + dir + "'");
        }
    }

    /** URL ǻꤵ줿꥽ե¸롣 */
    private static void saveURL(URL url, File file) throws IOException {
        InputStream in = null;
        OutputStream out = null;

        try {
            in = url.openStream();
            out = new FileOutputStream(file);
            FileUtils.copyStream(in, out);
        } finally {
            if (in != null)
                in.close();
            if (out != null)
                out.close();
        }
    }

    /** URL ǻꤵ줿ե̾ʬ֤ */
    private static String getURLBaseName(URL url) {
        String path = url.getPath();
        int idx = path.lastIndexOf('/');
        if (idx == -1) {
            return path;
        } else {
            return path.substring(idx + 1);
        }
    }

    /**
     * ꤵ줿 URL οƤ URL ֤
     * : http://www.hoge.com/abc/def -&gt; http://www.hoge.com/abc/
     */
    private static URL getParentURL(URL url) {
        try {
            return new URL(url, ".");
        } catch (MalformedURLException urle) {
            throw new RuntimeException("Cannot happen: " + urle.toString());
        }
    }
}
