/*
 * $Id: Shrinker.java,v 1.12 2003/04/06 01:50:49 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.shrinker;

import java.io.*;
import java.util.*;
import java.util.zip.Deflater;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;
import java.util.jar.Manifest;
import org.apache.bcel.classfile.ClassParser;
import org.apache.bcel.classfile.JavaClass;
import jp.sourceforge.imodoki.util.CollectionUtils;
import jp.sourceforge.imodoki.util.FileUtils;
import jp.sourceforge.imodoki.util.ExecUtils;
import jp.sourceforge.imodoki.util.OptionParser;
import jp.sourceforge.imodoki.util.OptionParserException;

/**
 * Ƽ說饹ե̾ġƤӽФ饹ե륵̾롣
 */
public class Shrinker {
    public static final String EXTTOOLS_DIR = "imodoki.exttools.dir";

    /** ƥġ֤Ƕ̤ƻȤƥݥǥ쥯ȥꡣ */
    public static File s_tempDir;

    private String m_classPath;
    private File m_inputJar;
    private File m_outputJar;
    private String[] m_preserves;
    private String m_preverifier;
    private int m_verbosity;

    private ToolDriver[] m_drivers;

    /**
     * 󥹥ȥ饯
     *
     * @param  classPath    饹ѥ
     * @param  inputJar     ϤȤʤ jar ե롣
     * @param  outputJar    Ϥ jar ե롣
     * @param  preserves    Ϥ˴ޤ᥽åɡեɤλꡣ
     * @param  preverifier  ץ٥եΥѥ
     * @param  verbosity    Verbosity ٥롣
     */
    public Shrinker(String classPath, File inputJar, File outputJar,
                    String[] preserves, String preverifier, int verbosity) {
        m_classPath = classPath;
        m_inputJar = inputJar;
        m_outputJar = outputJar;
        m_preserves = preserves;
        m_preverifier = preverifier;
        m_verbosity = verbosity;
        m_drivers = initToolDrivers();
    }

    private static ToolDriver[] initToolDrivers() {
        String prop = System.getProperty(EXTTOOLS_DIR);
        if (prop == null) {
            throw new RuntimeException("System property `" + EXTTOOLS_DIR +
                "' not set");
        }

        /**
         * ġɥ饤С
         * 椫֤ˡѲǽʤΤƤӽФ롣
         * ϽפǤ (RetroGuard ɬû̾դʤ)
         * ƤΥġ뤬Ƽ¹Ԥ뤬̣ʤäꡢդ
         * ե륵ĤǤޤȤ߹碌⤢Τա
         */
        ToolDriver[] drivers = new ToolDriver[] {
            new IModokiExtractorDriver(),
            new RetroGuardDriver(),
            new ProGuardDriver(),
            new JargDriver(),
            new JAXDriver(),
        };

        return drivers;
    }

    /**
     * ̾Ԥʤ
     */
    public boolean process() {
        /* ȥǥ쥯ȥ */
        s_tempDir = FileUtils.createTempDir("shrinker");
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                FileUtils.deleteRecursive(s_tempDir);
            }
        });

        /* Ƽġ¹Ԥ */
        File inputJar = m_inputJar;
        File outputJar = null;
        for (int i = 0; i < m_drivers.length; i++) {
            ToolDriver driver = m_drivers[i];
            if (driver.available()) {

                if (m_verbosity >= ToolDriver.TERSE) {
                    System.out.println("Shrinker: running " +
                                       driver.getName());
                }

                outputJar = callDriver(driver, inputJar);
                inputJar = outputJar;
            }
        }

        /*  */
        try {
            if (outputJar == null) {
                System.err.println("Shrinker: no tool available");
                outputJar = m_inputJar;
            }

            if (m_preverifier != null) {
                if (m_verbosity >= ToolDriver.TERSE) {
                    System.out.println("Shrinker: preverifying...");
                }
                preverify(outputJar, m_outputJar);
            } else {
                repackJar(outputJar, m_outputJar);
            }
        } catch (IOException ioe) {
            System.err.println(ioe.toString());
            return false;
        } finally {
            if (outputJar != m_inputJar) {
                outputJar.delete();
            }

            FileUtils.deleteRecursive(s_tempDir);
        }

        return true;
    }

    /**
     * inputJar ȥե˽Ϥ롣
     *  inputJar ȥե File
     * ֥Ȥ֤Ԥ inputJar ֤
     */
    private File callDriver(ToolDriver driver, File inputJar) {
        File outputJar = null;
        try {
            outputJar = File.createTempFile("shrinker_out", ".zip", s_tempDir);

            driver.process(m_classPath, inputJar, outputJar,
                           m_preserves, m_verbosity);

            if (outputJar.length() == 0) {
                throw new IOException("Failure due to unknown reason in " + driver.getClass().getName());
            }
        } catch (IOException ioe) {
            System.err.println("Shrinking failed: " + ioe.getMessage());
            System.err.println("Process continuing.");
            if (outputJar != null) {
                outputJar.delete();
            }
            return inputJar;
        }

        /*  */
        if (inputJar != m_inputJar) {
            inputJar.delete();
        }
        return outputJar;
    }

    /**
     * ץ٥եư롣
     */
    private void preverify(File inJar, File outJar) throws IOException {
        File tempDirIn  = new File(s_tempDir, "preverify_in");
        File tempDirOut = new File(s_tempDir, "preverify_out");
        try {
            tempDirIn.mkdir();
            tempDirOut.mkdir();

            FileUtils.expandZip(inJar, tempDirIn);

            String[] cmdarray = new String[] {
                m_preverifier,
                "-classpath",
                m_classPath,
                "-d",
                tempDirOut.getPath(),
                tempDirIn.getPath(),
            };

            if (m_verbosity >= ToolDriver.VERBOSE) {
                System.out.println("Shrinker: running preverifier: " +
                                   CollectionUtils.join(cmdarray, " "));
            }

            int exitCode = ExecUtils.execCommand(cmdarray);
            if (exitCode != 0) {
                throw new IOException("Preverify failed (exit code " +
                                      exitCode + ")");
            }

            fixManifest(tempDirIn);

            FileUtils.createJar(outJar, tempDirIn,
                                FileUtils.RESOURCE_FILENAME_FILTER,
                                null, Deflater.BEST_COMPRESSION);
            FileUtils.updateJar(outJar, tempDirOut, null, null,
                                Deflater.BEST_COMPRESSION);
        } finally {
            FileUtils.deleteRecursive(tempDirIn);
            FileUtils.deleteRecursive(tempDirOut);
        }
    }

    /**
     * jar ե뤫ǥ쥯ȥꥨȥǹΰ̥٥
     * ̤ľ
     */
    private static void repackJar(File inJar, File outJar) throws IOException {
        /* ǥ쥯ȥŸƤư̤롣
           ä̵̡*/
        File tempDir = new File(s_tempDir, "repack");
        try {
            tempDir.mkdir();

            FileUtils.expandZip(inJar, tempDir);

            fixManifest(tempDir);

            FileUtils.createJar(outJar, tempDir, null, null,
                                Deflater.BEST_COMPRESSION);
        } finally {
            FileUtils.deleteRecursive(tempDir);
        }
    }

    /**
     * dir ʲ Manifest ե롣
     * RetroGuard ˤäղä롢Digest ʤɤ̾(J2MEӤˤ)
     * ɬפʤΤǽ롣
     */
    private static void fixManifest(File dir) throws IOException {
        File manFile = FileUtils.newFile3(dir, "META-INF", "MANIFEST.MF");
        if (!manFile.exists()) {
            return;
        }

        InputStream is = new FileInputStream(manFile);
        Manifest man = new Manifest(is);
        is.close();

        /* Main Attributes ʳΥȥ */
        // : ¸⤷ʤ
        man.getEntries().clear();

        OutputStream os = new FileOutputStream(manFile);
        try {
            man.write(os);
        } finally {
            os.close();
        }
    }

    /**
     * MIDlet Υ֥饹򸫤ĤФpreserve ٤ФΥꥹȤ֤
     */
    private static Set preserveMIDlet(File inJar) throws IOException {
        String targetParent = "javax.microedition.midlet.MIDlet";

        Set result = new TreeSet();

        /* ƥ饹 targetParent Ǥ륯饹õ */
        // TODO: ϻ¹饹ޤĴ٤٤
        ZipFile zipFile = new ZipFile(inJar);
        try {
            for (Enumeration e = zipFile.entries(); e.hasMoreElements(); ) {
                ZipEntry entry = (ZipEntry)e.nextElement();
                String name = entry.getName();
                if (name.endsWith(".class") && !entry.isDirectory()) {
                    InputStream is = zipFile.getInputStream(entry);
                    ClassParser parser = new ClassParser(is, name);
                    JavaClass clazz = parser.parse();
                    String parent = clazz.getSuperclassName();
                    if (parent.equals(targetParent)) {
                        /* ̾ä result ɲä */
                        // TODO: ϥºݤ¸ߤ뤳Ȥ
                        //       ǧɬפ
                        String className = clazz.getClassName();
                        result.add(className + ".<init>()");
                        result.add(className + ".startApp()");
                        result.add(className + ".pauseApp()");
                        result.add(className + ".destroyApp(Z)");
                    }
                }
            }
        } finally {
            zipFile.close();
        }

        return result;
    }

    /** ¼Υᥤ */
    public static boolean internalMain(String[] args) {
        String classPath = null;
        File inputJar = null;
        File outputJar = null;
        String preverifier = null;
        boolean quiet = false, verbose = false, debug = false;
        Set preserves = new TreeSet();
        boolean preserveMIDlet = false;

        OptionParser optParser = new OptionParser();
        optParser.addOption("-classpath", "<path>",
                            "Specify library class path");
        optParser.addOption("-injar", "<file>",
                            "Specify input jar file");
        optParser.addOption("-outjar", "<file>",
                            "Specify output jar file");
        optParser.addOption("-preserve", "<list>",
                            "Specify comma-separated list of methods or\n" +
                            "fields to be preserved\n" +
                            "ex. `FooClass.fooMethod(II),BarClass.barField'");
        optParser.addOption("-preservemidlet",
                            "Preserve MIDlet classes");
        optParser.addOption("-preverifier", "<path>",
                            "Specify the path to the preverifier\n" +
                            "and turn on preverifying");
        optParser.addOption("-quiet",
                            "Silent execution");
        optParser.addOption("-verbose",
                            "Verbose execution");
        optParser.addOption("-debug",
                            "Print debugging information");
        optParser.addOption("-help",
                            "Show this help");

        try {
            String opt;
            while ((opt = optParser.parse(args)) != null) {
                if (opt.equals("-classpath")) {
                    classPath = optParser.getOptionArg();
                } else if (opt.equals("-injar")) {
                    inputJar = new File(optParser.getOptionArg());
                } else if (opt.equals("-outjar")) {
                    outputJar = new File(optParser.getOptionArg());
                } else if (opt.equals("-preserve")) {
                    List list =
                        CollectionUtils.split(optParser.getOptionArg(), ",");
                    preserves.addAll(list);
                } else if (opt.equals("-preservemidlet")) {
                    preserveMIDlet = true;
                } else if (opt.equals("-preverifier")) {
                    preverifier = optParser.getOptionArg();
                } else if (opt.equals("-quiet")) {
                    quiet = true;
                } else if (opt.equals("-verbose")) {
                    verbose = true;
                } else if (opt.equals("-debug")) {
                    debug = true;
                } else if (opt.equals("-help")) {
                    usage(optParser);
                    return true;
                } else {
                    throw new RuntimeException("Unreachable");
                }
            }
        } catch (OptionParserException ope) {
            System.err.println(ope.getMessage());
            return false;
        }

        String[] restArgs = optParser.getArgs();
        if (restArgs.length != 0) {
            usage(optParser);
            return false;
        }
        
        if (classPath == null || inputJar == null || outputJar == null ||
            (!preserveMIDlet && preserves.isEmpty())) {
            usage(optParser);
            return false;
        }

        if (preserveMIDlet) {
            try {
                Set set = preserveMIDlet(inputJar);
                preserves.addAll(set);
            } catch (IOException ioe) {
                System.err.println(ioe);
                return false;
            }
        }

        int verbosity = ToolDriver.TERSE;
        if (quiet) {
            verbosity = ToolDriver.QUIET;
        } else if (debug) {
            verbosity = ToolDriver.DEBUG;
        } else if (verbose) {
            verbosity = ToolDriver.VERBOSE;
        }

        Shrinker shrinker =
            new Shrinker(classPath, inputJar, outputJar,
                         (String[])preserves.toArray(new String[0]),
                         preverifier, verbosity);
        boolean success = shrinker.process();
        return success;
    }

    /** ᥤ */
    public static void main(String[] args) {
        boolean success = internalMain(args);
        System.exit(success ? 0 : 1);
    }

    private static void usage(OptionParser optParser) {
        System.out.println("usage: shrinker [options...] -classpath path -injar file -outjar file");
        System.out.println();
        System.out.println("options:");
        optParser.printUsage(System.out, 2, 24);

        System.out.println();
        System.out.println("available tools:");
        String prop = System.getProperty(EXTTOOLS_DIR);
        if (prop == null) {
            System.out.println("  warning: please set system property `" +
                               EXTTOOLS_DIR + "'");
        } else {
            ToolDriver[] drivers = initToolDrivers();
            for (int i = 0; i < drivers.length; i++) {
                ToolDriver driver = drivers[i];
                System.out.print("  " + driver.getName() + ": ");
                if (driver.available()) {
                    System.out.println("available");
                } else {
                    System.out.println("not available");
                }
            }
        }
    }
}
