/*
 * $Id: FileUtils.java,v 1.9 2003/04/06 01:50:49 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.util;

import java.io.*;
import java.util.*;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.Deflater;
import java.util.jar.Manifest;

/**
 * ե桼ƥƥ
 */
public class FileUtils {
    /**
     * ե򥳥ԡ롣
     *
     * @param  srcFile  ԡե
     * @param  dstFile  ԡե
     */
    public static void copyFile(File srcFile, File dstFile)
        throws IOException {
        InputStream in = null;
        OutputStream out = null;

        try {
            in = new FileInputStream(srcFile);
            out = new FileOutputStream(dstFile);
            copyStream(in, out);
        } finally {
            if (in != null)
                in.close();
            if (out != null)
                out.close();
        }
    }

    /**
     * ȥ꡼򥳥ԡ롣
     *
     * @param  in   ϥȥ꡼ࡣ
     * @param  out  ϥȥ꡼ࡣ
     */
    public static void copyStream(InputStream in, OutputStream out)
        throws IOException {
        byte[] buf = new byte[4096];
        int len;

        while ((len = in.read(buf)) != -1)
            out.write(buf, 0, len);
    }

    /**
     * File#listFiles(FileFilter) κƵǡ
     *
     * @param  dir     ϥǥ쥯ȥꡣ
     * @param  filter  ե륿
     * @return dir ʲΥեΥꥹȡ
     */
    public static File[] listFilesRecursive(File dir, FileFilter filter) {
        List list = new ArrayList();
        listFilesRecursiveInternal(dir, filter, list);
        return (File[])(list.toArray(new File[0]));
    }

    private static void listFilesRecursiveInternal(File dir, FileFilter filter,
                                                   List list) {
        File entries[] = dir.listFiles();
        if (entries == null)
            return;
        for (int i = 0; i < entries.length; i++) {
            File entry = entries[i];
            if ((filter == null) || filter.accept(entry)) {
                list.add(entry);
            }
            if (entry.isDirectory()) {
                listFilesRecursiveInternal(entry, filter, list);
            }
        }
    }

    /**
     * File#list(FileFilter) κƵǡ֤ String ֥Ȥ
     * dir ХѥɽʸʸΥǥ쥯ȥڤϡ
     * ץåȥե¸Ǥ롣
     *
     * @param  dir     ϥǥ쥯ȥꡣ
     * @param  filter  ե륿
     * @return dir ʲΥեΥꥹȡ
     */
    public static String[] listRecursive(File dir, FilenameFilter filter) {
        List list = new ArrayList();
        listRecursiveInternal(dir, filter, list, dir, "");
        return (String[])(list.toArray(new String[0]));
    }

    private static void listRecursiveInternal(File dir, FilenameFilter filter,
                                              List list, File baseDir,
                                              String pathBase) {
        String names[] = dir.list();
        if (names == null)
            return;
        for (int i = 0; i < names.length; i++) {
            String name = names[i];
            String path = pathBase + name;
            if ((filter == null) || filter.accept(baseDir, path)) {
                list.add(path);
            }
            File entry = new File(dir, name);
            if (entry.isDirectory()) {
                listRecursiveInternal(entry, filter, list, baseDir,
                                      path + File.separatorChar);
            }
        }
    }

    /** 饹ե˥ޥå FileFilter */
    public static final FileFilter CLASSFILE_FILTER =
    new FileFilter() {
        public boolean accept(File file) {
            if (file.isFile() && file.getName().endsWith(".class")) {
                return true;
            } else {
                return false;
            }
        }
    };

    /** 饹ե˥ޥå FilenameFilter */
    public static final FilenameFilter CLASSFILENAME_FILTER =
    new FilenameFilter() {
        public boolean accept(File dir, String name) {
            File file = new File(dir, name);
            if (file.isFile() && name.endsWith(".class")) {
                return true;
            } else {
                return false;
            }
        }
    };

    /** 饹ե˥ޥå ZipEntryFilter */
    public static final ZipEntryFilter ZIPENTRY_CLASSFILE_FILTER =
    new ZipEntryFilter() {
        public boolean accept(ZipEntry entry) {
            if (!entry.isDirectory() && entry.getName().endsWith(".class")) {
                return true;
            } else {
                return false;
            }
        }
    };

    /** 饹եʳ˥ޥå FilenameFilter */
    public static final FilenameFilter RESOURCE_FILENAME_FILTER =
    new FilenameFilter() {
        public boolean accept(File dir, String name) {
            File file = new File(dir, name);
            if (file.isFile() && !name.endsWith(".class")) {
                return true;
            } else {
                return false;
            }
        }
    };

    private static int m_tempDirCounter = -1;
    private static Object m_tempDirLock = new Object();

    /** ƥݥǥ쥯ȥ File ֥Ȥ֤ */
    public static File createTempDir(String prefix) {
        File parent = new File(System.getProperty("java.io.tmpdir", "."));
        File dir;

        synchronized (m_tempDirLock) {
            do {
                dir = generateFile(prefix, parent);
            } while (!dir.mkdirs());

            return dir;
        }
    }

    private static File generateFile(String prefix, File dir) {
        if (m_tempDirCounter == -1) {
            m_tempDirCounter = new Random().nextInt() & 0xffff;
        }
        m_tempDirCounter++;
        return new File(dir, prefix + Integer.toString(m_tempDirCounter));
    }

    /**
     * ƵŪ˥ǥ쥯ȥ(ϥե)롣
     *
     * @param  dir  оݥǥ쥯ȥϥե롣
     * @return ˥顼 false
     */
    public static boolean deleteRecursive(File dir) {
        File[] list = dir.listFiles();
        boolean success;

        if (list == null) {             /* file */
            success = dir.delete();
        } else {                        /* directory */
            success = true;
            for (int i = 0; i < list.length; i++) {
                success &= deleteRecursive(list[i]);
            }
            if (success) {
                success &= dir.delete();
            }
        }

        return success;
    }

    /**
     * ǥ쥯ȥݤȥԡ롣
     * UNIX ΥޥɤǸСcp -r srcDir/* destDir/ Ȥ
     *
     * @param  srcDir   ԡǥ쥯ȥꡣ
     * @param  destDir  ԡǥ쥯ȥꡣ¸ߤƤʤк롣
     *
     * @exception IOException  ԡ˼ԤȤ
     */
    public static void copyDirectory(File srcDir, File destDir)
        throws IOException {
        if (!destDir.isDirectory()) {
            if (destDir.mkdirs() == false) {
                throw new IOException("Failed to create directory `" +
                                      destDir + "'");
            }
        }

        String[] list = srcDir.list();
        if (list == null) {
            throw new IOException("Source directory `" + srcDir +
                                  "' is not a directory");
        }

        for (int i = 0; i < list.length; i++) {
            String name = list[i];
            File srcFile = new File(srcDir, name);
            File destFile = new File(destDir, name);
            if (srcFile.isDirectory()) {
                copyDirectory(srcFile, destFile);
            } else {
                copyFile(srcFile, destFile);
            }
        }
    }

    /**
     * path = new File(dir, file) ǤȤȤ dir, path 
     * file 롣̤ file ХѥȤʤ롣
     * path  dir βˤʤˤ IllegalArgumentException 
     * ꤲ롣
     *
     * @param  dir   ǥ쥯ȥꡣ
     * @param  path  ѥ
     *
     * @exception IllegalArgumentException  path  dir βˤʤ硣
     */
    public static File stripPrefixDir(File dir, File path) {
        String dirStr = dir.getPath();
        String pathStr = path.getPath();

        if (!pathStr.startsWith(dirStr)) {
            throw new IllegalArgumentException(path + " is not under " + dir);
        }

        int len = dirStr.length();
        /* relPath Ƭ File.separator ˤʤʤФ򥹥åפ */
        if (pathStr.length() > len &&
            pathStr.charAt(len) == File.separatorChar) {
            len++;
        }
        String relPath = pathStr.substring(len);

        return new File(relPath);
    }

    /**
     * zip եꤵ줿ǥ쥯ȥŸ롣
     * MANIFEST ե̰ʤ
     *
     * @param zipFile  zip ե롣
     * @param destDir  Ÿǥ쥯ȥꡣ
     * @param filter   ե񤭽Ф٤ɤȽꤹե륿
     *                 null ξϥե륿󥰤ʤ
     *
     * @exception IOException  ϥ顼Ȥ
     */
    public static void expandZip(File zipFile, File destDir,
                                 ZipEntryFilter filter) throws IOException {
        FileInputStream fis = new FileInputStream(zipFile);
        ZipInputStream zis =
            new ZipInputStream(new BufferedInputStream(fis));
        try {
            ZipEntry entry;
            while((entry = zis.getNextEntry()) != null) {
                if (filter != null && !filter.accept(entry)) {
                    zis.closeEntry();
                    continue;
                }

                String pathStr =
                    entry.getName().replace('/', File.separatorChar);
                File path = new File(destDir, pathStr);
                if (entry.isDirectory()) {
                    path.mkdirs();
                } else {
                    File parent = path.getParentFile();
                    if (parent != null) {
                        parent.mkdirs();
                    }

                    OutputStream os = new FileOutputStream(path);
                    try {
                        copyStream(zis, os);
                    } finally {
                        os.close();
                    }
                }
                zis.closeEntry();
            }
        } finally {
            zis.close();
        }
    }

    public static void expandZip(File zipFile, File destDir)
        throws IOException {
        expandZip(zipFile, destDir, null);
    }

    /** ޥ˥եȥե jar ȥ̾ */
    public static final String MANIFEST_NAME = "META-INF/MANIFEST.MF";

    /**
     * ꤵ줿ǥ쥯ȥʲΥե jar ե˳Ǽ롣
     * jar եϿ롣manifestFile  null Ǥʤ
     * ޤϻꤵ줿ǥ쥯ȥľ META-INF/MANIFEST.MF Ȥ
     * ե뤬Сjar եκǽΥȥȤƳǼ롣
     * ǥ쥯ȥꥨȥϳǼʤ
     *
     * @param jarFile  jar ե롣
     * @param srcDir   ǥ쥯ȥꡣ
     * @param filter   եǼ٤ɤȽꤹե륿
     *                 null ξϥե륿󥰤ʤ
     * @param manifestFile
     *                 ޥ˥եȥե롣srcDir õ
     *                 null
     * @param level    ̥٥롣
     *
     * @exception IOException  ϥ顼Ȥ
     */
    public static void createJar(File jarFile, File srcDir,
                                 FilenameFilter filter,
                                 File manifestFile,
                                 int level) throws IOException {
        String[] files = listRecursive(srcDir, filter);
        Arrays.sort(files);

        FileOutputStream fos = new FileOutputStream(jarFile);
        ZipOutputStream zos =
            new ZipOutputStream(new BufferedOutputStream(fos));
//      ̥᥽åɤϥݡȤƤʤSTORED ξ CRC 
//      ׻ɬפݤʤΤǡ
//         zos.setMethod(method);
        zos.setLevel(level);
        try {
            if (manifestFile != null) {
                writeManifestEntry(zos, manifestFile);
            } else {
                String manName =
                    MANIFEST_NAME.replace('/', File.separatorChar);
                File manFile = new File(srcDir, manName);
                if (manFile.isFile() &&
                    (filter == null || filter.accept(srcDir, manName))) {
                    writeManifestEntry(zos, manFile);
                }
            }

            for (int i = 0; i < files.length; i++) {
                String fname = files[i];
                File file = new File(srcDir, fname);
                if (file.isDirectory()) {
                    continue;
                }

                String entryName = fname.replace(File.separatorChar, '/');
                if (entryName.equalsIgnoreCase(MANIFEST_NAME)) {
                    continue;
                }

                writeZipEntry(zos, entryName, file);
            }
        } finally {
            try {
                zos.close();
            } catch (IOException ioe) {
                System.err.println("IOException on close -----");
                ioe.printStackTrace();
                System.err.println("-----");
            }
        }
    }

    public static void createJar(File jarFile, File srcDir,
                                 FilenameFilter filter) throws IOException {
        createJar(jarFile, srcDir, filter, null, Deflater.DEFAULT_COMPRESSION);
    }

    private static void writeZipEntry(ZipOutputStream zos, String entryName,
                                      File file) throws IOException {
        ZipEntry entry = new ZipEntry(entryName);
        zos.putNextEntry(entry);

        InputStream is = new FileInputStream(file);
        try {
            copyStream(is, zos);
        } finally {
            is.close();
        }

        zos.closeEntry();
    }

    private static void writeManifestEntry(ZipOutputStream zos, File file)
        throws IOException {
        ZipEntry entry = new ZipEntry(MANIFEST_NAME);
        zos.putNextEntry(entry);

        /*
          Manifest-Version °ղä롣
          ʤեΥ󥳡ǥ󥰤ϸΤޤݻ롣
          (ʤȤ JDK1.4 λǤ Manifest 饹μǤϡ)
         */
        InputStream is = new FileInputStream(file);
        Manifest man = new Manifest(is);
        is.close();
        man.getMainAttributes().putValue("Manifest-Version", "1.0");
        man.write(zos);

        zos.closeEntry();
    }

    /**
     * ꤵ줿ǥ쥯ȥʲΥե jar ե˹롣
     * jar ե¸ߤƱ̾ΥȥϾ񤭤롣
     * manifestFile  null Ǥʤꤵ줿ǥ쥯ȥľ
     * META-INF/MANIFEST.MF Ȥե뤬Сjar ե
     * ǽΥȥȤƳǼ롣ǥ쥯ȥꥨȥϳǼʤ
     * jar ե뤬¸ߤʤȤޤϰե뤬Ǥʤ
     * ȤˤϡFileNotFoundException ꤲ롣
     *
     * @param jarFile  jar ե롣
     * @param srcDir   ǥ쥯ȥꡣ
     * @param filter   եǼ٤ɤȽꤹե륿
     *                 null ξϥե륿󥰤ʤ
     * @param manifestFile
     *                 ޥ˥եȥե롣srcDir õ
     *                 null
     * @param level    Ǽեΰ̥٥롣
     *
     * @exception IOException  ϥ顼Ȥ
     */
    public static void updateJar(File jarFile, File srcDir,
                                 FilenameFilter filter,
                                 File manifestFile,
                                 int level) throws IOException {
        ZipInputStream zis = null;
        ZipOutputStream zos = null;
        File tmpFile = null;

        try {
            FileInputStream fis = new FileInputStream(jarFile);
            zis = new ZipInputStream(new BufferedInputStream(fis));

            String[] fnames = listRecursive(srcDir, filter);
            Map fileMap = getFileMap(fnames, srcDir);

            String p = jarFile.getParent();
            File pDir = new File(p != null ? p : ".");
            tmpFile = File.createTempFile("update", ".jar", pDir);

            FileOutputStream fos = new FileOutputStream(tmpFile);
            zos = new ZipOutputStream(new BufferedOutputStream(fos));
//      ̥᥽åɤϥݡȤƤʤSTORED ξ CRC 
//      ׻ɬפݤʤΤǡ
//             zos.setMethod(method);
            zos.setLevel(level);

            boolean manArchived = false;
            if (manifestFile != null) {
                writeManifestEntry(zos, manifestFile);
                manArchived = true;
            } else {
                String manName =
                    MANIFEST_NAME.replace('/', File.separatorChar);
                File manFile = new File(srcDir, manName);
                if (manFile.isFile() &&
                    (filter == null || filter.accept(srcDir, manName))) {
                    writeManifestEntry(zos, manFile);
                    manArchived = true;
                }
            }

            /* () jar եδ¸Υȥ򥳥ԡ */
            ZipEntry entry;
            while((entry = zis.getNextEntry()) != null) {
                String entryName = entry.getName();
                if (fileMap.containsKey(entryName)) {
                    /* եǼ */
                    File file = (File)fileMap.get(entryName);
                    if (!file.isDirectory() &&
                        !entryName.equalsIgnoreCase(MANIFEST_NAME)) {
                        writeZipEntry(zos, entryName, file);
                        fileMap.remove(entryName);
                    }
                } else if (!entry.isDirectory()) {
                    /* jar ȥ򥳥ԡ */
                    if (!manArchived ||
                        !entryName.equalsIgnoreCase(MANIFEST_NAME)) {
                        ZipEntry newEntry = (ZipEntry)entry.clone();
                        zos.putNextEntry(newEntry);
                        copyStream(zis, zos);
                    }
                }
                zis.closeEntry();
            }

            /* ĤΥեǼ */
            Iterator itr = fileMap.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mapEntry = (Map.Entry)itr.next();
                String entryName = (String)mapEntry.getKey();
                File file = (File)mapEntry.getValue();
                if (!file.isDirectory() &&
                    !entryName.equalsIgnoreCase(MANIFEST_NAME)) {
                    writeZipEntry(zos, entryName, file);
                }
            }
        } catch (IOException ioe) {
            if (zis != null) {
                zis.close();
            }
            if (zos != null) {
                try {
                    zos.close();
                } catch (IOException ioe2) {
                    System.err.println("IOException on close -----");
                    ioe2.printStackTrace();
                    System.err.println("-----");
                }
            }
            if (tmpFile != null) {
                tmpFile.delete();
            }
            throw ioe;
        }

        zis.close();
        try {
            zos.close();
        } catch (IOException ioe) {
            System.err.println("IOException on close -----");
            ioe.printStackTrace();
            System.err.println("-----");
        }
        jarFile.delete();
        if (!tmpFile.renameTo(jarFile)) {
            tmpFile.delete();
            throw new IOException("Rename failed: from " + tmpFile +
                                  " to " + jarFile);
        }
    }

    public static void updateJar(File jarFile, File srcDir,
                                 FilenameFilter filter) throws IOException {
        updateJar(jarFile, srcDir, filter, null, Deflater.DEFAULT_COMPRESSION);
    }

    /** ȥ̾  ե̾ map ֤ */
    private static Map getFileMap(String[] fnames, File dir) {
        Map map = new TreeMap();

        for (int i = 0; i < fnames.length; i++) {
            String fname = fnames[i];
            String entryName = fname.replace(File.separatorChar, '/');
            map.put(entryName, new File(dir, fname));
        }

        return map;
    }

    /**
     * Ant ΥѥҤ顢File ֥Ȥ롣
     *
     * @param  path  ѥҡѥڤϥץåȥե¸ʸǤ
     *               褤'/' Ǥ褤
     * @return File ֥ȡ
     */
    public static File decodeAntLikePath(String path) {
        return new File(path.replace('/', File.separatorChar));
    }

    /**
     * path/name1/name2 Ȥѥ̾롣
     */
    public static File newFile3(File path, String name1, String name2) {
        return new File(new File(path, name1), name2);
    }
}
