/*
 * $Id: Extractor.java,v 1.9 2003/04/06 01:50:48 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.extractor;

import java.io.PrintStream;
import java.io.IOException;
import java.io.File;
import java.io.FileOutputStream;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.classfile.Field;

/*
FUTURE PLAN:
  - isMethodUnresolvedAbstract() ǥޡ줿᥽åɤȤ
  - write-only field ν
 */

/**
 * 饹ե뷲ɬפʥ᥽åɡեɡ饹Ф롣
 */
public class Extractor {
    private Repository m_repository;
    private ClassPattern m_includePattern;
    private Mark m_mark; 
    private Marker m_marker;
    private JavaClass[] m_result;

    private PrintStream m_logStream;
    private boolean m_verbose;

    /**
     * 󥹥ȥ饯
     *
     * @param  classPath  饹ѥ
     */
    public Extractor(String classPath, ClassPattern includePattern) {
        m_repository = new Repository(classPath);
        m_includePattern = includePattern;
        m_mark = new Mark();
        m_marker = new Marker(m_repository, m_includePattern, m_mark);

        m_logStream = null;
        m_verbose = false;
    }

    /**
     * ꤵ줿᥽åɤФоݤȤƥޡ롣
     *
     * @param  className  饹̾
     * @param  name       ᥽å̾
     * @param  signature  Υͥ㡣
     *                    "(ꥹ)֤" ηǤ⡢
     *                    "(ꥹ)" ηǤ⡢
     *                    "ꥹ" ηǤ褤
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤ᥽å
     *                         եɤäȤꤲ롣
     */
    public void preserveMethod(String className, String name, String signature)
        throws IOException, LinkageException {
        m_marker.preserveMethod(className, name, signature);
    }

    /**
     * ꤵ줿᥽åɤФоݤȤƥޡ롣
     *
     * @param  spec  ᥽åɤΥͥ㡣
     *               "饹̾.᥽å̾(ꥹ)" η
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤ᥽å
     *                         եɤäȤꤲ롣
     */
    public void preserveMethod(String spec)
        throws IOException, LinkageException {
        int idx = spec.lastIndexOf('.');
        if (idx == -1) {
            throw new LinkageException("Bad method spec: " + spec);
        }
        int idx2 = spec.indexOf('(');
        if (idx2 == -1) {
            throw new LinkageException("Bad method spec: " + spec);
        }
        String className = spec.substring(0, idx);
        String name = spec.substring(idx + 1, idx2);
        String signature = spec.substring(idx2);
        preserveMethod(className, name, signature);
    }

    /**
     * ꤵ줿եɤФоݤȤƥޡ롣
     *
     * @param  className  饹̾
     * @param  name       ե̾
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤեɤ
     *                         äȤꤲ롣
     */
    public void preserveField(String className, String name)
        throws IOException, LinkageException {
        m_marker.preserveField(className, name);
    }

    /**
     * ꤵ줿եɤФоݤȤƥޡ롣
     *
     * @param  spec  եɤΥͥ㡣
     *               "饹̾.ե̾" η
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤեɤ
     *                         äȤꤲ롣
     */
    public void preserveField(String spec)
        throws IOException, LinkageException {
        int idx = spec.lastIndexOf('.');
        if (idx == -1) {
            throw new LinkageException("Bad field spec: " + spec);
        }
        String className = spec.substring(0, idx);
        String name = spec.substring(idx + 1);
        preserveField(className, name);
    }

    /**
     * нԤʤ
     *
     * @exception LinkageException
     *                         ޡ˸Ĥʤ᥽å
     *                         եɤäȤꤲ롣
     */
    public void process() throws IOException, LinkageException {
        m_marker.postProcess();

        log("Removing ...");

        Remover remover = new Remover(m_mark);
        List result = new ArrayList();

        JavaClass[] classes = m_repository.getClasses();
        for (int i = 0; i < classes.length; i++) {
            JavaClass clazz = classes[i];
            String className = clazz.getClassName();
            if (m_includePattern.matches(className)) {
                if (m_verbose) {
                    logProcess(clazz);
                }
                result.add(remover.processClass(clazz));
            }
        }

        m_mark.clear();
        m_repository.close();

        m_result = (JavaClass[])result.toArray(new JavaClass[0]);
    }

    public JavaClass[] getJavaClasses() {
        return m_result;
    }

    /**
     * ̤Υ饹եǥ쥯ȥʲ˽Ϥ롣
     */
    public void dumpToDirectory(File dir) throws IOException {
        JavaClass[] classes = getJavaClasses();
        for (int i = 0; i < classes.length; i++) {
            JavaClass clazz = classes[i];
            String pathName =
                clazz.getClassName().replace('.', File.separatorChar) +
                ".class";
            File path = new File(dir, pathName);
            File parent = path.getParentFile();
            if (!parent.exists()) {
                if (!parent.mkdirs()) {
                    throw new IOException("mkdirs() failed: " + parent);
                }
            }

            clazz.dump(path);
        }
    }

    /**
     * ̤Υ饹ե zip ե˽Ϥ롣
     *
     * @param  file  zip ե File ֥ȡ
     * @param  compressionLevel  ̥٥롣
     */
    public void dumpToZipFile(File file, int compressionLevel)
        throws IOException {
        JavaClass[] classes = getJavaClasses();
        Arrays.sort(classes, new Comparator() {
            public int compare(Object o1, Object o2) {
                return ((JavaClass)o1).getClassName().compareTo(
                    ((JavaClass)o2).getClassName());
            }
        });

        ZipOutputStream zos =
            new ZipOutputStream(new FileOutputStream(file));
        zos.setLevel(compressionLevel);
        try {
            for (int i = 0; i < classes.length; i++) {
                JavaClass clazz = classes[i];
                String pathName =
                    clazz.getClassName().replace('.', '/') + ".class";
                ZipEntry entry = new ZipEntry(pathName);
                zos.putNextEntry(entry);
                zos.write(clazz.getBytes());
                zos.closeEntry();
            }
        } finally {
            zos.close();
        }
    }

    /**
     * ̤Υ饹ե zip ե˥ǥեȤ
     * ̥٥ǽϤ롣
     *
     * @param  file  zip ե File ֥ȡ
     */
    public void dumpToZipFile(File file) throws IOException {
        dumpToZipFile(file, Deflater.DEFAULT_COMPRESSION);
    }

    public void setLogStream(PrintStream stream) {
        m_logStream = stream;
    }

    public void setVerbose(boolean flag) {
        m_verbose = flag;
    }

    /** ƤΥϤ롣 */
    private void logProcess(JavaClass clazz) {
        if (m_logStream == null)
            return;

        String className = clazz.getClassName();

        log("Including Class: " + className);

        Method[] methods = clazz.getMethods();
        for (int j = 0; j < methods.length; j++) {
            Method method = methods[j];
            if (!m_mark.isMethodMarked(className, method.getName(),
                                       method.getSignature())) {
                log("Removing method: " + method.getName() +
                    ClassUtils.canonicalizeMethodSignature(method.getSignature()));
            }
        }
        Field[] fields = clazz.getFields();
        for (int j = 0; j < fields.length; j++) {
            Field field = fields[j];
            if (!m_mark.isFieldMarked(className, field.getName())) {
                log("Removing field: " + field.getName());
            }
        }
    }

    /** Ϥ롣 */
    private void log(String mesg) {
        if (m_logStream != null) {
            m_logStream.println(mesg);
        }
    }

    /**
     * ȴᥤ
     * ܵǻȤ Java ΥɤǸƤӽФۤ褤
     */
    public static void main(String[] args_ary) throws IOException {
        String classPath = ".";
        String output = null;
        List includePackages = new ArrayList();
        List includeClasses = new ArrayList();
        List preserves = new ArrayList();
        boolean verbose = false;

        List args = Arrays.asList(args_ary);

        for (Iterator i = args.iterator(); i.hasNext(); ) {
            String arg = (String)i.next();
            if (arg.charAt(0) == '-') {
                if (arg.equals("-classpath")) {
                    classPath = requireNextArg(i, arg);
                } else if (arg.equals("-output")) {
                    output = requireNextArg(i, arg);
                } else if (arg.equals("-includepackage") ||
                           arg.equals("-includepkg")) {
                    includePackages.addAll(split(requireNextArg(i, arg), ","));
                } else if (arg.equals("-includeclass") ||
                           arg.equals("-includecls")) {
                    includeClasses.addAll(split(requireNextArg(i, arg), ","));
                } else if (arg.equals("-preserve")) {
                    preserves.addAll(split(requireNextArg(i, arg), ","));
                } else if (arg.equals("-verbose")) {
                    verbose = true;
                } else {
                    System.err.println("Unrecognized option: " + arg);
                    System.exit(1);
                }
            } else {
                System.err.println("Unrecognized argument: " + arg);
                System.exit(1);
            }
        }
        
        if (output == null || preserves.isEmpty()) {
            System.err.println("usage: java " +
                               Extractor.class.getName() +
                               " [-classpath path] [-includepkg package,...]" +
                               " [-includecls class,...]" +
                               " [-preserve method|field] [-output dir|zip]" +
                               " [-verbose]");
            System.exit(1);
        }

        ClassPattern includePattern = new ClassPattern();

        for (Iterator itr = includePackages.iterator(); itr.hasNext(); ) {
            includePattern.addPackagePattern((String)itr.next());
        }
        for (Iterator itr = includeClasses.iterator(); itr.hasNext(); ) {
            includePattern.addClassPattern((String)itr.next());
        }

        Extractor extr = new Extractor(classPath, includePattern);
        extr.setLogStream(System.out);
        extr.setVerbose(verbose);

        extr.log("Marking ...");

        for (Iterator itr = preserves.iterator(); itr.hasNext(); ) {
            String item = (String)itr.next();
            if (item.indexOf('(') != -1) {
                extr.preserveMethod(item);
            } else {
                extr.preserveField(item);
            }
        }

        extr.process();

        String out = output.toLowerCase();
        if (out.endsWith(".zip") || out.endsWith(".jar")) {
            extr.dumpToZipFile(new File(output));
        } else {
            extr.dumpToDirectory(new File(output));
        }
    }

    private static String requireNextArg(Iterator itr, String option) {
        if (!itr.hasNext()) {
            System.err.println("Missing argument: " + option);
            System.exit(1);
        }

        return (String)itr.next();
    }

    public static List split(String str, String delim) {
        StringTokenizer st = new StringTokenizer(str, delim);
        List list = new ArrayList(st.countTokens());

        while (st.hasMoreTokens())
            list.add(st.nextToken());

        return list;
    }
}
