/*
 * $Id: Marker.java,v 1.10 2003/04/06 01:50:48 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.extractor;

import java.io.IOException;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.classfile.Field;
import org.apache.bcel.classfile.Code;
import org.apache.bcel.classfile.ConstantCP;
import org.apache.bcel.classfile.ConstantNameAndType;
import org.apache.bcel.classfile.ConstantPool;
import org.apache.bcel.generic.FieldOrMethod;
import org.apache.bcel.generic.Instruction;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InvokeInstruction;
import org.apache.bcel.generic.FieldInstruction;
import org.apache.bcel.generic.InstructionList;

/**
 * 饹ե뷲ɬפʥ᥽åɡեɡ饹ޡ롣
 */
public class Marker {
    private Repository m_repository;
    private ClassPattern m_includePattern;
    private Mark m_mark;
    private ClassMark m_parametricMark;

    /**
     * 󥹥ȥ饯
     *
     * @param  repository      饹ɤ߹ߤ˻Ȥݥȥꡣ
     * @param  includePattern  Ϸ̤˴ޤޤǽΤ륯饹
     *                         ѥåΥѥ
     * @param  mark            Mark ֥ȡޡ̤
     *                         ֥ȤѤ롣
     */
    public Marker(Repository repository, ClassPattern includePattern,
                  Mark mark) {
        m_repository = repository;
        m_includePattern = includePattern;
        m_mark = mark;
        m_parametricMark = new ClassMark();
    }

    /**
     * ꤵ줿᥽åɤФоݤȤƥޡ롣
     * ޤ᥽åɤΥɤˤäƻȤƤ¾Υƥƥ
     * ƵŪ˥ޡ롣
     *
     * <p>: ŪˤϡФоݤȤʤ᥽åɤǤʤ
     * ФоݤȤʤ᥽åɤ conformant ʿƥ饹Υ᥽å
     * ̤˱ƶ뤢᥽åɤޡΤ˻Ȥ롣</p>
     *
     * @param  className  饹̾
     * @param  name       ᥽å̾
     * @param  signature  Υͥ㡣
     *                    "(ꥹ)֤" ηǤ⡢
     *                    "(ꥹ)" ηǤ⡢
     *                    "ꥹ" ηǤ褤
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤ᥽å
     *                         եɤäȤꤲ롣
     */
    public void preserveMethod(String className, String name, String signature)
        throws IOException, LinkageException {
        signature = ClassUtils.canonicalizeMethodSignature(signature);

        JavaClass clazz = m_repository.lookupClass(className);
        Method[] methods = clazz.getMethods();
        Method method = lookupMethod(methods, name, signature);
        if (method == null) {
            /* ƥ饹Ͽƥ󥿡եõ */
            markSuperclassMethod(clazz, name, signature);
            return;
        }

        preserveMethod(clazz, method);
    }

    /**
     * preserveMethod(String,String,String) β
     */
    public void preserveMethod(JavaClass clazz, Method method)
        throws IOException, LinkageException {

        assertMethodBelongsToClass(method, clazz);

        String className = clazz.getClassName();
        String name = method.getName();
        String signature =
            ClassUtils.canonicalizeMethodSignature(method.getSignature());

        if (!m_mark.isMethodMarked(className, name, signature)) {
            m_mark.setMethodMarked(className, name, signature);

            if (m_includePattern.matches(className)) {
                Code code = method.getCode();
                if (code != null) {
                    markCode(code);
                }
            } else {
                /* ַ̤˴ޤʤ饹פΥ᥽åɤƤӽФƤơ
                   ġΰ˻ȷϤƤϡ
                   Ϥ줿ȤΥ֥饹λ accessible 
                   ᥽åƤޡɬפΤǡ᥽åɤ
                   եɤΥޡȤ̤˥ޡƤ
                 */
                int len = signature.length();
                for (int i = 1; i < len; i++) {
                    if (signature.charAt(i) == 'L') {
                        int end = signature.indexOf(';', i + 1);
                        String argClsName = signature.substring(i + 1, end);
                        m_parametricMark.setMarked(argClsName);
                        i = end;
                    }
                }
            }
        }
    }

    /**
     * ƥ饹/󥿡եƱͥΥ᥽åɤޡ롣
     */
    private void markSuperclassMethod(JavaClass clazz, String name,
                                      String signature)
        throws IOException, LinkageException {
        String className = clazz.getClassName();

        if (clazz.isInterface()) {
            String[] parents = clazz.getInterfaceNames();
            if (parents.length == 0) {
                throw new LinkageException("No such method: " + className +
                                           "." + name + signature);
            }
            boolean foundAny = false;
            for (int i = 0; i < parents.length; i++) {
                try {
                    preserveMethod(parents[i], name, signature);
                    foundAny = true;
                } catch (LinkageException le) { }
            }
            if (!foundAny) {
                throw new LinkageException("No such method: " + className +
                                           "." + name + signature);
            }
        } else {
            if (clazz.getSuperclassNameIndex() == 0) {
                throw new LinkageException("No such method: " + className +
                                           "." + name + signature);
            }
            preserveMethod(clazz.getSuperclassName(), name, signature);
        }
    }

    /**
     * ꤵ줿եɤФоݤȤƥޡ롣
     *
     * @param  className  饹̾
     * @param  name       ե̾
     *
     * @exception IOException  饹Υ I/O 顼
     *                         ȯȤꤲ롣
     * @exception LinkageException
     *                         ޡ˸Ĥʤ᥽å
     *                         եɤäȤꤲ롣
     */
    public void preserveField(String className, String name)
        throws IOException, LinkageException {
        if (!m_includePattern.matches(className)) {
            return;
        }

        JavaClass clazz = m_repository.lookupClass(className);
        Field[] fields = clazz.getFields();
        Field field = lookupField(fields, name);
        if (field == null) {
            /* Ĥʤпƥ饹õ */
            if (!clazz.isInterface() && clazz.getSuperclassNameIndex() != 0) {
                preserveField(clazz.getSuperclassName(), name);
                return;
            } else {
                throw new LinkageException("No such field: " + className +
                                           "." + name);
            }
        }

        m_mark.setFieldMarked(className, name);

        String type = ClassUtils.getComponentObject(field.getSignature());
        if (type != null && m_includePattern.matches(type)) {
            m_repository.lookupClass(type);
        }
    }

    /**
     * 󥹥ȥ饯󥳡˴ޤޤ᥽åɤեɤؤ
     * Ȥޡ롣
     *
     * @param  code  Code °֥ȡ
     */
    private void markCode(Code code) throws IOException, LinkageException {
        InstructionList list = new InstructionList(code.getCode());
        ConstantPool cp = code.getConstantPool();

        Iterator itr = list.iterator();
        while (itr.hasNext()) {
            Instruction inst =
                ((InstructionHandle)itr.next()).getInstruction();
            if (inst instanceof InvokeInstruction) {
                InvokeInstruction invokeInst = (InvokeInstruction)inst;
                preserveMethod(getInstClassName(invokeInst, cp),
                               getInstName(invokeInst, cp),
                               getInstSignature(invokeInst, cp));
            } else if (inst instanceof FieldInstruction) {
                FieldInstruction fieldInst = (FieldInstruction)inst;
                preserveField(getInstClassName(fieldInst, cp),
                              getInstName(fieldInst, cp));
            }
        }
    }

    /**
     * org.apache.bcel.generic.FieldOrMethod#getClassName(ConstantPoolGen)
     * ʡConstantPoolGen ͳʤΤǡ꥽񤬾ʤ
     */
    private static String getInstClassName(FieldOrMethod fm, ConstantPool cp) {
        ConstantCP cmr = (ConstantCP)cp.getConstant(fm.getIndex());
        return cmr.getClass(cp);
    }

    /**
     * org.apache.bcel.generic.FieldOrMethod#getName(ConstantPoolGen)
     * ʡConstantPoolGen ͳʤΤǡ꥽񤬾ʤ
     */
    private static String getInstName(FieldOrMethod fm, ConstantPool cp) {
        ConstantCP cmr  = (ConstantCP)cp.getConstant(fm.getIndex());
        ConstantNameAndType cnat =
            (ConstantNameAndType)cp.getConstant(cmr.getNameAndTypeIndex());
        return cnat.getName(cp);
    }

    /**
     * org.apache.bcel.generic.FieldOrMethod#getSignature(ConstantPoolGen)
     * ʡConstantPoolGen ͳʤΤǡ꥽񤬾ʤ
     */
    private static String getInstSignature(FieldOrMethod fm, ConstantPool cp) {
        ConstantCP cmr  = (ConstantCP)cp.getConstant(fm.getIndex());
        ConstantNameAndType cnat =
            (ConstantNameAndType)cp.getConstant(cmr.getNameAndTypeIndex());
        return cnat.getSignature(cp);
    }

    /**
     * ݥȥ˳ǼƤ뤹٤ƤΥ饹ˤĤơ
     * Υ饹οƥ饹䡢Υ饹Ƥ륤󥿡ե
     * ꤵƤ᥽åɤޡƤʤХޡ롣
     *
     * @return 1ĤǤ᥽åɤޡ true ֤
     *         true ֤硢ϢŪ˥ޡ٤᥽åɤ
     *         ǽΤǡ⤦ markPolymorphic() 
     *         ƤӽФɬפ롣
     */
    private boolean markPolymorphic() throws IOException, LinkageException {
        boolean marked = false;

        JavaClass[] classes = m_repository.getClasses();
        for (int i = 0; i < classes.length; i++) {
            JavaClass clazz = classes[i];
            if (!clazz.isInterface()) {
                String className = clazz.getClassName();
                Method[] methods = clazz.getMethods();
                for (int j = 0; j < methods.length; j++) {
                    Method method = methods[j];
                    if (!method.isStatic() &&
                        !method.isPrivate() &&
                        !m_mark.isMethodMarked(className,
                                               method.getName(),
                                               method.getSignature())) {
                        if (/* ƥ饹Ʊ̾Υ᥽åɤ뤫
                               󥿡եƱ̾Υ᥽åɤꡢ
                               ޡƤС
                               Υ饹᥽åɤޡ */
                            isMethodMarkedParent(clazz,
                                                 method.getName(),
                                                 method.getSignature(),
                                                 clazz.getPackageName()) ||
                            /* 󥿡ե abstract ǵꤵ
                               ᥽åɤʤСɬޡ */
                            (m_includePattern.matches(className) &&
                             isMethodUnresolvedAbstract(clazz,
                                                        method.getName(),
                                                        method.getSignature()))) {
                            preserveMethod(clazz, method);
                            marked = true;
                        }
                    }
                }
            }
        }

        return marked;
    }

    /**
     * ƥ饹ϼ󥿡եƱ̾Υ᥽åɤꡢ줬
     * ޡƤ뤫ɤå롣
     *
     * @param  clazz        õ򳫻Ϥ륯饹
     * @param  name         ᥽å̾
     * @param  signature    ᥽åɥͥ㡣
     * @param  origPackage  õ᥽åɤΤ륯饹Υѥå̾
     *                      С饤ɲǽå뤿ɬס
     * @return ޡƤ true
     */
    private boolean isMethodMarkedParent(JavaClass clazz,
                                         String name, String signature,
                                         String origPackage)
        throws IOException {
        /* 󥹥ȥ饯ϷѾʤ */
        if (name.equals("<init>")) {
            return false;
        }

        /* 󥿡եĴ٤ */

        String[] ifs = clazz.getInterfaceNames();
        for (int i = 0; i < ifs.length; i++) {
            JavaClass iface = m_repository.lookupClass(ifs[i]);
            Method[] methods = iface.getMethods();
            Method ifMethod = lookupMethod(methods, name, signature);
            if (ifMethod != null) {
                if (m_mark.isMethodMarked(ifs[i], name, signature)) {
                    return true;
                }
            }
            if (isMethodMarkedParent(iface, name, signature, origPackage)) {
                return true;
            }
        }

        /* ƥ饹Ĵ٤ */

        if (clazz.isInterface() || clazz.getSuperclassNameIndex() == 0) {
            return false;
        }

        JavaClass superClass =
            m_repository.lookupClass(clazz.getSuperclassName());
        Method[] methods = superClass.getMethods();
        Method superMethod = lookupMethod(methods, name, signature);
        if (superMethod != null) {
            if (superMethod.isPrivate() ||
                (!superMethod.isProtected() && !superMethod.isPublic() &&
                 !superClass.getPackageName().equals(origPackage))) {
                return false;
            } else if (m_mark.isMethodMarked(superClass.getClassName(),
                                             name, signature)) {
                return true;
            }
        }

        return isMethodMarkedParent(superClass, name, signature, origPackage);
    }

    /**
     * ᥽åɤ̤˴ޤޤʤ󥿡եƥ饹
     * abstract ȤƵꤵƤơ褵Ƥʤ true 
     * ֤
     *
     * @param  clazz        õ򳫻Ϥ륯饹
     * @param  name         ᥽å̾
     * @param  signature    ᥽åɥͥ㡣
     * @return abstract 褵Ƥʤ true
     */
    private boolean isMethodUnresolvedAbstract(JavaClass clazz,
                                               String name, String signature)
        throws IOException {
        /* 󥹥ȥ饯ϷѾʤ */
        if (name.equals("<init>")) {
            return false;
        }

        /* 󥿡եĴ٤ */

        String[] ifs = clazz.getInterfaceNames();
        for (int i = 0; i < ifs.length; i++) {
            String ifname = ifs[i];
            JavaClass iface = m_repository.lookupClass(ifname);
            if (!m_includePattern.matches(ifname)) {
                Method[] methods = iface.getMethods();
                Method ifMethod = lookupMethod(methods, name, signature);
                if (ifMethod != null)
                    return true;
            }
            if (isMethodUnresolvedAbstract(iface, name, signature)) {
                return true;
            }
        }

        /* ƥ饹Ĵ٤ */

        if (clazz.isInterface()) {
            return false;
        }

        while (clazz.getSuperclassNameIndex() != 0) {
            JavaClass superClass =
                m_repository.lookupClass(clazz.getSuperclassName());
            Method[] methods = superClass.getMethods();
            Method superMethod = lookupMethod(methods, name, signature);
            if (superMethod != null) {
                if (superMethod.isAbstract() &&
                    !m_includePattern.matches(superClass.getClassName())) {
                    return true;
                } else {
                    return false;
                }
            }
            
            clazz = superClass;
        }
        
        return false;
    }

    /**
     * ݥȥ˳ǼƤơ̤˴ޤ*ʤ*٤ƤΥ饹ˤĤơ
     * Υ饹οƥ饹䡢Υ饹Ƥ륤󥿡ե
     * ꤵƤ᥽åɤַ̤˴ޤʤ饹פΥ᥽åɤ
     * ȤϤƤФΥ饹Ƥ accessible 
     * ᥽åɤޡ롣
     *
     * <p>: ɤƷ̤˴ޤʤ饹ޡ뤫
     * ⤷̤˴ޤ륯饹ޡȡ
     * java.lang.String#equals(Java.lang.Object) ʤɤƤФ줿
     * ۤȤɤ٤ƤΥ᥽åɤޡƤޤᡣ
     * 򤱤뤿ˡַ̤˴ޤʤ饹ϡ̤˴ޤ륯饹
     * ˤĤΤʤפȤ֤˴ŤơѤ
     * ǽΤƤΥ᥽åɤޡ롣
     * ̤˴ޤʤ饹ޡƤ츫̣ʤ褦˻פ뤬
     * markPolymorphic() ǥޡΤǰ̣롣</p>
     *
     * @return 1ĤǤ᥽åɤޡ true ֤
     *         true ֤硢ϢŪ˥ޡ٤᥽åɤ
     *         ǽ롣
     */
    private boolean markParametric() throws IOException, LinkageException {
        boolean marked = false;

        JavaClass[] classes = m_repository.getClasses();
        for (int i = 0; i < classes.length; i++) {
            JavaClass clazz = classes[i];
            String className = clazz.getClassName();
            if (!m_includePattern.matches(className)) {
                if (isClassParametric(clazz)) {
                    /* Ƥ accessible  non-static ʥ᥽åɤ
                       ޡ */
                    Method[] methods = clazz.getMethods();
                    for (int j = 0; j < methods.length; j++) {
                        Method method = methods[j];
                        if (!method.isStatic() &&
                            !method.isPrivate() &&
                            !method.getName().equals("<init>") &&
                            !m_mark.isMethodMarked(className,
                                                   method.getName(),
                                                   method.getSignature())) {
                            preserveMethod(clazz, method);
                            marked = true;
                        }
                    }
                }
            }
        }

        return marked;
    }

    /**
     * Ϳ줿饹ַ̤˴ޤʤ饹פΥ᥽åɤΰȤ
     * ϤƤ뷿Υ֥饹Ǥäꡢ֥󥿡եǤä
     *  true ֤
     */
    private boolean isClassParametric(JavaClass clazz) throws IOException {

        /* ʬȤĴ٤ */

        String className = clazz.getClassName();

        if (m_parametricMark.isMarked(className)) {
            return true;
        }

        /* 󥿡եĴ٤ */

        String[] ifs = clazz.getInterfaceNames();
        for (int i = 0; i < ifs.length; i++) {
            String ifname = ifs[i];
            if (m_parametricMark.isMarked(ifname)) {
                m_parametricMark.setMarked(className);       // Ψ
                return true;
            }
            JavaClass iface = m_repository.lookupClass(ifname);
            if (isClassParametric(iface)) {
                m_parametricMark.setMarked(className);       // Ψ
                return true;
            }
        }

        /* ƥ饹Ĵ٤ */

        if (clazz.isInterface() || clazz.getSuperclassNameIndex() == 0) {
            return false;
        }

        JavaClass superClass =
            m_repository.lookupClass(clazz.getSuperclassName());

        if (isClassParametric(superClass)) {
            m_parametricMark.setMarked(className);       // Ψ
            return true;
        }

        return false;
    }

    /**
     * ݥȥ˳ǼƤơ̤˴ޤͽΤ٤Ƥ
     * 饹ˤĤơ᥽å <clinit>() ޡ롣
     *
     * @return 1ĤǤ᥽åɤޡ true ֤
     *         true ֤硢ϢŪ˥ޡ٤᥽åɤ
     *         ǽ롣
     */
    private boolean markClinit()
        throws IOException, LinkageException {
        boolean marked = false;

        JavaClass[] classes = m_repository.getClasses();
        for (int i = 0; i < classes.length; i++) {
            JavaClass clazz = classes[i];
            String className = clazz.getClassName();
            if (m_includePattern.matches(className)) {
                Method[] methods = clazz.getMethods();
                String name = "<clinit>";
                String signature = "()";
                Method method = lookupMethod(methods, name, signature);
                if (method != null &&
                    !m_mark.isMethodMarked(className,
                                           name, signature)) {
                    preserveMethod(clazz, method);
                    marked = true;
                }
            }
        }

        return marked;
    }

    /**
     * ޡθ򤹤롣
     */
    public void postProcess() throws IOException, LinkageException {
        boolean cont;
        do {
            cont = markParametric();
            cont |= markPolymorphic();
            cont |= markClinit();
        } while (cont);
    }

    /**
     * Ϳ줿̾Ȱͥ˥ޥå᥽åɤ󤫤
     * 롣
     *
     * @param  methods    Method 
     * @param  name       ᥽åɤ̾
     * @param  signature  ͥ㡣
     * @return ĤäϤ Method ֥ȡ
     *         Ĥʤ null
     */
    private static Method lookupMethod(Method[] methods,
                                       String name, String signature) {
        signature = ClassUtils.canonicalizeMethodSignature(signature);

        for (int i = 0; i < methods.length; i++) {
            if (methods[i].getName().equals(name) &&
                methods[i].getSignature().startsWith(signature)) {
                return methods[i];
            }
        }

        return null;
    }

    /**
     * Ϳ줿̾˥ޥåեɤ󤫤饵롣
     *
     * @param  fields  Field 
     * @param  name    եɤ̾
     * @return ĤäϤ Field ֥ȡ
     *         Ĥʤ null
     */
    private static Field lookupField(Field[] fields, String name) {
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getName().equals(name)) {
                return fields[i];
            }
        }

        return null;
    }

    /**
     * method ֥Ȥ clazz ֥Ȥ°Ƥ뤫ɤ
     * å롣
     */
    private static void assertMethodBelongsToClass(Method method,
                                                   JavaClass clazz) {
        Method[] methods = clazz.getMethods();
        for (int i = 0; i < methods.length; i++) {
            if (methods[i] == method) {
                return;
            }
        }

        throw new RuntimeException("Assertion failed");
    }
}
