/*
 * $Id: Remover.java,v 1.4 2003/04/06 01:50:48 ymakise Exp $
 */

/*
 * Copyright (c) 2002-2003, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.extractor;

import java.io.IOException;
import java.io.File;
import org.apache.bcel.Constants;
import org.apache.bcel.classfile.ClassParser;
import org.apache.bcel.classfile.Code;
import org.apache.bcel.classfile.CodeException;
import org.apache.bcel.classfile.Constant;
import org.apache.bcel.classfile.ConstantPool;
import org.apache.bcel.classfile.ConstantValue;
import org.apache.bcel.classfile.ExceptionTable;
import org.apache.bcel.classfile.Field;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.generic.ClassGen;
import org.apache.bcel.generic.ConstantPoolGen;
import org.apache.bcel.generic.FieldGen;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;
import org.apache.bcel.generic.MethodGen;
import org.apache.bcel.generic.ObjectType;
import org.apache.bcel.generic.Type;

/**
 * ޡȤˡ饹ե뤫פʥ᥽åɤեɤ롣
 * ѤȤơץμ¹Ԥפ Attribute 롣
 * ޤ󥹥ȥספʥȥ롣
 */
public class Remover {
    private Mark m_mark;

    /**
     * 󥹥ȥ饯
     *
     * @param  mark  ᥽åɤեɤ/פ̤ޡ
     */
    public Remover(Mark mark) {
        m_mark = mark;
    }

    /**
     * 饹ե (JavaClass ֥) פʥ᥽åɤ
     * եɤ롣
     * /פȽ̤ϡ󥹥ȥ饯Ϳ줿 Mark ֥
     * ѤƹԤʤ
     * ѤȤơץμ¹Ԥפ Attribute 롣
     * ޤ󥹥ȥספʥȥ롣
     *
     * @param  clazz  о JavaClass ֥ȡ
     * @return  JavaClass ֥ȡ
     */
    public JavaClass processClass(JavaClass clazz) {
        String className = clazz.getClassName();
        ConstantPoolGen origCpGen =
            new ConstantPoolGen(clazz.getConstantPool());

        ClassGen classGen = new ClassGen(className,
                                         clazz.getSuperclassName(),
                                         className.replace('.', '/'), // fake
                                         clazz.getAccessFlags(),
                                         clazz.getInterfaceNames());
        ConstantPoolGen cpGen = classGen.getConstantPool();

        Method[] methods = clazz.getMethods();
        for (int i = 0; i < methods.length; i++) {
            Method method = methods[i];
            String name = method.getName();
            String signature = method.getSignature();
            if (m_mark.isMethodMarked(className, name, signature)) {
                classGen.addMethod(processMethod(method, origCpGen, cpGen));
            }
        }

        Field[] fields = clazz.getFields();
        for (int i = 0; i < fields.length; i++) {
            Field field = fields[i];
            String name = field.getName();
            if (m_mark.isFieldMarked(className, name)) {
                classGen.addField(processField(field, origCpGen, cpGen));
            }
        }

        return classGen.getJavaClass();
    }

    private Method processMethod(Method method,
                                 ConstantPoolGen origCpGen,
                                 ConstantPoolGen cpGen) {
        InstructionList il = null;
        Code code = method.getCode();
        if (code != null) {
            il = processInstructionList(code.getCode(), origCpGen, cpGen);
        }

        MethodGen methodGen =
            new MethodGen(method.getAccessFlags(),
                          Type.getReturnType(method.getSignature()),
                          Type.getArgumentTypes(method.getSignature()),
                          null,
                          method.getName(),
                          null,
                          il,
                          cpGen);
        methodGen.stripAttributes(true);

        if (code != null) {
            methodGen.setMaxStack(code.getMaxStack());
            methodGen.setMaxLocals(code.getMaxLocals());

            CodeException[] ces = code.getExceptionTable();
        
            if(ces != null) {
                for (int j = 0; j < ces.length; j++) {
                    CodeException ce     = ces[j];
                    int           type   = ce.getCatchType();
                    ObjectType    c_type = null;

                    if(type > 0) {
                        String cen = method.getConstantPool().
                            getConstantString(type, Constants.CONSTANT_Class);
                        c_type = new ObjectType(cen);
                    }

                    int end_pc = ce.getEndPC();
                    int length = code.getCode().length;
            
                    InstructionHandle end;

                    if(length == end_pc) { // May happen, because end_pc is exclusive
                        end = il.getEnd();
                    } else {
                        end = il.findHandle(end_pc);
                        end = end.getPrev(); // Make it inclusive
                    }

                    methodGen.addExceptionHandler(
                        il.findHandle(ce.getStartPC()),
                        end,
                        il.findHandle(ce.getHandlerPC()),
                        c_type);
                }
            }
        }

        ExceptionTable et = method.getExceptionTable();
        if (et != null) {
            String[] names = et.getExceptionNames();
            for (int j = 0; j < names.length; j++) {
                methodGen.addException(names[j]);
            }
        }

        return methodGen.getMethod();
    }

    private InstructionList processInstructionList(byte[] code,
                                                   ConstantPoolGen origCpGen,
                                                   ConstantPoolGen cpGen) {
        InstructionList il = new InstructionList(code);
        il.replaceConstantPool(origCpGen, cpGen);
        return il;
    }

    private Field processField(Field field,
                               ConstantPoolGen origCpGen,
                               ConstantPoolGen cpGen) {
        FieldGen fieldGen =
            new FieldGen(field.getAccessFlags(),
                         Type.getType(field.getSignature()),
                         field.getName(),
                         cpGen);

        ConstantValue cv = field.getConstantValue();
        if (cv != null) {
            Constant val = origCpGen.getConstant(cv.getConstantValueIndex());
            int idx = cpGen.addConstant(val, origCpGen);
            ConstantValue newcv =
                new ConstantValue(cpGen.addUtf8("ConstantValue"),
                                  2, idx, cpGen.getConstantPool());
            fieldGen.addAttribute(newcv);
        }

        return fieldGen.getField();
    }
}
