<?php
// $Horde: turba/lib/List.php,v 1.13.2.3 2002/04/12 16:10:07 jan Exp $

/**
 * The Turba_List:: class provides an interface for dealing with a
 * list of Turba_AbstractObjects.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.1.1.1 $
 * @since   Turba 0.0.1
 * @package turba
 */
class Turba_List {

    /** The array containing the Turba_Objects represented in this
        list. */
    var $objects = array();

    /** An array of objects which have just been added during this
        page load. */
    var $fresh = array();

    /**
     * Constructs a new Turba_List object.
     */
    function Turba_List() {}

    /**
     * Inserts a new list.
     *
     * @param          object Turba_Object $object  The object to insert.
     * @param optional boolean             $new     This object is from a new search (defaults to true).
     */
    function insert($object, $new = true)
    {
        $key = $object->source->name . ':' . $object->getValue('__key');
        if (!isset($this->objects[$key])) {
            if ($new) {
                $this->fresh[$key] = 1;
            }
            $this->objects[$key] = $object;
        }
    }

    /**
     * Merges an existing Turba_List into this one.
     *
     * @param          mixed   $list  The list to merge - either a Turba_List object or an array.
     * @param optional boolean $new   These objects are coming from a new search (defaults to true).
     */
    function merge($list, $new = true)
    {
        if (is_object($list)) {
            $list = $list->objects;
        }
        if (is_array($list)) {
            foreach ($list as $object) {
                $this->insert($object, $new);
            }
        }
    }

    /**
     * Reset our internal pointer to the beginning of the list. Use
     * this to hide the internal storage (array, list, etc.) from
     * client objects.
     */
    function reset()
    {
        reset($this->objects);
    }

    /**
     * Return the next Turba_Object in the list. Use this to hide
     * internal implementation details from client objects.
     *
     * @return Turba_Object $object   The next object in the list.
     */
    function next()
    {
        list(,$tmp) = each($this->objects);
        return $tmp;
    }

    /**
     * Return the number of Turba_Objects that are in the list. Use
     * this to hide internal implementation details from client
     * objects.
     *
     * @return integer $count The number of objects in the list.
     */
    function count()
    {
        return count($this->objects);
    }

    /**
     * Filters/Sorts the list based on the specified sort routine.
     *
     * @param $sort         The sort method.
     * @param $low          The low end of the sort range.
     * @param $high         The high end of the sort range.
     */
    function sort($sort = 'lastname', $low = null, $high = null)
    {
        if (isset($low))  $low = strtolower($low);
        if (isset($high)) $high = strtolower($high);
        $sorted_objects = array();

        foreach ($this->objects as $key => $object) {
            $lastname = $this->guessLastname($object->getValue('name'));
            $object->setValue('lastname', $lastname);
            $lower = strtolower($lastname);
            if ((!isset($low)  || (substr($lower, 0, strlen($low)) >= $low)) &&
                (!isset($high) || (substr($lower, 0, strlen($high)) <= $high))) {
                $sorted_objects[$key] = $object;
            }
        }

        $GLOBALS['usortCriteria'] = $sort;
        usort($sorted_objects, array($this, 'cmp'));

        $this->objects = $sorted_objects;
    }

    /**
     * Returns a best guess at the lastname in a string.
     *
     * @param $name     String contain the full name.
     *
     * @return          String containing the last name.
     */
    function guessLastname($name)
    {
        if (!empty($name)) {
            /* Assume that last names are always before any commas. */
            if (is_int(strpos($name, ','))) {
                $name = substr($name, 0, strpos($name, ','));
            }

            /* Take out anything in parentheses. */
            $name = preg_replace('|\(.*\)|', '', $name);

            /* Remove single quotes. */
            $name = str_replace('\'', '', $name);

            $name = trim(preg_replace('|\s+|', ' ', $name));
            $namelist = explode(' ', $name);
            $name = $namelist[($nameindex = (count($namelist) - 1))];

            while (strlen($name) < 5 &&
                   strspn($name[(strlen($name) - 1)], '.:-') &&
                   isset($namelist[($nameindex - 1)])) {
                $nameindex--;
                $name = $namelist[$nameindex];
            }
        }
        return $name;
    }

    /**
     * Usort helper function. Compares two Turba_AbstractObjects
     * based on the global variable $usortCriteria.
     *
     * @param $a        The first Turba_AbstractObject to compare.
     * @param $b        The second Turba_AbstroctObject to compare.
     *
     * @return          Integer comparison of the two addresses from strcmp().
     */
    function cmp($a, $b)
    {
        global $usortCriteria;

        $acmp = strtolower($a->getValue($usortCriteria));
        $bcmp = strtolower($b->getValue($usortCriteria));

        return strcmp($acmp, $bcmp);
    }

    function isFresh($object)
    {
        return isset($this->fresh[$object->source->name . ':' . $object->getValue('__key')]);
    }

    function serialize()
    {
        $data = array();
        foreach ($this->objects as $key => $object) {
            $data[$key] = $object->attributes;
        }
        return $data;
    }

    function unserialize($data)
    {
        if (!isset($data) || !is_array($data)) {
            return null;
        }

        $objects = array();
        foreach ($data as $key => $object) {
            list($source, $key) = explode(':', $key);
            $sourceOb = &Turba_Source::singleton($source, $GLOBALS['cfgSources'][$source]);
            $objects[] = new Turba_Object($sourceOb, $object);
        }

        $tmp = new Turba_List();
        $tmp->merge($objects, false);
        return $tmp;
    }

}
?>
