/******************************************************************************/
/*! @file printbuf.cc
    @brief json-c.
    @author Masashi Astro Tachibana, Apolloron Project.
 ******************************************************************************/

/*
 * $Id: printbuf.c,v 1.5 2006/01/26 02:16:28 mclark Exp $
 *
 * Copyright (c) 2004, 2005 Metaparadigm Pte. Ltd.
 * Michael Clark <michael@metaparadigm.com>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the MIT license. See COPYING for details.
 *
 *
 * Copyright (c) 2008-2009 Yahoo! Inc.  All rights reserved.
 * The copyrights to the contents of this file are licensed under the MIT License
 * (http://www.opensource.org/licenses/mit-license.php)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "apolloron.h"

#include "bits.h"
#include "printbuf.h"

#define vasprintf(x,y,z) _vasprintf((x),(y),(z))
namespace {
int _vasprintf(char **buf, const char *fmt, va_list ap) {
#ifndef WIN32
    char _T_emptybuffer = '\0';
#endif /* !defined(WIN32) */
    int chars;
    char *b;

    if (!buf) {
        return -1;
    }

#ifdef WIN32
    chars = _vscprintf(fmt, ap)+1;
#else /* !defined(WIN32) */
    /* CAW: RAWR! We have to hope to god here that vsnprintf doesn't overwrite
       our buffer like on some 64bit sun systems.... but hey, its time to move on */
    chars = vsnprintf(&_T_emptybuffer, 0, fmt, ap)+1;
    if(chars < 0) {
        chars *= -1;    /* CAW: old glibc versions have this problem */
    }
#endif /* defined(WIN32) */

    b = new char [chars];
    if (!b) {
        return -1;
    }

    if ((chars = vsprintf(b, fmt, ap)) < 0) {
        delete [] b;
    } else {
        *buf = b;
    }

    return chars;
}

}

namespace apolloron {

struct printbuf* printbuf_new(void) {
    struct printbuf *p;

    p = new struct printbuf;
    memset(p, 0, sizeof(struct printbuf));
    if(!p) return NULL;
    p->size = 32;
    p->bpos = 0;
    if(!(p->buf = new char [p->size])) {
        delete p;
        return NULL;
    }
    return p;
}


int printbuf_memappend(struct printbuf *p, const char *buf, int size) {
    char *t;
    if(p->size - p->bpos <= size) {
        int new_size = json_max(p->size * 2, p->bpos + size + 8);
        if(!(t = new char [new_size])) return -1;
        if (p->buf) {
            memcpy(t, p->buf, p->size);
            delete [] p->buf;
        }
        p->size = new_size;
        p->buf = t;
    }
    memcpy(p->buf + p->bpos, buf, size);
    p->bpos += size;
    p->buf[p->bpos]= '\0';
    return size;
}

int sprintbuf(struct printbuf *p, const char *msg, ...) {
    va_list ap;
    char *t = NULL;
    int size;
    char buf[128];

    /* user stack buffer first */
    va_start(ap, msg);
    size = vsnprintf(buf, 128, msg, ap);
    va_end(ap);
    /* if string is greater than stack buffer, then use dynamic string
       with vasprintf.  Note: some implementation of vsnprintf return -1
       if output is truncated whereas some return the number of bytes that
       would have been written - this code handles both cases. */
    if(size == -1 || size > 127) {
        va_start(ap, msg);
        if((size = vasprintf(&t, msg, ap)) == -1) {
            va_end(ap);
            return -1;
        }
        va_end(ap);
        printbuf_memappend(p, t, size);
        delete [] t;
        return size;
    } else {
        printbuf_memappend(p, buf, size);
        return size;
    }
}

void printbuf_reset(struct printbuf *p) {
    p->buf[0] = '\0';
    p->bpos = 0;
}

void printbuf_free(struct printbuf *p) {
    if(p) {
        delete [] p->buf;
        delete p;
    }
}

} // namespace apolloron
