package com.snail.msglet;

import org.apache.commons.configuration.XMLConfiguration;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;


/**
 * Msgletの設定ファイル msg.xml を読み込みます .
 * <pre>
 * 1) msg.xml に従って 一つのMsgletコンテナを作成します
 * 2) msg.xml に従って 複数のMsgletを作成し、Msgletコンテナに登録します
 * 3) Msgletコンテナを起動します
 * 4) 終了待ちに入ります
 * 5) "shutdown"と入力されるか、JavaVMが終了されるときには、
 *    Msgletコンテナをシャットダウンしてから終了します。
 * </pre>
 * @author kagyuu
 */
public final class Startup {
  /**
   * Msgletコンテナ自身のユーザ名の格納されている msg.xml 上の path.
   */
  private static final String USER_NAME = "context-param.user-name";

  /**
   * Msgletコンテナ自身のニックネーム名の格納されている msg.xml 上の path.
   */
  private static final String NICK_NAME = "context-param.nick-name";

  /**
   * Msgletコンテナ自身のグループ名の格納されている msg.xml 上の path.
   */
  private static final String GROUP = "context-param.group";

  /**
   * Msgletコンテナ自身のデバックモードフラグの格納されている msg.xml 上の path.
   */
  private static final String DEBUG = "context-param.debug";

  /**
   * セッションが開始されていないときに開始するMsglet名が格納されている msg.xml 上の path.
   */
  private static final String WELCOME_MSGLET = "welcome-msglet";

  /**
   * Creates a new Startup object.
   */
  private Startup() {
    super();
  }

  /**
   * @param args コマンド引数
   */
  public static void main(final String[] args) {
    MsgletContainer container = null;

    try {
      XMLConfiguration config = new XMLConfiguration("msg.xml");

      String userName = config.getString(USER_NAME, "Msglet");
      String nickName = config.getString(NICK_NAME, "Msglet");
      String group = config.getString(GROUP, "Msglet");
      boolean debug = config.getBoolean(DEBUG, false);
      String welcome = config.getString(WELCOME_MSGLET);

      container = new MsgletContainer(
          userName,
          nickName,
          group,
          debug,
          welcome);

      container.setMsgletMap(MsgletFactory.createMsglets(config));

      container.login();
      container.start();

      waitTerminate();
    } catch (Exception ex) {
      ex.printStackTrace();
      System.exit(-1);
    } finally {
      if (container != null) {
        try {
          container.logout();
        } catch (IOException e) {
          e.printStackTrace();
        }
      }
    }
  }

  /**
   * 空文字か、"shutodwn"が入力されるまで入力待ちをします .
   */
  private static void waitTerminate() {
    BufferedReader breader =
      new BufferedReader(new InputStreamReader(System.in));

    try {
      String line = null;

      while ((line = breader.readLine()) != null) {
        if ("shutdown".equals(line)) {
          break;
        }
      }
    } catch (IOException ignoreEx) {
      ignoreEx.printStackTrace();
    }
  }
}
