﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace PerPixelLighting
{
    class Program
    {
        static void Main(string[] args)
        {
            // let user select driver type

            E_DRIVER_TYPE driverType = E_DRIVER_TYPE.EDT_DIRECT3D9;

            Console.WriteLine(
                @"Please select the driver you want for this example:);
(a) Direct3D 9.0c
(b) Direct3D 8.1
(c) OpenGL 1.5
(d) Software Renderer
(e) Burning's Software Renderer
(f) NullDevice
(otherKey) exit");

            var keyInfo = Console.ReadKey(true);

            switch (keyInfo.KeyChar)
            {
                case 'a':
                    driverType = E_DRIVER_TYPE.EDT_DIRECT3D9;
                    break;
                case 'b':
                    driverType = E_DRIVER_TYPE.EDT_DIRECT3D8;
                    break;
                case 'c':
                    driverType = E_DRIVER_TYPE.EDT_OPENGL;
                    break;
                case 'd':
                    driverType = E_DRIVER_TYPE.EDT_SOFTWARE;
                    break;
                case 'e':
                    driverType = E_DRIVER_TYPE.EDT_BURNINGSVIDEO;
                    break;
                case 'f':
                    driverType = E_DRIVER_TYPE.EDT_NULL;
                    break;
                default:
                    return;
            }

            // create device

            IrrlichtDevice device = IrrlichtNet.createDevice(driverType,
                                                                new dimension2di(640, 480));

            if (device == null)
                return; // could not create selected driver.


            /*
	        Before we start with the interesting stuff, we do some simple things:
	        Store pointers to the most important parts of the engine (video driver,
	        scene manager, gui environment) to safe us from typing too much, add an
	        irrlicht engine logo to the window and a user controlled first person
	        shooter style camera. Also, we let the engine know that it should store
	        all textures in 32 bit. This necessary because for parallax mapping, we
	        need 32 bit textures.
	        */

            IVideoDriver driver = device.getVideoDriver();
            ISceneManager smgr = device.getSceneManager();
            IGUIEnvironment env = device.getGUIEnvironment();

            driver.setTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_ALWAYS_32_BIT, true);

            // add irrlicht logo
            env.addImage(driver.getTexture("../../media/irrlichtlogo2.png"),
                         new position2di(10, 10));

            // add camera
            ICameraSceneNode camera =
                smgr.addCameraSceneNodeFPS(null, 100.0f, .3f);
            camera.setPosition(new vector3df(-200, 200, -200));

            // disable mouse cursor
            device.getCursorControl().setVisible(false);


            /*
	        Because we want the whole scene to look a little bit scarier, we add
	        some fog to it. This is done by a call to IVideoDriver::setFog(). There
	        you can set various fog settings. In this example, we use pixel fog,
	        because it will work well with the materials we'll use in this example.
	        Please note that you will have to set the material flag EMF_FOG_ENABLE
	        to 'true' in every scene node which should be affected by this fog.
	        */
            driver.setFog(new SColor(0, 138, 125, 81), true, 250, 1000, 0, true);

            /*
	        To be able to display something interesting, we load a mesh from a .3ds
	        file which is a room I modeled with anim8or. It is the same room as
	        from the specialFX example. Maybe you remember from that tutorial, I am
	        no good modeler at all and so I totally messed up the texture mapping
	        in this model, but we can simply repair it with the
	        IMeshManipulator::makePlanarTextureMapping() method.
	        */

            IAnimatedMesh roomMesh = smgr.getMesh(
                "../../media/room.3ds");
            ISceneNode room = null;

            if (roomMesh != null)
            {
                smgr.getMeshManipulator().makePlanarTextureMapping(
                    roomMesh.getMesh(0), 0.003f);

                /*
		        Now for the first exciting thing: If we successfully loaded the
		        mesh we need to apply textures to it. Because we want this room
		        to be displayed with a very cool material, we have to do a
		        little bit more than just set the textures. Instead of only
		        loading a color map as usual, we also load a height map which
		        is simply a grayscale texture. From this height map, we create
		        a normal map which we will set as second texture of the room.
		        If you already have a normal map, you could directly set it,
		        but I simply didn't find a nice normal map for this texture.
		        The normal map texture is being generated by the
		        makeNormalMapTexture method of the VideoDriver. The second
		        parameter specifies the height of the heightmap. If you set it
		        to a bigger value, the map will look more rocky.
		        */

                ITexture colorMap =
                    driver.getTexture("../../media/rockwall.bmp");
                ITexture normalMap =
                    driver.getTexture("../../media/rockwall_height.bmp");

                driver.makeNormalMapTexture(normalMap, 9.0f);

                /*
		        But just setting color and normal map is not everything. The
		        material we want to use needs some additional informations per
		        vertex like tangents and binormals. Because we are too lazy to
		        calculate that information now, we let Irrlicht do this for us.
		        That's why we call IMeshManipulator::createMeshWithTangents().
		        It creates a mesh copy with tangents and binormals from another
		        mesh. After we've done that, we simply create a standard
		        mesh scene node with this mesh copy, set color and normal map
		        and adjust some other material settings. Note that we set
		        EMF_FOG_ENABLE to true to enable fog in the room.
		        */

                IMesh tangentMesh = smgr.getMeshManipulator().createMeshWithTangents(
                    roomMesh.getMesh(0));

                room = smgr.addMeshSceneNode(tangentMesh);
                room.setMaterialTexture(0, colorMap);
                room.setMaterialTexture(1, normalMap);

                room.getMaterial(0).SpecularColor.set(0, 0, 0, 0);

                room.setMaterialFlag(E_MATERIAL_FLAG.EMF_FOG_ENABLE, true);
                room.setMaterialType(E_MATERIAL_TYPE.EMT_PARALLAX_MAP_SOLID);
                // adjust height for parallax effect
                room.getMaterial(0).MaterialTypeParam = 0.035f;

                // drop mesh because we created it with a create.. call.
                //tangentMesh.drop();
            }

            /*
	        After we've created a room shaded by per pixel lighting, we add a
	        sphere into it with the same material, but we'll make it transparent.
	        In addition, because the sphere looks somehow like a familiar planet,
	        we make it rotate. The procedure is similar as before. The difference
	        is that we are loading the mesh from an .x file which already contains
	        a color map so we do not need to load it manually. But the sphere is a
	        little bit too small for our needs, so we scale it by the factor 50.
	        */

            // add earth sphere

            IAnimatedMesh earthMesh = smgr.getMesh("../../media/earth.x");
            if (earthMesh != null)
            {
                //perform various task with the mesh manipulator
                IMeshManipulator manipulator = smgr.getMeshManipulator();

                // create mesh copy with tangent informations from original earth.x mesh
                IMesh tangentSphereMesh =
                    manipulator.createMeshWithTangents(earthMesh.getMesh(0));

                // set the alpha value of all vertices to 200
                manipulator.setVertexColorAlpha(tangentSphereMesh, 200);

                // scale the mesh by factor 50
                CMatrix4f m = new CMatrix4f();
                m.setScale(new vector3df(50, 50, 50));
                manipulator.transformMesh(tangentSphereMesh, m);

                ISceneNode sphere = smgr.addMeshSceneNode(tangentSphereMesh);

                sphere.setPosition(new vector3df(-70, 130, 45));

                // load heightmap, create normal map from it and set it
                ITexture earthNormalMap = driver.getTexture("../../media/earthbump.bmp");
                driver.makeNormalMapTexture(earthNormalMap, 20.0f);
                sphere.setMaterialTexture(1, earthNormalMap);

                // adjust material settings
                sphere.setMaterialFlag(E_MATERIAL_FLAG.EMF_FOG_ENABLE, true);
                sphere.setMaterialType(E_MATERIAL_TYPE.EMT_NORMAL_MAP_TRANSPARENT_VERTEX_ALPHA);

                // add rotation animator
                ISceneNodeAnimator anim =
                    smgr.createRotationAnimator(new vector3df(0, 0.1f, 0));
                sphere.addAnimator(anim);
                //anim.drop();

                // drop mesh because we created it with a create.. call.
                //tangentSphereMesh.drop();
            }

            /*
	        Per pixel lighted materials only look cool when there are moving
	        lights. So we add some. And because moving lights alone are so boring,
	        we add billboards to them, and a whole particle system to one of them.
	        We start with the first light which is red and has only the billboard
	        attached.
	        */

            // add light 1 (nearly red)
            ILightSceneNode light1 =
                smgr.addLightSceneNode(null, new vector3df(0, 0, 0),
                                       new SColorf(0.5f, 1.0f, 0.5f, 0.0f), 800.0f);


            // add fly circle animator to light 1
            ISceneNodeAnimator anim2 =
                smgr.createFlyCircleAnimator(new vector3df(50, 300, 0), 190.0f, -0.003f);
            light1.addAnimator(anim2);
            //anim2.drop();

            // attach billboard to the light
            ISceneNode bill =
                smgr.addBillboardSceneNode(light1, new dimension2df(60, 60));

            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZWRITE_ENABLE, false);
            bill.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
            bill.setMaterialTexture(0, driver.getTexture("../../media/particlered.bmp"));

            /*
	        Now the same again, with the second light. The difference is that we
	        add a particle system to it too. And because the light moves, the
	        particles of the particlesystem will follow. If you want to know more
	        about how particle systems are created in Irrlicht, take a look at the
	        specialFx example. Maybe you will have noticed that we only add 2
	        lights, this has a simple reason: The low end version of this material
	        was written in ps1.1 and vs1.1, which doesn't allow more lights. You
	        could add a third light to the scene, but it won't be used to shade the
	        walls. But of course, this will change in future versions of Irrlicht
	        where higher versions of pixel/vertex shaders will be implemented too.
	        */

            // add light 2 (gray)
            ISceneNode light2 =
                smgr.addLightSceneNode(null, new vector3df(0, 0, 0),
                                       new SColorf(1.0f, 0.2f, 0.2f, 0.0f), 800.0f);

            // add fly circle animator to light 2
            anim2 = smgr.createFlyCircleAnimator(new vector3df(0, 150, 0), 200.0f,
                                                0.001f, new vector3df(0.2f, 0.9f, 0.0f));
            light2.addAnimator(anim2);
            //anim2.drop();

            // attach billboard to light
            bill = smgr.addBillboardSceneNode(light2, new dimension2df(120, 120));
            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            bill.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZWRITE_ENABLE, false);
            bill.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
            bill.setMaterialTexture(0, driver.getTexture("../../media/particlewhite.bmp"));

            // add particle system
            IParticleSystemSceneNode ps =
                smgr.addParticleSystemSceneNode(false, light2);

            // create and set emitter
            IParticleEmitter em = ps.createBoxEmitter(
                new aabbox3df(-3, 0, -3, 3, 1, 3),
                new vector3df(0.0f, 0.03f, 0.0f),
                80, 100,
                new SColor(0, 255, 255, 255), new SColor(0, 255, 255, 255),
                400, 1100);
            em.setMinStartSize(new dimension2df(30.0f, 40.0f));
            em.setMaxStartSize(new dimension2df(30.0f, 40.0f));

            ps.setEmitter(em);
            //em.drop();

            // create and set affector
            IParticleAffector paf = ps.createFadeOutParticleAffector();
            ps.addAffector(paf);
            //paf.drop();

            // adjust some material settings
            ps.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            ps.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZWRITE_ENABLE, false);
            ps.setMaterialTexture(0, driver.getTexture("../../media/fireball.bmp"));
            ps.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_VERTEX_ALPHA);


            MyEventReceiver receiver = new MyEventReceiver(room, env, driver);
            device.setEventReceiver(receiver);

            /*
	        Finally, draw everything. That's it.
	        */

            int lastFPS = -1;

            while (device.run())
                if (device.isWindowActive())
                {
                    driver.beginScene(true, true, new SColor());

                    smgr.drawAll();
                    env.drawAll();

                    driver.endScene();

                    int fps = driver.getFPS();

                    if (lastFPS != fps)
                    {
                        string str = "Per pixel lighting example - Irrlicht Engine [";
                        str += driver.getName();
                        str += "] FPS:";
                        str += fps;

                        device.setWindowCaption(str);
                        lastFPS = fps;
                    }
                }

            //device.drop();

            return;
        }
    }

    /*
    For this example, we need an event receiver, to make it possible for the user
    to switch between the three available material types. In addition, the event
    receiver will create some small GUI window which displays what material is
    currently being used. There is nothing special done in this class, so maybe you
    want to skip reading it.
    */
    internal class MyEventReceiver : IEventReceiver
    {
        public MyEventReceiver(ISceneNode room,
                               IGUIEnvironment env, IVideoDriver driver)
        {
            // store pointer to room so we can change its drawing mode
            Room = room;
            Driver = driver;

            // set a nicer font
            IGUISkin skin = env.getSkin();
            IGUIFont font = env.getFont("../../media/fonthaettenschweiler.bmp");
            if (font != null)
                skin.setFont(font);

            // add window and listbox
            IGUIWindow window = env.addWindow(
                new recti(460, 375, 630, 470), false, "Use 'E' + 'R' to change")
            ;

            ListBox = env.addListBox(
                new recti(2, 22, 165, 88), window);

            ListBox.addItem("Diffuse");
            ListBox.addItem("Bump mapping");
            ListBox.addItem("Parallax mapping");
            ListBox.setSelected(1);

            // create problem text
            ProblemText = env.addStaticText(
                "Your hardware or this renderer is not able to use the needed shaders for this material. Using fall back materials.",
                new recti(150, 20, 470, 80));

            ProblemText.setOverrideColor(new SColor(100, 255, 255, 255));

            // set start material (prefer parallax mapping if available)
            IMaterialRenderer renderer =
                Driver.getMaterialRenderer((uint) E_MATERIAL_TYPE.EMT_PARALLAX_MAP_SOLID);
            if (renderer != null && renderer.getRenderCapability() == 0)
                ListBox.setSelected(2);

            // set the material which is selected in the listbox
            setMaterial();
        }

        public override bool OnEvent(SEvent ev)
        {
            // check if user presses the key 'E' or 'R'
            if (ev.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT &&
                !ev.getKeyInput().PressedDown && Room != null && ListBox != null)
            {
                // change selected item in listbox

                int sel = ListBox.getSelected();
                if (ev.getKeyInput().Key == EKEY_CODE.KEY_KEY_R)
                    ++sel;
                else if (ev.getKeyInput().Key == EKEY_CODE.KEY_KEY_E)
                    --sel;
                else
                    return false;

                if (sel > 2) sel = 0;
                if (sel < 0) sel = 2;
                ListBox.setSelected(sel);

                // set the material which is selected in the listbox
                setMaterial();
            }

            return false;
        }


        // sets the material of the room mesh the the one set in the
        // list box.
        private void setMaterial()
        {
            E_MATERIAL_TYPE type = E_MATERIAL_TYPE.EMT_SOLID;

            // change material setting
            switch (ListBox.getSelected())
            {
                case 0:
                    type = E_MATERIAL_TYPE.EMT_SOLID;
                    break;
                case 1:
                    type = E_MATERIAL_TYPE.EMT_NORMAL_MAP_SOLID;
                    break;
                case 2:
                    type = E_MATERIAL_TYPE.EMT_PARALLAX_MAP_SOLID;
                    break;
            }

            Room.setMaterialType(type);

            /*
		    We need to add a warning if the materials will not be able to
		    be displayed 100% correctly. This is no problem, they will be
		    renderered using fall back materials, but at least the user
		    should know that it would look better on better hardware. We
		    simply check if the material renderer is able to draw at full
		    quality on the current hardware. The
		    IMaterialRenderer::getRenderCapability() returns 0 if this is
		    the case.
		    */
            IMaterialRenderer renderer = Driver.getMaterialRenderer((uint) type);

            // display some problem text when problem
            if (renderer == null || renderer.getRenderCapability() != 0)
                ProblemText.setVisible(true);
            else
                ProblemText.setVisible(false);
        }



        private IGUIStaticText ProblemText;
        private IGUIListBox ListBox;

        private ISceneNode Room;
        private IVideoDriver Driver;
    }

}
