﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace Meshviewer
{
    // Values used to identify individual GUI elements
    enum GUI_ID
    {
        GUI_ID_DIALOG_ROOT_WINDOW = 0x10000,

        GUI_ID_X_SCALE,
        GUI_ID_Y_SCALE,
        GUI_ID_Z_SCALE,

        GUI_ID_OPEN_MODEL,
        GUI_ID_SET_MODEL_ARCHIVE,
        GUI_ID_LOAD_AS_OCTREE,

        GUI_ID_SKY_BOX_VISIBLE,
        GUI_ID_TOGGLE_DEBUG_INFO,

        GUI_ID_DEBUG_OFF,
        GUI_ID_DEBUG_BOUNDING_BOX,
        GUI_ID_DEBUG_NORMALS,
        GUI_ID_DEBUG_SKELETON,
        GUI_ID_DEBUG_WIRE_OVERLAY,
        GUI_ID_DEBUG_HALF_TRANSPARENT,
        GUI_ID_DEBUG_BUFFERS_BOUNDING_BOXES,
        GUI_ID_DEBUG_ALL,

        GUI_ID_MODEL_MATERIAL_SOLID,
        GUI_ID_MODEL_MATERIAL_TRANSPARENT,
        GUI_ID_MODEL_MATERIAL_REFLECTION,

        GUI_ID_CAMERA_MAYA,
        GUI_ID_CAMERA_FIRST_PERSON,

        GUI_ID_POSITION_TEXT,

        GUI_ID_ABOUT,
        GUI_ID_QUIT,

        GUI_ID_TEXTUREFILTER,
        GUI_ID_SKIN_TRANSPARENCY,
        GUI_ID_SKIN_ANIMATION_FPS,

        GUI_ID_BUTTON_SET_SCALE,
        GUI_ID_BUTTON_SCALE_MUL10,
        GUI_ID_BUTTON_SCALE_DIV10,
        GUI_ID_BUTTON_OPEN_MODEL,
        GUI_ID_BUTTON_SHOW_ABOUT,
        GUI_ID_BUTTON_SHOW_TOOLBOX,
        GUI_ID_BUTTON_SELECT_ARCHIVE,

        // And some magic numbers
        MAX_FRAMERATE = 1000,
        DEFAULT_FRAMERATE = 30
    }

    class Program
    {
        static IrrlichtDevice Device = null;
        static string StartUpModelFile = "";
        static string MessageText = "";
        static string Caption = "";
        static ISceneNode Model = null;
        static ISceneNode SkyBox = null;
        static bool Octree = false;
        static bool UseLight = false;
        static ICameraSceneNode[] Camera = new ICameraSceneNode[2];
        
	    static readonly int MAX_FRAMERATE = 1000;
	    static readonly int DEFAULT_FRAMERATE = 30;

        static void Main(string[] args)
        {
            // ask user for driver

            E_DRIVER_TYPE driverType = E_DRIVER_TYPE.EDT_DIRECT3D8;
            driverType = IrrlichtNet.driverChoiceConsole();
            if (driverType == E_DRIVER_TYPE.EDT_COUNT)
            {
                return;
            }

            // create device and exit if creation failed

            SEventReceiver receiver = new SEventReceiver();
            Device = IrrlichtNet.createDevice(driverType, new dimension2dui(800, 600),
                                                 16, false, false, false, receiver);

            if (Device == null)
                return; // could not create selected driver.

            Device.setResizable(true);

            Device.setWindowCaption("Irrlicht Engine - Loading...");

            IVideoDriver driver = Device.getVideoDriver();
            IGUIEnvironment env = Device.getGUIEnvironment();
            ISceneManager smgr = Device.getSceneManager();
            smgr.getParameters().setAttribute(IrrlichtNet.COLLADA_CREATE_SCENE_INSTANCES, true);

            driver.setTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_ALWAYS_32_BIT, true);

            smgr.addLightSceneNode(null, new vector3df(200, 200, 200),
                                   new SColorf(1.0f, 1.0f, 1.0f), 2000);
            smgr.setAmbientLight(new SColorf(0.3f,0.3f,0.3f));
            // add our media directory as "search path"
            Device.getFileSystem().addFolderFileArchive("../../media/");

            /*
            The next step is to read the configuration file. It is stored in the xml
            format and looks a little bit like this:

            @verbatim
            <?xml version="1.0"?>
            <config>
                <startUpModel file="some filename" />
                <messageText caption="Irrlicht Engine Mesh Viewer">
                    Hello!
                </messageText>
            </config>
            @endverbatim

            We need the data stored in there to be written into the global variables
            StartUpModelFile, MessageText and Caption. This is now done using the
            Irrlicht Engine integrated XML parser:
            */

            // read configuration from xml file

            IIrrXMLReaderReferenceCounted xml = Device.getFileSystem().createXMLReader("config.xml");

            while (xml != null && xml.read())
            {
                switch (xml.getNodeType())
                {
                    case EXML_NODE.EXN_TEXT:
                        // in this xml file, the only text which occurs is the
                        // messageText
                        MessageText = xml.getNodeData();
                        break;
                    case EXML_NODE.EXN_ELEMENT:
                        {
                            if ("startUpModel" == xml.getNodeName())
                                StartUpModelFile = xml.getAttributeValue("file");
                            else if ("messageText" == xml.getNodeName())
                                Caption = xml.getAttributeValue("caption");
                        }
                        break;
                    default:
                        break;
                }
            }

            if (xml != null)
            {
                //xml.drop(); // don't forget to delete the xml reader
            }

            if (args.Length > 1)
                StartUpModelFile = args[1];

            /*
            That wasn't difficult. Now we'll set a nicer font and create the Menu.
            It is possible to create submenus for every menu item. The call
            menu.addItem("File", -1, true, true); for example adds a new menu
            Item with the name "File" and the id -1. The following parameter says
            that the menu item should be enabled, and the last one says, that there
            should be a submenu. The submenu can now be accessed with
            menu.getSubMenu(0), because the "File" entry is the menu item with
            index 0.
            */

            // set a nicer font

            IGUISkin skin = env.getSkin();
            IGUIFont font = env.getFont("fonthaettenschweiler.bmp");
            if (font != null)
                skin.setFont(font);

            // create menu
            IGUIContextMenu menu = env.addMenu();
            menu.addItem("File", -1, true, true);
            menu.addItem("View", -1, true, true);
            menu.addItem("Camera", -1, true, true);
            menu.addItem("Help", -1, true, true);

            IGUIContextMenu submenu;
            submenu = menu.getSubMenu(0);
            submenu.addItem("Open Model File & Texture...", (int)GUI_ID.GUI_ID_OPEN_MODEL);
            submenu.addItem("Set Model Archive...", (int)GUI_ID.GUI_ID_SET_MODEL_ARCHIVE);
            submenu.addItem("Load as Octree", (int)GUI_ID.GUI_ID_LOAD_AS_OCTREE);
            submenu.addSeparator();
            submenu.addItem("Quit", (int)GUI_ID.GUI_ID_QUIT);

            submenu = menu.getSubMenu(1);
            submenu.addItem("sky box visible", (int)GUI_ID.GUI_ID_SKY_BOX_VISIBLE, true, false, true);
            submenu.addItem("toggle model debug information", (int)GUI_ID.GUI_ID_TOGGLE_DEBUG_INFO, true, true);
            submenu.addItem("model material", -1, true, true);

            submenu = submenu.getSubMenu(1);
            submenu.addItem("Off", (int)GUI_ID.GUI_ID_DEBUG_OFF);
            submenu.addItem("Bounding Box", (int)GUI_ID.GUI_ID_DEBUG_BOUNDING_BOX);
            submenu.addItem("Normals", (int)GUI_ID.GUI_ID_DEBUG_NORMALS);
            submenu.addItem("Skeleton", (int)GUI_ID.GUI_ID_DEBUG_SKELETON);
            submenu.addItem("Wire overlay", (int)GUI_ID.GUI_ID_DEBUG_WIRE_OVERLAY);
            submenu.addItem("Half-Transparent", (int)GUI_ID.GUI_ID_DEBUG_HALF_TRANSPARENT);
            submenu.addItem("Buffers bounding boxes", (int)GUI_ID.GUI_ID_DEBUG_BUFFERS_BOUNDING_BOXES);
            submenu.addItem("All", (int)GUI_ID.GUI_ID_DEBUG_ALL);

            submenu = menu.getSubMenu(1).getSubMenu(2);
            submenu.addItem("Solid", (int)GUI_ID.GUI_ID_MODEL_MATERIAL_SOLID);
            submenu.addItem("Transparent", (int)GUI_ID.GUI_ID_MODEL_MATERIAL_TRANSPARENT);
            submenu.addItem("Reflection", (int)GUI_ID.GUI_ID_MODEL_MATERIAL_REFLECTION);

            submenu = menu.getSubMenu(2);
            submenu.addItem("Maya Style", (int)GUI_ID.GUI_ID_CAMERA_MAYA);
            submenu.addItem("First Person", (int)GUI_ID.GUI_ID_CAMERA_FIRST_PERSON);

            submenu = menu.getSubMenu(3);
            submenu.addItem("About", (int)GUI_ID.GUI_ID_ABOUT);

            /*
            Below the menu we want a toolbar, onto which we can place colored
            buttons and important looking stuff like a senseless combobox.
            */

            // create toolbar

            IGUIToolBar bar = env.addToolBar();

            ITexture image = driver.getTexture("open.png");
            bar.addButton((int)GUI_ID.GUI_ID_BUTTON_OPEN_MODEL, null, "Open a model", image, null, false, true);

            image = driver.getTexture("tools.png");
            bar.addButton((int)GUI_ID.GUI_ID_BUTTON_SHOW_TOOLBOX, null, "Open Toolset", image, null, false, true);

            image = driver.getTexture("zip.png");
            bar.addButton((int)GUI_ID.GUI_ID_BUTTON_SELECT_ARCHIVE, null, "Set Model Archive", image, null, false, true);

            image = driver.getTexture("help.png");
            bar.addButton((int)GUI_ID.GUI_ID_BUTTON_SHOW_ABOUT, null, "Open Help", image, null, false, true);

            // create a combobox with some senseless texts

            IGUIComboBox box = env.addComboBox(new recti(250, 4, 350, 23), bar, 108);
            box.addItem("No filtering");
            box.addItem("Bilinear");
            box.addItem("Trilinear");
            box.addItem("Anisotropic");
            box.addItem("Isotropic");

            /*
            To make the editor look a little bit better, we disable transparent gui
            elements, and add an Irrlicht Engine logo. In addition, a text showing
            the current frames per second value is created and the window caption is
            changed.
            */

            // disable alpha

            for (int i = 0; i < (int) EGUI_DEFAULT_COLOR.EGDC_COUNT; ++i)
            {
                SColor col = env.getSkin().getColor((EGUI_DEFAULT_COLOR) i);
                col.setAlpha(255);
                env.getSkin().setColor((EGUI_DEFAULT_COLOR) i, col);
            }

            // add a tabcontrol

            createToolBox();

            // create fps text

            IGUIStaticText fpstext = env.addStaticText("",
                                                       new recti(400, 4, 570, 23), true, false, bar);

            IGUIStaticText postext = env.addStaticText("",
                                                       new recti(10, 50, 470, 80), false, false, null, (int)GUI_ID.GUI_ID_POSITION_TEXT);
            postext.setVisible(false);

            // set window caption

            Caption += " - [";
            Caption += driver.getName();
            Caption += "]";
            Device.setWindowCaption(Caption);

            /*
            That's nearly the whole application. We simply show the about message
            box at start up, and load the first model. To make everything look
            better, a skybox is created and a user controled camera, to make the
            application a little bit more interactive. Finally, everything is drawn
            in a standard drawing loop.
            */

            // show about message box and load default model
            if (args.Length == 1)
                showAboutText();
            loadModel(StartUpModelFile);

            // add skybox

            SkyBox = smgr.addSkyBoxSceneNode(
                driver.getTexture("irrlicht2_up.jpg"),
                driver.getTexture("irrlicht2_dn.jpg"),
                driver.getTexture("irrlicht2_lf.jpg"),
                driver.getTexture("irrlicht2_rt.jpg"),
                driver.getTexture("irrlicht2_ft.jpg"),
                driver.getTexture("irrlicht2_bk.jpg"));

            // add a camera scene node
            Camera[0] = smgr.addCameraSceneNodeMaya();
            Camera[0].setFarValue(20000.0f);
            // Maya cameras reposition themselves relative to their target, so target the location
            // where the mesh scene node is placed.
            Camera[0].setTarget(new vector3df(0, 30, 0));

            Camera[1] = smgr.addCameraSceneNodeFPS();
            Camera[1].setFarValue(20000.0f);
            Camera[1].setPosition(new vector3df(0, 0, -70));
            Camera[1].setTarget(new vector3df(0, 30, 0));

            setActiveCamera(Camera[0]);

            // load the irrlicht engine logo
            IGUIImage img =
                env.addImage(driver.getTexture("irrlichtlogo2.png"),
                             new position2di(10, (int)driver.getScreenSize().Height - 128));

            // lock the logo's edges to the bottom left corner of the screen
            img.setAlignment(EGUI_ALIGNMENT.EGUIA_UPPERLEFT, EGUI_ALIGNMENT.EGUIA_UPPERLEFT,
                             EGUI_ALIGNMENT.EGUIA_LOWERRIGHT, EGUI_ALIGNMENT.EGUIA_LOWERRIGHT);

            // draw everything

            while (Device.run() && driver != null)
            {
                if (Device.isWindowActive())
                {
                    driver.beginScene(true, true, new SColor(150, 50, 50, 50));

                    smgr.drawAll();
                    env.drawAll();

                    driver.endScene();

                    StringBuilder str = new StringBuilder("FPS: ");
                    str.Append(driver.getFPS());
                    str.Append(" Tris: ");
                    str.Append(driver.getPrimitiveCountDrawn());
                    fpstext.setText(str.ToString());

                    ICameraSceneNode cam = Device.getSceneManager().getActiveCamera();
                    str = new StringBuilder("Pos: ");
                    str.Append(cam.getPosition().X);
                    str.Append(" ");
                    str.Append(cam.getPosition().Y);
                    str.Append(" ");
                    str.Append(cam.getPosition().Z);
                    str.Append(" Tgt: ");
                    str.Append(cam.getTarget().X);
                    str.Append(" ");
                    str.Append(cam.getTarget().Y);
                    str.Append(" ");
                    str.Append(cam.getTarget().Z);
                    postext.setText(str.ToString());
                }
                else
                    Device.yield();
            }

            //Device.drop();
            return;
        }



        /*
        Toggle between various cameras
        */
        static void setActiveCamera(ICameraSceneNode newActive)
        {
            if (null == Device)
                return;

            ICameraSceneNode active = Device.getSceneManager().getActiveCamera();
            active.setInputReceiverEnabled(false);

            newActive.setInputReceiverEnabled(true);
            Device.getSceneManager().setActiveCamera(newActive);
        }
        /*
	        Set the skin transparency by changing the alpha values of all skin-colors
        */
        static void SetSkinTransparency(int alpha, IGUISkin skin)
        {
            for (int i = 0; i < (int)EGUI_DEFAULT_COLOR.EGDC_COUNT; ++i)
            {
                SColor col = skin.getColor((EGUI_DEFAULT_COLOR)i);
                col.setAlpha((uint)alpha);
                skin.setColor((EGUI_DEFAULT_COLOR)i, col);
            }
        }

        /*
          Update the display of the model scaling
        */
        static void UpdateScaleInfo(ISceneNode model)
        {
            IGUIElement toolboxWnd = Device.getGUIEnvironment().getRootGUIElement().getElementFromId((int)GUI_ID.GUI_ID_DIALOG_ROOT_WINDOW, true);
            if (toolboxWnd == null)
                return;
            if (model == null)
            {
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_X_SCALE, true).setText("-");
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_Y_SCALE, true).setText("-");
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_Z_SCALE, true).setText("-");
            }
            else
            {
                vector3df scale = model.getScale();
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_X_SCALE, true).setText(string.Format("{0}", scale.X));
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_Y_SCALE, true).setText(string.Format("{0}", scale.Y));
                toolboxWnd.getElementFromId((int)GUI_ID.GUI_ID_Z_SCALE, true).setText(string.Format("{0}", scale.Z));
            }
        }
        /*
        The three following functions do several stuff used by the mesh viewer. The
        first function showAboutText() simply displays a messagebox with a caption and
        a message text. The texts will be stored in the MessageText and Caption
        variables at startup.
        */
        static void showAboutText()
        {
            // create modal message box with the text
            // loaded from the xml file.
            Device.getGUIEnvironment().addMessageBox(
                Caption, MessageText);
        }

        /*
        The second function loadModel() loads a model and displays it using an
        addAnimatedMeshSceneNode and the scene manager. Nothing difficult. It also
        displays a short message box, if the model could not be loaded.
        */
        static void loadModel(string fn)
        {
            // modify the name if it a .pk3 file

            string filename = fn;

            string extension = System.IO.Path.GetExtension(filename);
            extension = extension.ToLower();

            // if a texture is loaded apply it to the current model..
            if (extension == ".jpg" || extension == ".pcx" ||
                extension == ".png" || extension == ".ppm" ||
                extension == ".pgm" || extension == ".pbm" ||
                extension == ".psd" || extension == ".tga" ||
                extension == ".bmp" || extension == ".wal" ||
		        extension == ".rgb" || extension == ".rgba")
            {
                ITexture texture =
                    Device.getVideoDriver().getTexture(filename);
                if (texture != null && Model != null)
                {
                    // always reload texture
                    Device.getVideoDriver().removeTexture(texture);
                    texture = Device.getVideoDriver().getTexture(filename);

                    Model.setMaterialTexture(0, texture);
                }
                return;
            }
            // if a archive is loaded add it to the FileArchive..
            else if (extension == ".pk3" || extension == ".zip")
            {
                Device.getFileSystem().addFileArchive(filename);
                return;
            }

            // load a model into the engine

            if (Model != null)
                Model.remove();

            Model = null;

	        if (extension==".irr")
	        {
                arraySceneNode outNodes = new arraySceneNode();
		        Device.getSceneManager().loadScene(filename);
		        Device.getSceneManager().getSceneNodesFromType((int)ESCENE_NODE_TYPE.ESNT_ANIMATED_MESH, outNodes);
		        if (outNodes.size() != 0)
			        Model = outNodes[0];
		        return;
	        }
            IAnimatedMesh m = Device.getSceneManager().getMesh(filename);

            if (m == null)
            {
                // model could not be loaded

                if (StartUpModelFile != filename)
                    Device.getGUIEnvironment().addMessageBox(
                    Caption, "The model could not be loaded. Maybe it is not a supported file format.");
                return;
            }

            // set default material properties

            if (Octree)
                Model = Device.getSceneManager().addOctTreeSceneNode(m.getMesh(0));
            else
            {
                IAnimatedMeshSceneNode animModel = Device.getSceneManager().addAnimatedMeshSceneNode(m);
                animModel.setAnimationSpeed(30);
                Model = animModel;
            }
            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, UseLight);
            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_NORMALIZE_NORMALS, UseLight);
            //	Model.setMaterialFlag(video::EMF_BACK_FACE_CULLING, false);
            Model.setDebugDataVisible((int)E_DEBUG_SCENE_TYPE.EDS_OFF);

            // we need to uncheck the menu entries. would be cool to fake a menu event, but
            // that's not so simple. so we do it brute force
            IGUIElement tmpGUI = Device.getGUIEnvironment().getRootGUIElement().getElementFromId((int)GUI_ID.GUI_ID_TOGGLE_DEBUG_INFO, true);
            IGUIContextMenu menu = IGUIContextMenu.cast(tmpGUI);
            if (menu != null)
            {
                for (uint item = 1; item < 6; item++)
                {
                    menu.setItemChecked(item, false);
                }
            }
            
	        UpdateScaleInfo(Model);
        }


        /*
        Finally, the third function creates a toolbox window. In this simple mesh
        viewer, this toolbox only contains a tab control with three edit boxes for
        changing the scale of the displayed model.
        */
        static void createToolBox()
        {
            // remove tool box if already there
            IGUIEnvironment env = Device.getGUIEnvironment();
            IGUIElement root = env.getRootGUIElement();
            IGUIElement e = root.getElementFromId((int)GUI_ID.GUI_ID_DIALOG_ROOT_WINDOW, true);
            if (e!=null)
                e.remove();

            // create the toolbox window
            IGUIWindow wnd = env.addWindow(new recti(600, 45, 800, 480),
                false, "Toolset", null, (int)GUI_ID.GUI_ID_DIALOG_ROOT_WINDOW);

            // create tab control and tabs
            IGUITabControl tab = env.addTabControl(
                new recti(2, 20, 800 - 602, 480 - 7), wnd, true, true);

            IGUITab t1 = tab.addTab("Config");

            // add some edit boxes and a button to tab one
            env.addStaticText("Scale:",
                    new recti(10, 20, 60, 45), false, false, t1);
            env.addStaticText("X:", new recti(22, 48, 40, 66), false, false, t1);
            env.addEditBox("1.0", new recti(40, 46, 130, 66), true, t1, (int)GUI_ID.GUI_ID_X_SCALE);
            env.addStaticText("Y:", new recti(22, 82, 40, 100), false, false, t1);
            env.addEditBox("1.0", new recti(40, 76, 130, 96), true, t1, (int)GUI_ID.GUI_ID_Y_SCALE);
            env.addStaticText("Z:", new recti(22, 108, 40, 126), false, false, t1);
            env.addEditBox("1.0", new recti(40, 106, 130, 126), true, t1, (int)GUI_ID.GUI_ID_Z_SCALE);

            env.addButton(new recti(10, 134, 85, 165), t1, (int)GUI_ID.GUI_ID_BUTTON_SET_SCALE, "Set");

	        // quick scale buttons
            env.addButton(new recti(65, 20, 95, 40), t1, (int)GUI_ID.GUI_ID_BUTTON_SCALE_MUL10, "* 10");
            env.addButton(new recti(100, 20, 130, 40), t1, (int)GUI_ID.GUI_ID_BUTTON_SCALE_DIV10, "* 0.1");

	        UpdateScaleInfo(Model);

            // add transparency control
            env.addStaticText("GUI Transparency Control:",
                    new recti(10, 200, 150, 225), true, false, t1);
            IGUIScrollBar scrollbar = env.addScrollBar(true,
                    new recti(10, 225, 150, 240), t1, (int)GUI_ID.GUI_ID_SKIN_TRANSPARENCY);
            scrollbar.setMax(255);
            scrollbar.setPos(255);

            // add framerate control
            env.addStaticText("Framerate:",
                    new recti(10, 240, 150, 265), true, false, t1);
            scrollbar = env.addScrollBar(true,
                    new recti(10, 265, 150, 280), t1, (int)GUI_ID.GUI_ID_SKIN_ANIMATION_FPS);
            scrollbar.setMax(MAX_FRAMERATE);
           	scrollbar.setMin(-MAX_FRAMERATE);
            scrollbar.setPos(DEFAULT_FRAMERATE);

            // bring irrlicht engine logo to front, because it
            // now may be below the newly created toolbox
            root.bringToFront(root.getElementFromId(666, true));
        }

        /*
        To get all the events sent by the GUI Elements, we need to create an event
        receiver. This one is really simple. If an event occurs, it checks the id of
        the caller and the event type, and starts an action based on these values. For
        example, if a menu item with id GUI_ID_OPEN_MODEL was selected, if opens a file-open-dialog.
        */

        private class SEventReceiver : IEventReceiver
        {

            public override bool OnEvent(SEvent ev)
            {
                // Escape swaps Camera Input
                if (ev.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT &&
                    ev.getKeyInput().PressedDown == false)
                {
                    return OnKeyUp(ev);
                }

                if (ev.EventType == EEVENT_TYPE.EET_GUI_EVENT)
                {
                    int id = ev.getGUIEvent().Caller.getID();
                    IGUIEnvironment env = Device.getGUIEnvironment();

                    switch (ev.getGUIEvent().EventType)
                    {
                        case EGUI_EVENT_TYPE.EGET_MENU_ITEM_SELECTED:
                            {
                                // a menu item was clicked
                                OnMenuItemSelected(ev, env);
                                break;
                            }

                        case EGUI_EVENT_TYPE.EGET_FILE_SELECTED:
                            {
                                // load the model file, selected in the file open dialog
                                IGUIFileOpenDialog dialog =
                                    IGUIFileOpenDialog.cast( ev.getGUIEvent().Caller);
                                loadModel(dialog.getFileName());
                            }
                            break;

                        case EGUI_EVENT_TYPE.EGET_SCROLL_BAR_CHANGED:

                            // control skin transparency
                            if (id == (int)GUI_ID.GUI_ID_SKIN_TRANSPARENCY)
                            {
                                int pos = IGUIScrollBar.cast( ev.getGUIEvent().Caller).getPos();
                                SetSkinTransparency(pos, env.getSkin());
                                //for (int i = 0; i < (int) EGUI_DEFAULT_COLOR.EGDC_COUNT; ++i)
                                //{
                                //    SColor col = env.getSkin().getColor((EGUI_DEFAULT_COLOR) i);
                                //    col.setAlpha((uint) pos);
                                //    env.getSkin().setColor((EGUI_DEFAULT_COLOR) i, col);
                                //}
                            }
                            else if (id == (int)GUI_ID.GUI_ID_SKIN_ANIMATION_FPS)
                            {
                                int pos = IGUIScrollBar.cast( ev.getGUIEvent().Caller).getPos();
                                if ((int) ESCENE_NODE_TYPE.ESNT_ANIMATED_MESH == Model.getType())
                                    ((IAnimatedMeshSceneNode) Model).setAnimationSpeed((float) pos);
                            }
                            break;

                        case EGUI_EVENT_TYPE.EGET_COMBO_BOX_CHANGED:

                            // control anti-aliasing/filtering
                            if (id == (int)GUI_ID.GUI_ID_TEXTUREFILTER)
                            {
                                OnTextureFilterSelected(ev);
                            }
                            break;

                        case EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED:

                            switch (id)
                            {
                                case (int)GUI_ID.GUI_ID_BUTTON_SET_SCALE:
                                    {
                                        // set scale
                                        IGUIElement root = env.getRootGUIElement();
                                        vector3df scale = new vector3df();
                                        string s;

                                        s = root.getElementFromId((int)GUI_ID.GUI_ID_X_SCALE, true).getText();
                                        scale.X = float.Parse(s);
                                        s = root.getElementFromId((int)GUI_ID.GUI_ID_Y_SCALE, true).getText();
                                        scale.Y = float.Parse(s);
                                        s = root.getElementFromId((int)GUI_ID.GUI_ID_Z_SCALE, true).getText();
                                        scale.Z = float.Parse(s);

                                        if (Model != null)
                                            Model.setScale(scale);
                                        UpdateScaleInfo(Model);
                                    }
                                    break;
                                case (int)GUI_ID.GUI_ID_BUTTON_SCALE_MUL10:
                                    env.addFileOpenDialog("Please select a model file to open");
                                    UpdateScaleInfo(Model);
                                    break;
                                case (int)GUI_ID.GUI_ID_BUTTON_OPEN_MODEL:
					                env.addFileOpenDialog("Please select a model file to open");
					                break;
                                case (int)GUI_ID.GUI_ID_BUTTON_SHOW_ABOUT:
                                    showAboutText();
                                    break;
                                case (int)GUI_ID.GUI_ID_BUTTON_SHOW_TOOLBOX:
                                    createToolBox();
                                    break;
                                case (int)GUI_ID.GUI_ID_BUTTON_SELECT_ARCHIVE:
                                    env.addFileOpenDialog("Please select your game archive/directory");
                                    break;
                            }

                            break;
                        default:
                            break;
                    }
                }

                return false;
            }

private static void OnTextureFilterSelected(SEvent ev)
{
                                int pos = IGUIComboBox.cast( ev.getGUIEvent().Caller).getSelected();
                                switch (pos)
                                {
                                    case 0:
                                        if (Model != null)
                                        {
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_BILINEAR_FILTER, false);
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_TRILINEAR_FILTER, false);
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_ANISOTROPIC_FILTER, false);
                                        }
                                        break;
                                    case 1:
                                        if (Model != null)
                                        {
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_BILINEAR_FILTER, true);
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_TRILINEAR_FILTER, false);
                                        }
                                        break;
                                    case 2:
                                        if (Model != null)
                                        {
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_BILINEAR_FILTER, false);
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_TRILINEAR_FILTER, true);
                                        }
                                        break;
                                    case 3:
                                        if (Model != null)
                                        {
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_ANISOTROPIC_FILTER, true);
                                        }
                                        break;
                                    case 4:
                                        if (Model != null)
                                        {
                                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_ANISOTROPIC_FILTER, false);
                                        }
                                        break;
                                }
}

private static void OnMenuItemSelected(SEvent ev, IGUIEnvironment env)
{

                                IGUIContextMenu menu = IGUIContextMenu.cast( ev.getGUIEvent().Caller);
                                int id2 = menu.getItemCommandId((uint) menu.getSelectedItem());

                                switch (id2)
                                {
                                    case (int)GUI_ID.GUI_ID_OPEN_MODEL: // File . Open Model
                                        env.addFileOpenDialog("Please select a model file to open");
                                        break;
                                    case (int)GUI_ID.GUI_ID_SET_MODEL_ARCHIVE: // File . Set Model Archive
                                        env.addFileOpenDialog("Please select your game archive/directory");
                                        break;
                                    case (int)GUI_ID.GUI_ID_LOAD_AS_OCTREE: // File . LoadAsOctree
                                        Octree = !Octree;
                                        menu.setItemChecked((uint) menu.getSelectedItem(), Octree);
                                        break;
                                    case (int)GUI_ID.GUI_ID_QUIT: // File . Quit
                                        Device.closeDevice();
                                        break;
                                    case (int)GUI_ID.GUI_ID_SKY_BOX_VISIBLE: // View . Skybox
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint)menu.getSelectedItem()));
                                        SkyBox.setVisible(!SkyBox.isVisible());
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_OFF: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 1, false);
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 2, false);
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 3, false);
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 4, false);
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 5, false);
                                        menu.setItemChecked((uint) menu.getSelectedItem() + 6, false);
                                        if (Model != null)
                                            Model.setDebugDataVisible((int) E_DEBUG_SCENE_TYPE.EDS_OFF);
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_BOUNDING_BOX: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int) (Model.isDebugDataVisible() ^ (int) E_DEBUG_SCENE_TYPE.EDS_BBOX));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_NORMALS: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int)
                                                (Model.isDebugDataVisible() ^ (int) E_DEBUG_SCENE_TYPE.EDS_NORMALS));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_SKELETON: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int)
                                                (Model.isDebugDataVisible() ^ (int) E_DEBUG_SCENE_TYPE.EDS_SKELETON));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_WIRE_OVERLAY: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int)
                                                (Model.isDebugDataVisible() ^
                                                 (int) E_DEBUG_SCENE_TYPE.EDS_MESH_WIRE_OVERLAY));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_HALF_TRANSPARENT: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int)
                                                (Model.isDebugDataVisible() ^
                                                 (int) E_DEBUG_SCENE_TYPE.EDS_HALF_TRANSPARENCY));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_BUFFERS_BOUNDING_BOXES: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem(),
                                                            !menu.isItemChecked((uint) menu.getSelectedItem()));
                                        if (Model != null)
                                            Model.setDebugDataVisible(
                                                (int)
                                                (Model.isDebugDataVisible() ^ (int) E_DEBUG_SCENE_TYPE.EDS_BBOX_BUFFERS));
                                        break;
                                    case (int)GUI_ID.GUI_ID_DEBUG_ALL: // View . Debug Information
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 1, true);
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 2, true);
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 3, true);
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 4, true);
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 5, true);
                                        menu.setItemChecked((uint) menu.getSelectedItem() - 6, true);
                                        if (Model != null)
                                            Model.setDebugDataVisible((int) E_DEBUG_SCENE_TYPE.EDS_FULL);
                                        break;
                                    case (int)GUI_ID.GUI_ID_ABOUT: // Help.About
                                        showAboutText();
                                        break;
                                    case (int)GUI_ID.GUI_ID_MODEL_MATERIAL_SOLID: // View . Material . Solid
                                        if (Model != null)
                                            Model.setMaterialType(E_MATERIAL_TYPE.EMT_SOLID);
                                        break;
                                    case (int)GUI_ID.GUI_ID_MODEL_MATERIAL_TRANSPARENT: // View . Material . Transparent
                                        if (Model != null)
                                            Model.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);
                                        break;
                                    case (int)GUI_ID.GUI_ID_MODEL_MATERIAL_REFLECTION: // View . Material . Reflection
                                        if (Model != null)
                                            Model.setMaterialType(E_MATERIAL_TYPE.EMT_SPHERE_MAP);
                                        break;

                                    case (int)GUI_ID.GUI_ID_CAMERA_MAYA:
                                        setActiveCamera(Camera[0]);
                                        break;
                                    case (int)GUI_ID.GUI_ID_CAMERA_FIRST_PERSON:
                                        setActiveCamera(Camera[1]);
                                        break;

                                }
}

private static bool OnKeyUp(SEvent ev)
{
                    if (ev.getKeyInput().Key == EKEY_CODE.KEY_ESCAPE)
                    {
                        if (Device != null)
                        {
                            ICameraSceneNode camera =
                                Device.getSceneManager().getActiveCamera();
                            if (camera != null)
                            {
                                camera.setInputReceiverEnabled(!camera.isInputReceiverEnabled());
                            }
                            return true;
                        }
                    }
                    else if (ev.getKeyInput().Key == EKEY_CODE.KEY_F1)
                    {
                        if (Device != null)
                        {
                            IGUIElement elem = Device.getGUIEnvironment().getRootGUIElement().getElementFromId((int)GUI_ID.GUI_ID_POSITION_TEXT);
                            if (elem != null)
                                elem.setVisible(!elem.isVisible());
                        }
                    }
                    else if (ev.KeyInput.Key == EKEY_CODE.KEY_KEY_M)
                    {
                        if (Device != null)
                        {
                            Device.minimizeWindow();
                        }
                    }
                    else if (ev.getKeyInput().Key == EKEY_CODE.KEY_KEY_L)
                    {
                        UseLight = !UseLight;
                        if (Model != null)
                        {
                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, UseLight);
                            Model.setMaterialFlag(E_MATERIAL_FLAG.EMF_NORMALIZE_NORMALS, UseLight);
                        }
                    }
                    return false;
}
        }
    }
}
