﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace MaterialViewer
{
    class Program
    {
        static void Main(string[] args)
        {

            /*
              A very short main as we do everything else in classes.
            */
            CApp APP = new CApp();

            if (!APP.init(args))
            {
                Console.WriteLine("init failed\n");
                return;
            }

            APP.run();
            APP.quit();

            return;
        }
    }

    /** Example 022 Material Viewer

This example can be used to play around with material settings and watch the results.
Only the default non-shader materials are used in here.

You have two nodes to make it easier to see which difference your settings will make.
Additionally you have one lightscenenode and you can set the global ambient values.
*/

    /*
        Variables within the empty namespace are globals which are restricted to this file.
    */
    public static class Statics
    {
        public static string[] DriverTypeNames =
    {
	    "NULL",
	    "SOFTWARE",
	    "BURNINGSVIDEO",
	    "DIRECT3D8",
	    "DIRECT3D9",
	    "OPENGL",
	    "",
    };

        // For the gui id's
        public enum EGUI_IDS
        {
            GUI_ID_OPEN_TEXTURE = 1,
            GUI_ID_QUIT,
            GUI_ID_MAX
        }

        // Name used in texture selection to clear the textures on the node
        public static string CLEAR_TEXTURE = "CLEAR texture";

        // some useful color constants
        public static SColor SCOL_BLACK = new SColor(255, 0, 0, 0);
        public static SColor SCOL_BLUE = new SColor(255, 0, 0, 255);
        public static SColor SCOL_CYAN = new SColor(255, 0, 255, 255);
        public static SColor SCOL_GRAY = new SColor(255, 128, 128, 128);
        public static SColor SCOL_GREEN = new SColor(255, 0, 255, 0);
        public static SColor SCOL_MAGENTA = new SColor(255, 255, 0, 255);
        public static SColor SCOL_RED = new SColor(255, 255, 0, 0);
        public static SColor SCOL_YELLOW = new SColor(255, 255, 255, 0);
        public static SColor SCOL_WHITE = new SColor(255, 255, 255, 255);
        /*
            Returns a new unique number on each call.
        */
        static int unique = (int)EGUI_IDS.GUI_ID_MAX;
        public static int makeUniqueId()
        {
            ++unique;
            return unique;
        }

        /*
            Find out which vertex-type is needed for the given material type.
        */
        public static E_VERTEX_TYPE getVertexTypeForMaterialType(E_MATERIAL_TYPE materialType)
        {
            switch (materialType)
            {
                case E_MATERIAL_TYPE.EMT_SOLID:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_SOLID_2_LAYER:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_LIGHTMAP:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_ADD:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_M2:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_M4:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_LIGHTING:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_LIGHTING_M2:
                case E_MATERIAL_TYPE.EMT_LIGHTMAP_LIGHTING_M4:
                    return E_VERTEX_TYPE.EVT_2TCOORDS;

                case E_MATERIAL_TYPE.EMT_DETAIL_MAP:
                    return E_VERTEX_TYPE.EVT_2TCOORDS;

                case E_MATERIAL_TYPE.EMT_SPHERE_MAP:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_REFLECTION_2_LAYER:
                    return E_VERTEX_TYPE.EVT_2TCOORDS;

                case E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_TRANSPARENT_ALPHA_CHANNEL:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_TRANSPARENT_ALPHA_CHANNEL_REF:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_TRANSPARENT_VERTEX_ALPHA:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_TRANSPARENT_REFLECTION_2_LAYER:
                    return E_VERTEX_TYPE.EVT_2TCOORDS;

                case E_MATERIAL_TYPE.EMT_NORMAL_MAP_SOLID:
                case E_MATERIAL_TYPE.EMT_NORMAL_MAP_TRANSPARENT_ADD_COLOR:
                case E_MATERIAL_TYPE.EMT_NORMAL_MAP_TRANSPARENT_VERTEX_ALPHA:
                case E_MATERIAL_TYPE.EMT_PARALLAX_MAP_SOLID:
                case E_MATERIAL_TYPE.EMT_PARALLAX_MAP_TRANSPARENT_ADD_COLOR:
                case E_MATERIAL_TYPE.EMT_PARALLAX_MAP_TRANSPARENT_VERTEX_ALPHA:
                    return E_VERTEX_TYPE.EVT_TANGENTS;

                case E_MATERIAL_TYPE.EMT_ONETEXTURE_BLEND:
                    return E_VERTEX_TYPE.EVT_STANDARD;

                case E_MATERIAL_TYPE.EMT_FORCE_32BIT:
                    return E_VERTEX_TYPE.EVT_STANDARD;
            }
            return E_VERTEX_TYPE.EVT_STANDARD;
        }

    }

    /*
        Custom GUI-control to edit colorvalues.
    */
    class CColorControl : IGUIElement_ForInheritance
    {
        // Constructor
        public CColorControl(IGUIEnvironment guiEnv, position2di pos, string text, IGUIElement parent, int id)
            : base(EGUI_ELEMENT_TYPE.EGUIET_ELEMENT, guiEnv, parent, id, new recti(pos, pos + new dimension2di(80, 75)))
        {
            DirtyFlag = true;
            ColorStatic = null;
            EditAlpha = null;
            EditRed = null;
            EditGreen = null;
            EditBlue = null;


            ButtonSetId = Statics.makeUniqueId();

            recti rectControls = new recti(0, 0, AbsolutePosition.getWidth(), AbsolutePosition.getHeight());
            IGUIStaticText groupElement = guiEnv.addStaticText("", rectControls, true, false, this, -1, false);
            groupElement.setNotClipped(true);

            guiEnv.addStaticText(text, new recti(0, 0, 80, 15), false, false, groupElement, -1, false);

            EditAlpha = addEditForNumbers(guiEnv, new position2di(0, 15), "a", -1, groupElement);
            EditRed = addEditForNumbers(guiEnv, new position2di(0, 30), "r", -1, groupElement);
            EditGreen = addEditForNumbers(guiEnv, new position2di(0, 45), "g", -1, groupElement);
            EditBlue = addEditForNumbers(guiEnv, new position2di(0, 60), "b", -1, groupElement);

            ColorStatic = guiEnv.addStaticText("", new recti(60, 15, 80, 75), true, false, groupElement, -1, true);

            guiEnv.addButton(new recti(60, 35, 80, 50), groupElement, ButtonSetId, "set");
            SetEditsFromColor(Color);
        }

        // event receiver
        public override bool OnEvent2(SEvent ev)
        {
            if (ev.EventType != EEVENT_TYPE.EET_GUI_EVENT)
                return false;

            if (ev.GUIEvent.Caller.getID() == ButtonSetId && ev.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
            {
                Color = GetColorFromEdits();
                SetEditsFromColor(Color);
            }

            return false;
        }

        // set the color values
        public void setColor(SColor col)
        {
            DirtyFlag = true;
            Color.CopyFrom(col);
            SetEditsFromColor(Color);
        }

        // get the color values
        public SColor getColor()
        {
            return Color;
        }

        // To reset the dirty flag
        public void resetDirty()
        {
            DirtyFlag = false;
        }

        // when the color was changed the dirty flag is set
        public bool isDirty()
        {
            return DirtyFlag;
        }


        // Add a staticbox for a description + an editbox so users can enter numbers
        protected IGUIEditBox addEditForNumbers(IGUIEnvironment guiEnv, position2di pos, string text, int id, IGUIElement parent)
        {
            recti rect = new recti(pos, pos + new dimension2di(10, 15));
            guiEnv.addStaticText(text, rect, false, false, parent, -1, false);
            rect = rect + new position2di(20, 0);
            rect.LowerRightCorner.X += 20;
            IGUIEditBox edit = guiEnv.addEditBox("0", rect, true, parent, id);
            return edit;
        }

        // Get the color value from the editfields
        protected SColor GetColorFromEdits()
        {
            SColor col = new SColor();

            uint alpha = col.getAlpha();
            if (EditAlpha != null)
            {
                uint.TryParse(EditAlpha.getText(), out alpha);
                if (alpha > 255)
                    alpha = 255;
            }
            col.setAlpha(alpha);

            uint red = col.getRed();
            if (EditRed != null)
            {
                uint.TryParse(EditRed.getText(), out red);
                if (red > 255)
                    red = 255;
            }
            col.setRed(red);

            uint green = col.getGreen();
            if (EditGreen != null)
            {
                uint.TryParse(EditGreen.getText(), out green);
                if (green > 255)
                    green = 255;
            }
            col.setGreen(green);

            uint blue = col.getBlue();
            if (EditBlue != null)
            {
                uint.TryParse(EditBlue.getText(), out blue);
                if (blue > 255)
                    blue = 255;
            }
            col.setBlue(blue);

            return col;
        }

        // Fill the editfields with the value for the given color
        protected void SetEditsFromColor(SColor col)
        {
            DirtyFlag = true;
            if (EditAlpha != null)
                EditAlpha.setText(string.Format("{0}", col.getAlpha()));
            if (EditRed != null)
                EditRed.setText(string.Format("{0}", col.getRed()));
            if (EditGreen != null)
                EditGreen.setText(string.Format("{0}", col.getGreen()));
            if (EditBlue != null)
                EditBlue.setText(string.Format("{0}", col.getBlue()));
            if (ColorStatic != null)
                ColorStatic.setBackgroundColor(col);
        }

        bool DirtyFlag;
        SColor Color = new SColor();
        int ButtonSetId;
        IGUIStaticText ColorStatic;
        IGUIEditBox EditAlpha;
        IGUIEditBox EditRed;
        IGUIEditBox EditGreen;
        IGUIEditBox EditBlue;
    };

    /*
        Custom GUI-control for to edit all colors typically used in materials and lights
    */
    class CAllColorsControl : IGUIElement_ForInheritance
    {
        // Constructor
        public CAllColorsControl(IGUIEnvironment guiEnv, position2di pos, string description, bool hasEmissive, IGUIElement parent, int id)
            : base(EGUI_ELEMENT_TYPE.EGUIET_ELEMENT, guiEnv, parent, id, new recti(pos, pos + new dimension2di(60, 250)))
        {
            ControlAmbientColor = null;
            ControlDiffuseColor = null;
            ControlSpecularColor = null;
            ControlEmissiveColor = null;
            recti rect = new recti(0, 0, 60, 15);
            guiEnv.addStaticText(description, rect, false, false, this, -1, false);
            createColorControls(guiEnv, new position2di(0, 15), hasEmissive);
        }

        //	// Destructor
        //	public virtual ~CAllColorsControl()
        //	{
        //		ControlAmbientColor.drop();
        //		ControlDiffuseColor.drop();
        //		ControlEmissiveColor.drop();
        //		ControlSpecularColor.drop();
        //	}

        // Set the color values to those within the material
        public void setColorsToMaterialColors(SMaterial material)
        {
            ControlAmbientColor.setColor(material.AmbientColor);
            ControlDiffuseColor.setColor(material.DiffuseColor);
            ControlEmissiveColor.setColor(material.EmissiveColor);
            ControlSpecularColor.setColor(material.SpecularColor);
        }

        // Update all changed colors in the material
        public void updateMaterialColors(SMaterial material)
        {
            if (ControlAmbientColor.isDirty())
                material.AmbientColor = ControlAmbientColor.getColor();
            if (ControlDiffuseColor.isDirty())
                material.DiffuseColor = ControlDiffuseColor.getColor();
            if (ControlEmissiveColor.isDirty())
                material.EmissiveColor = ControlEmissiveColor.getColor();
            if (ControlSpecularColor.isDirty())
                material.SpecularColor = ControlSpecularColor.getColor();
        }

        // Set the color values to those from the light data
        public void setColorsToLightDataColors(SLight lightData)
        {
            ControlAmbientColor.setColor(lightData.AmbientColor.toSColor());
            ControlAmbientColor.setColor(lightData.DiffuseColor.toSColor());
            ControlAmbientColor.setColor(lightData.SpecularColor.toSColor());
        }

        // Update all changed colors in the light data
        public void updateMaterialColors(SLight lightData)
        {
            if (ControlAmbientColor.isDirty())
                lightData.AmbientColor = new SColorf(ControlAmbientColor.getColor());
            if (ControlDiffuseColor.isDirty())
                lightData.DiffuseColor = new SColorf(ControlDiffuseColor.getColor());
            if (ControlSpecularColor.isDirty())
                lightData.SpecularColor = new SColorf(ControlSpecularColor.getColor());
        }

        // To reset the dirty flags
        public void resetDirty()
        {
            ControlAmbientColor.resetDirty();
            ControlDiffuseColor.resetDirty();
            ControlSpecularColor.resetDirty();
            ControlEmissiveColor.resetDirty();
        }

        protected void createColorControls(IGUIEnvironment guiEnv, position2di pos, bool hasEmissive)
        {
            ControlAmbientColor = new CColorControl(guiEnv, pos, "ambient", this, -1);
            ControlDiffuseColor = new CColorControl(guiEnv, pos + new position2di(0, 75), "diffuse", this, -1);
            ControlSpecularColor = new CColorControl(guiEnv, pos + new position2di(0, 150), "specular", this, -1);
            if (hasEmissive)
            {
                ControlEmissiveColor = new CColorControl(guiEnv, pos + new position2di(0, 225), "emissive", this, -1);
            }
        }

        CColorControl ControlAmbientColor;
        CColorControl ControlDiffuseColor;
        CColorControl ControlSpecularColor;
        CColorControl ControlEmissiveColor;
    };

    /*
        GUI-Control to offer a selection of available textures.
    */
    class CTextureControl : IGUIElement_ForInheritance
    {
        public CTextureControl(IGUIEnvironment guiEnv, IVideoDriver driver, position2di pos, IGUIElement parent, int id)
            : base(EGUI_ELEMENT_TYPE.EGUIET_ELEMENT, guiEnv, parent, id, new recti(pos, pos + new dimension2di(100, 15)))
        {
            DirtyFlag = true;
            ComboTexture = null;

            recti rectCombo = new recti(0, 0, AbsolutePosition.getWidth(), AbsolutePosition.getHeight());
            ComboTexture = guiEnv.addComboBox(rectCombo, this);
            updateTextures(driver);
        }

        public override bool OnEvent2(SEvent ev)
        {
            if (ev.EventType != EEVENT_TYPE.EET_GUI_EVENT)
                return false;

            if (ev.GUIEvent.Caller == ComboTexture && ev.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_COMBO_BOX_CHANGED)
            {
                DirtyFlag = true;
            }

            return false;
        }

        // Workaround for a problem with comboboxes.
        // We have to get in front when the combobox wants to get in front or combobox-list might be drawn below other elements.
        public override bool bringToFront(IGUIElement element)
        {

            bool result = base.bringToFront(element);
            if (Parent != null && element == ComboTexture)
                result &= Parent.bringToFront(this);
            return result;
        }

        // return selected texturename (if any, otherwise 0)
        public string getSelectedTextureName()
        {
            int selected = ComboTexture.getSelected();
            if (selected < 0)
                return "";
            return ComboTexture.getItem((uint)selected);
        }

        // reset the dirty flag
        public void resetDirty()
        {
            DirtyFlag = false;
        }

        // when the texture was changed the dirty flag is set
        public bool isDirty()
        {
            return DirtyFlag;
        }

        // Put the names of all currenlty loaded textures in a combobox
        public void updateTextures(IVideoDriver driver)
        {
            int oldSelected = ComboTexture.getSelected();
            int selectNew = -1;
            string oldTextureName = "";
            if (oldSelected >= 0)
            {
                oldTextureName = ComboTexture.getItem((uint)oldSelected);
            }
            ComboTexture.clear();
            for (uint i = 0; i < driver.getTextureCount(); ++i)
            {
                ITexture texture = driver.getTextureByIndex(i);
                string name = texture.getName().getPath().c_str();
                ComboTexture.addItem(name);
                if (!string.IsNullOrEmpty(oldTextureName) && selectNew < 0 && name == oldTextureName)
                    selectNew = (int)i;
            }

            // add another name which can be used to clear the texture
            ComboTexture.addItem(Statics.CLEAR_TEXTURE);
            if (Statics.CLEAR_TEXTURE == oldTextureName)
                selectNew = (int)ComboTexture.getItemCount() - 1;

            if (selectNew >= 0)
                ComboTexture.setSelected(selectNew);

            DirtyFlag = true;
        }

        bool DirtyFlag;
        IGUIComboBox ComboTexture;
    };

    /*
        Control which allows setting some of the material values for a meshscenenode
    */
    class SMeshNodeControl
    {
        // constructor
        public SMeshNodeControl()
        {
            Initialized = false;
            Driver = null;
            MeshManipulator = null;
            SceneNode = null;
            SceneNode2T = null;
            SceneNodeTangents = null;
            AllColorsControl = null;
            ButtonLighting = null;
            InfoLighting = null;
            ComboMaterial = null;
            TextureControl1 = null;
            TextureControl2 = null;
            ControlVertexColors = null;
        }

        // Destructor
        //public virtual ~SMeshNodeControl()
        //{
        //    if ( TextureControl1 )
        //        TextureControl1.drop();
        //    if ( TextureControl2 )
        //        TextureControl2.drop();
        //    if ( ControlVertexColors )
        //        ControlVertexColors.drop();
        //}

        public void init(IMeshSceneNode node, IrrlichtDevice device, position2di pos, string description)
        {
            if (Initialized || node == null || device == null) // initializing twice or with invalid data not allowed
                return;

            Driver = device.getVideoDriver();
            IGUIEnvironment guiEnv = device.getGUIEnvironment();
            ISceneManager smgr = device.getSceneManager();
            MeshManipulator = smgr.getMeshManipulator();

            SceneNode = node;
            IMeshManipulator meshManip = smgr.getMeshManipulator();

            IMesh mesh2T = meshManip.createMeshWith2TCoords(node.getMesh());
            SceneNode2T = smgr.addMeshSceneNode(mesh2T, null, -1, SceneNode.getPosition(), SceneNode.getRotation(), SceneNode.getScale());
            //mesh2T.drop();

            IMesh meshTangents = meshManip.createMeshWithTangents(node.getMesh(), false, false, false);
            SceneNodeTangents = smgr.addMeshSceneNode(meshTangents, null, -1
                                                , SceneNode.getPosition(), SceneNode.getRotation(), SceneNode.getScale());
            //meshTangents.drop();

            SMaterial material = SceneNode.getMaterial(0);
            material.Lighting = true;
            AllColorsControl = new CAllColorsControl(guiEnv, pos, description, true, guiEnv.getRootGUIElement(), -1);
            AllColorsControl.setColorsToMaterialColors(material);

            recti rectBtn = new recti(pos + new position2di(0, 320), pos + new position2di(0, 320) + new position2di(60, 15));
            ButtonLighting = guiEnv.addButton(rectBtn, null, -1, "Lighting");
            ButtonLighting.setIsPushButton(true);
            ButtonLighting.setPressed(material.Lighting);
            recti rectInfo = new recti(rectBtn.LowerRightCorner.X, rectBtn.UpperLeftCorner.Y, rectBtn.LowerRightCorner.X + 40, rectBtn.UpperLeftCorner.Y + 15);
            InfoLighting = guiEnv.addStaticText("", rectInfo, true, false);
            InfoLighting.setTextAlignment(EGUI_ALIGNMENT.EGUIA_CENTER, EGUI_ALIGNMENT.EGUIA_CENTER);

            recti rectCombo = new recti(pos.X, rectBtn.LowerRightCorner.Y, pos.X + 100, rectBtn.LowerRightCorner.Y + 15);
            ComboMaterial = guiEnv.addComboBox(rectCombo);
            for (int i = 0; i <= (int)E_MATERIAL_TYPE.EMT_ONETEXTURE_BLEND; ++i)
            {
                ComboMaterial.addItem(IrrlichtNet.sBuiltInMaterialTypeNames[i]);
            }
            ComboMaterial.setSelected((int)material.MaterialType);

            position2di posTex = new position2di(rectCombo.UpperLeftCorner.X, rectCombo.LowerRightCorner.Y);
            TextureControl1 = new CTextureControl(guiEnv, Driver, posTex, guiEnv.getRootGUIElement(), -1);
            posTex.Y += 15;
            TextureControl2 = new CTextureControl(guiEnv, Driver, posTex, guiEnv.getRootGUIElement(), -1);

            position2di posVertexColors = new position2di(posTex.X, posTex.Y + 15);
            ControlVertexColors = new CColorControl(guiEnv, posVertexColors, "Vertex colors", guiEnv.getRootGUIElement(), -1);

            S3DVertex[] vertices = node.getMesh().getMeshBuffer(0).getVerticesAsS3DVertex();
            if (vertices != null)
            {
                ControlVertexColors.setColor(vertices[0].Color);
            }

            Initialized = true;
        }

        public void update()
        {
            if (!Initialized)
                return;

            SMaterial material = SceneNode.getMaterial(0);
            SMaterial material2T = SceneNode2T.getMaterial(0);
            SMaterial materialTangents = SceneNodeTangents.getMaterial(0);

            int selectedMaterial = ComboMaterial.getSelected();
            if (selectedMaterial >= (int)E_MATERIAL_TYPE.EMT_SOLID && selectedMaterial <= (int)E_MATERIAL_TYPE.EMT_ONETEXTURE_BLEND)
            {
                E_VERTEX_TYPE vertexType = Statics.getVertexTypeForMaterialType((E_MATERIAL_TYPE)selectedMaterial);
                switch (vertexType)
                {
                    case E_VERTEX_TYPE.EVT_STANDARD:
                        material.MaterialType = (E_MATERIAL_TYPE)selectedMaterial;
                        SceneNode.setVisible(true);
                        SceneNode2T.setVisible(false);
                        SceneNodeTangents.setVisible(false);
                        break;
                    case E_VERTEX_TYPE.EVT_2TCOORDS:
                        material2T.MaterialType = (E_MATERIAL_TYPE)selectedMaterial;
                        SceneNode.setVisible(false);
                        SceneNode2T.setVisible(true);
                        SceneNodeTangents.setVisible(false);
                        break;
                    case E_VERTEX_TYPE.EVT_TANGENTS:
                        materialTangents.MaterialType = (E_MATERIAL_TYPE)selectedMaterial;
                        SceneNode.setVisible(false);
                        SceneNode2T.setVisible(false);
                        SceneNodeTangents.setVisible(true);
                        break;
                }
            }

            updateMaterial(material);
            updateMaterial(material2T);
            updateMaterial(materialTangents);

            if (ButtonLighting.isPressed())
                InfoLighting.setText("on");
            else
                InfoLighting.setText("off");

            AllColorsControl.resetDirty();
            TextureControl1.resetDirty();
            TextureControl2.resetDirty();
            ControlVertexColors.resetDirty();
        }

        public void updateTextures()
        {
            TextureControl1.updateTextures(Driver);
            TextureControl2.updateTextures(Driver);
        }

        protected void updateMaterial(SMaterial material)
        {
            AllColorsControl.updateMaterialColors(material);
            material.Lighting = ButtonLighting.isPressed();
            if (TextureControl1.isDirty())
            {
                material.TextureLayer[0].Texture = Driver.getTexture(TextureControl1.getSelectedTextureName());
            }
            if (TextureControl2.isDirty())
            {
                material.TextureLayer[1].Texture = Driver.getTexture(TextureControl2.getSelectedTextureName());
            }
            if (ControlVertexColors.isDirty())
            {
                MeshManipulator.setVertexColors(SceneNode.getMesh(), ControlVertexColors.getColor());
                MeshManipulator.setVertexColors(SceneNode2T.getMesh(), ControlVertexColors.getColor());
                MeshManipulator.setVertexColors(SceneNodeTangents.getMesh(), ControlVertexColors.getColor());
            }
        }

        protected bool Initialized;
        protected IVideoDriver Driver;
        protected IMeshManipulator MeshManipulator;
        protected IMeshSceneNode SceneNode;
        protected IMeshSceneNode SceneNode2T;
        protected IMeshSceneNode SceneNodeTangents;
        protected CAllColorsControl AllColorsControl;
        protected IGUIButton ButtonLighting;
        protected IGUIStaticText InfoLighting;
        protected IGUIComboBox ComboMaterial;
        protected CTextureControl TextureControl1;
        protected CTextureControl TextureControl2;
        protected CColorControl ControlVertexColors;
    };

    /*
        Control to allow setting the color values of a lightscenenode.
    */
    class SLightNodeControl
    {
        // constructor
        public SLightNodeControl()
        {
            Initialized = false;
            SceneNode = null;
            AllColorsControl = null;
        }

        public void init(ILightSceneNode node, IGUIEnvironment guiEnv, position2di pos, string description)
        {
            if (Initialized || node == null || guiEnv == null) // initializing twice or with invalid data not allowed
                return;
            SceneNode = node;
            AllColorsControl = new CAllColorsControl(guiEnv, pos, description, false, guiEnv.getRootGUIElement(), -1);
            SLight lightData = SceneNode.getLightData();
            AllColorsControl.setColorsToLightDataColors(lightData);
            Initialized = true;
        }

        public void update()
        {
            if (!Initialized)
                return;

            SLight lightData = SceneNode.getLightData();
            AllColorsControl.updateMaterialColors(lightData);
        }

        protected bool Initialized;
        protected ILightSceneNode SceneNode;
        protected CAllColorsControl AllColorsControl;
    };

    /*
        Application configuration
    */
    class SConfig
    {
        public SConfig()
        {
            RenderInBackground = true;
            DriverType = E_DRIVER_TYPE.EDT_BURNINGSVIDEO;
            ScreenSize = new dimension2dui(640, 480);
        }
        public bool RenderInBackground;
        public E_DRIVER_TYPE DriverType;
        public dimension2dui ScreenSize;
    };

    /*
        Main application class
    */
    class CApp : IEventReceiver
    {


        // constructor
        public CApp()
        {
            IsRunning = false;
            Device = null;
            Camera = null;
            GlobalAmbient = null;
        }

        //// destructor
        //public ~CApp()
        //{
        //}

        // stop running - will quit at end of mainloop
        public void stopApp()
        {
            IsRunning = false;
        }

        // Event handler
        public override bool OnEvent(SEvent ev)
        {
            if (ev.EventType == EEVENT_TYPE.EET_GUI_EVENT)
            {
                IGUIEnvironment env = Device.getGUIEnvironment();

                switch (ev.GUIEvent.EventType)
                {
                    case EGUI_EVENT_TYPE.EGET_MENU_ITEM_SELECTED:
                        {
                            IGUIContextMenu menu = IGUIContextMenu.cast(ev.GUIEvent.Caller);
                            int id = menu.getItemCommandId((uint)menu.getSelectedItem());

                            switch (id)
                            {
                                case (int)Statics.EGUI_IDS.GUI_ID_OPEN_TEXTURE: // File . Open Texture
                                    env.addFileOpenDialog("Please select a texture file to open");
                                    break;
                                case (int)Statics.EGUI_IDS.GUI_ID_QUIT: // File . Quit
                                    stopApp();
                                    break;
                            }
                        }
                        break;

                    case EGUI_EVENT_TYPE.EGET_FILE_SELECTED:
                        {
                            // load the model file, selected in the file open dialog
                            IGUIFileOpenDialog dialog =
                                IGUIFileOpenDialog.cast(ev.GUIEvent.Caller);
                            loadTexture(dialog.getFileName());
                        }
                        break;

                    default:
                        break;
                }
            }

            return false;
        }



        // Application initialization
        // returns true when it was succesful initialized, otherwise false.
        public bool init(string[] argv)
        {
            // ask user for driver
            Config.DriverType = IrrlichtNet.driverChoiceConsole();
            if (Config.DriverType == E_DRIVER_TYPE.EDT_COUNT)
                return false;

            // create the device with the settings from our config
            Device = IrrlichtNet.createDevice(Config.DriverType, Config.ScreenSize);
            if (Device == null)
                return false;
            Device.setWindowCaption(Statics.DriverTypeNames[(int)Config.DriverType]);
            Device.setEventReceiver(this);

            ISceneManager smgr = Device.getSceneManager();
            IVideoDriver driver = Device.getVideoDriver();
            IGUIEnvironment guiEnv = Device.getGUIEnvironment();

            // set a nicer font
            IGUISkin skin = guiEnv.getSkin();
            IGUIFont font = guiEnv.getFont("../../media/fonthaettenschweiler.bmp");
            if (font != null)
                skin.setFont(font);

            // remove some alpha value because it makes those menus harder to read otherwise
            SColor col3dHighLight = new SColor();
            col3dHighLight.CopyFrom(skin.getColor(EGUI_DEFAULT_COLOR.EGDC_APP_WORKSPACE));
            col3dHighLight.setAlpha(255);
            SColor colHighLight = new SColor();
            colHighLight.CopyFrom(col3dHighLight);
            skin.setColor(EGUI_DEFAULT_COLOR.EGDC_HIGH_LIGHT, colHighLight);
            skin.setColor(EGUI_DEFAULT_COLOR.EGDC_3D_HIGH_LIGHT, col3dHighLight);

            // Add some textures which are useful to test material settings
            createDefaultTextures(driver);

            // create a menu 
            IGUIContextMenu menuBar = guiEnv.addMenu();
            menuBar.addItem("File", -1, true, true);

            IGUIContextMenu subMenuFile = menuBar.getSubMenu(0);
            subMenuFile.addItem("Open texture ...", (int)Statics.EGUI_IDS.GUI_ID_OPEN_TEXTURE);
            subMenuFile.addSeparator();
            subMenuFile.addItem("Quit", (int)Statics.EGUI_IDS.GUI_ID_QUIT);

            // a static camera
            Camera = smgr.addCameraSceneNode(null, new vector3df(0, 0, 0),
                                                new vector3df(0, 0, 100),
                                                -1);

            // add the nodes which are used to show the materials
            IMeshSceneNode nodeL = smgr.addCubeSceneNode(30.0f, null, -1,
                                               new vector3df(-35, 0, 100),
                                               new vector3df(0, 0, 0),
                                               new vector3df(1.0f, 1.0f, 1.0f));
            NodeLeft.init(nodeL, Device, new position2di(10, 20), "left node");

            IMeshSceneNode nodeR = smgr.addCubeSceneNode(30.0f, null, -1,
                                               new vector3df(35, 0, 100),
                                               new vector3df(0, 0, 0),
                                               new vector3df(1.0f, 1.0f, 1.0f));
            NodeRight.init(nodeR, Device, new position2di(530, 20), "right node");

            // add one light
            ILightSceneNode nodeLight = smgr.addLightSceneNode(null, new vector3df(0, 0, 0),
                                                            new SColorf(1.0f, 1.0f, 1.0f),
                                                            100.0f);
            LightControl.init(nodeLight, guiEnv, new position2di(270, 20), "light");

            // one large cube around everything. That's mainly to make the light more obvious.
            IMeshSceneNode backgroundCube = smgr.addCubeSceneNode(200.0f, null, -1, new vector3df(0, 0, 0),
                                               new vector3df(45, 0, 0),
                                               new vector3df(1.0f, 1.0f, 1.0f));
            backgroundCube.getMaterial(0).BackfaceCulling = false;	 		// we are within the cube, so we have to disable backface culling to see it
            backgroundCube.getMaterial(0).EmissiveColor.set(255, 50, 50, 50);	// we keep some self lighting to keep texts visible

            // set the ambient light value
            GlobalAmbient = new CColorControl(guiEnv, new position2di(270, 300), "global ambient", guiEnv.getRootGUIElement(), -1);
            GlobalAmbient.setColor(smgr.getAmbientLight().toSColor());

            return true;
        }

        // Update one frame
        protected bool update()
        {

            IVideoDriver videoDriver = Device.getVideoDriver();
            if (!Device.run())
                return false;

            if (Device.isWindowActive() || Config.RenderInBackground)
            {
                IGUIEnvironment guiEnv = Device.getGUIEnvironment();
                ISceneManager smgr = Device.getSceneManager();
                IGUISkin skin = guiEnv.getSkin();

                // update our controls
                NodeLeft.update();
                NodeRight.update();
                LightControl.update();

                // update ambient light settings
                if (GlobalAmbient.isDirty())
                {
                    smgr.setAmbientLight(new SColorf(GlobalAmbient.getColor()));
                    GlobalAmbient.resetDirty();
                }

                // draw everythings
                SColor bkColor = new SColor();
                bkColor.CopyFrom(skin.getColor(EGUI_DEFAULT_COLOR.EGDC_APP_WORKSPACE));
                videoDriver.beginScene(true, true, bkColor);

                smgr.drawAll();
                guiEnv.drawAll();

                videoDriver.endScene();
            }

            return true;
        }

        // Run the application. Our main loop.
        public void run()
        {
            IsRunning = true;

            if (Device == null)
                return;

            // main application loop
            while (IsRunning)
            {
                if (!update())
                    break;

                Device.sleep(5);
            }
        }

        // Close down the application
        public void quit()
        {
            IsRunning = false;
            //GlobalAmbient.drop();
            GlobalAmbient = null;
            if (Device != null)
            {
                Device.closeDevice();
                //Device.drop();
                Device.Dispose();
                Device = null;
            }
        }

        // Create some useful textures.
        // Note that the function put readability over speed, you shouldn't use setPixel at runtime but for initialization it's nice.
        protected void createDefaultTextures(IVideoDriver driver)
        {
            const uint width = 256;
            const uint height = 256;
            IImage imageA8R8G8B8 = driver.createImage(ECOLOR_FORMAT.ECF_A8R8G8B8, new dimension2dui(width, height));
            if (imageA8R8G8B8 == null)
                return;
            uint pitch = imageA8R8G8B8.getPitch();

            // some nice caro with 9 typical colors
            for (uint y = 0; y < height; ++y)
            {
                for (uint x = 0; x < pitch; ++x)
                {
                    if (y < height / 3)
                    {
                        if (x < width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_BLACK);
                        else if (x < 2 * width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_BLUE);
                        else
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_CYAN);
                    }
                    else if (y < 2 * height / 3)
                    {
                        if (x < width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_GRAY);
                        else if (x < 2 * width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_GREEN);
                        else
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_MAGENTA);
                    }
                    else
                    {
                        if (x < width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_RED);
                        else if (x < 2 * width / 3)
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_YELLOW);
                        else
                            imageA8R8G8B8.setPixel(x, y, Statics.SCOL_WHITE);
                    }
                }
            }
            driver.addTexture("CARO_A8R8G8B8", imageA8R8G8B8);

            // all white
            imageA8R8G8B8.fill(Statics.SCOL_WHITE);
            driver.addTexture("WHITE_A8R8G8B8", imageA8R8G8B8);

            // all black
            imageA8R8G8B8.fill(Statics.SCOL_BLACK);
            driver.addTexture("BLACK_A8R8G8B8", imageA8R8G8B8);

            // gray-scale
            for (uint y = 0; y < height; ++y)
            {
                for (uint x = 0; x < pitch; ++x)
                {
                    imageA8R8G8B8.setPixel(x, y, new SColor(y, x, x, x));
                }
            }
            driver.addTexture("GRAYSCALE_A8R8G8B8", imageA8R8G8B8);
        }

        // Load a texture and make sure nodes know it when more textures are available.
        protected void loadTexture(string name)
        {
            Device.getVideoDriver().getTexture(name);
            NodeLeft.updateTextures();
            NodeRight.updateTextures();
        }

        SConfig Config = new SConfig();
        volatile bool IsRunning;
        IrrlichtDevice Device;
        ICameraSceneNode Camera;
        SMeshNodeControl NodeLeft = new SMeshNodeControl();
        SMeshNodeControl NodeRight = new SMeshNodeControl();
        SLightNodeControl LightControl = new SLightNodeControl();
        CColorControl GlobalAmbient;
    };


}