﻿/** Example 021 Quake3 Explorer

This Tutorial shows how to load different Quake 3 maps.

Features:
	- Load BSP Archives at Runtime from the menu
	- Load a Map from the menu. Showing with Screenshot
	- Set the VideoDriver at runtime from menu
	- Adjust GammaLevel at runtime
	- Create SceneNodes for the Shaders
	- Load EntityList and create Entity SceneNodes
	- Create Players with Weapons and with Collison Respsone
	- Play music

You can download the Quake III Arena demo ( copyright id software )
at the following location:
ftp://ftp.idsoftware.com/idstuff/quake3/win32/q3ademo.exe

Copyright 2006-2009 Burningwater, Thomas Alten
*/
using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace Quake3Explorer
{
    class Program
    {
        static int Main(string[] args)
        {
	        GameData game = new GameData( System.Environment.CurrentDirectory + "\\");

	        // start without asking for driver
	        game.retVal = 1;
	        do
	        {
		    	// ask user for driver
		    	game.deviceParam.DriverType=IrrlichtNet.driverChoiceConsole();
    			if (game.deviceParam.DriverType==E_DRIVER_TYPE.EDT_COUNT)
	    			game.retVal = 3;

		        runGame ( game );
	        } while ( game.retVal < 3 );

	        return game.retVal;
        }

        static void runGame(GameData game)
        {
            if (game.retVal >= 3)
                return;

            game.Device = IrrlichtNet.createDeviceEx(game.deviceParam);
            if (null == game.Device)
            {
                // could not create selected driver.
                game.retVal = 0;
                return;
            }

            // create an event receiver based on current game data
            CQuake3EventHandler eventHandler = new CQuake3EventHandler(game);

            //! load stored config
            game.load("explorer.cfg");

            //! add our media directory and archive to the file system
            for (uint i = 0; i < game.CurrentArchiveList.size(); ++i)
            {
                eventHandler.AddArchive(game.CurrentArchiveList[i].ToString());
            }

            // Load a Map or startup to the GUI
            if (game.CurrentMapName != "")
            {
                eventHandler.LoadMap(game.CurrentMapName, 1);
                if (0 == game.loadParam.loadSkyShader)
                    eventHandler.AddSky(1, "skydome2");
                eventHandler.CreatePlayers();
                eventHandler.CreateGUI();
                eventHandler.SetGUIActive(0);

                // set player to last position on restart
                if (game.retVal == 2)
                {
                    eventHandler.GetPlayer(0).setpos(game.PlayerPosition, game.PlayerRotation);
                }
            }
            else
            {
                // start up empty
                eventHandler.AddSky(1, "skydome2");
                eventHandler.CreatePlayers();
                eventHandler.CreateGUI();
                eventHandler.SetGUIActive(1);
                Sound.background_music("IrrlichtTheme.ogg");
            }


            game.retVal = 3;
            while (game.Device.run())
            {
                eventHandler.Animate();
                eventHandler.Render();
                //if ( !game.Device.isWindowActive() )
                game.Device.yield();
            }

            game.Device.setGammaRamp(1.0f, 1.0f, 1.0f, 0.0f, 0.0f);
            eventHandler.Dispose();
        }
    }

    /*!
	    Game Data is used to hold Data which is needed to drive the game
    */
    class GameData
    {
        public GameData(string startupDir)
        {
            retVal = 0;
            //createExDevice = 0;
            Device = null;
            StartupDir = startupDir;
            setDefault();
        }
        public void setDefault()
        {
            debugState = (int)E_DEBUG_SCENE_TYPE.EDS_OFF;
            gravityState = 1;
            flyTroughState = 0;
            wireFrame = 0;
            guiActive = 1;
            guiInputActive = 0;
            GammaValue = 1.0f;

            // default deviceParam;
            deviceParam.DriverType = E_DRIVER_TYPE.EDT_DIRECT3D9;
            deviceParam.WindowSize.Width = 800;
            deviceParam.WindowSize.Height = 600;
            deviceParam.Fullscreen = false;
            deviceParam.Bits = 24;
            deviceParam.ZBufferBits = 32;
            deviceParam.Vsync = false;
            deviceParam.AntiAlias = 0;

            // default Quake3 loadParam
            loadParam.defaultLightMapMaterial = E_MATERIAL_TYPE.EMT_LIGHTMAP;
            loadParam.defaultModulate = E_MODULATE_FUNC.EMFN_MODULATE_1X;
            loadParam.defaultFilter = E_MATERIAL_FLAG.EMF_ANISOTROPIC_FILTER;
            loadParam.verbose = 2;
            loadParam.mergeShaderBuffer = 1;		// merge meshbuffers with same material
            loadParam.cleanUnResolvedMeshes = 1;	// should unresolved meshes be cleaned. otherwise blue texture
            loadParam.loadAllShaders = 1;			// load all scripts in the script directory
            loadParam.loadSkyShader = 0;			// load sky Shader
            loadParam.alpharef = 1;

            sound = 0;

            CurrentMapName = "";
            CurrentArchiveList.clear();

            //! Explorer Media directory
            CurrentArchiveList.push_back(StartupDir + "../../media/");

            //! Add the original quake3 files before you load your custom map
            //! Most mods are using the original shaders, models&items&weapons
            CurrentArchiveList.push_back("/q/baseq3/");


            CurrentArchiveList.push_back(StartupDir + "../../media/map-20kdm2.pk3");

        }
        public int save(string filename)
        {
            return 0;
            //if (Device != null)
            //{
            //    return 0;
            //}

            //string buf;
            //uint i;

            //// Store current Archive for restart
            //CurrentArchiveList.clear();
            //IFileSystem fs = Device.getFileSystem();
            //for ( i = 0; i != fs.getFileArchiveCount(); ++i )
            //{
            //    CurrentArchiveList.push_back ( fs.getFileArchive(i).getFileList().getPath() );
            //}

            //// Store Player Position and Rotation
            //ICameraSceneNode camera = Device.getSceneManager().getActiveCamera ();
            //if ( camera != null )
            //{
            //    PlayerPosition = camera.getPosition ();
            //    PlayerRotation = camera.getRotation ();
            //}

            //IWriteFile file = fs.createAndWriteFile ( filename );
            //if (file != null)
            //{
            //    return 0;
            //}
            //buf = string.Format("playerposition {0} {1} {2}\nplayerrotation {3} {4} {5}\n",
            //        PlayerPosition.X, PlayerPosition.Z, PlayerPosition.Y,
            //        PlayerRotation.X, PlayerRotation.Z, PlayerRotation.Y);
            //file.write ( buf, (s32) strlen ( buf ) );
            //for ( i = 0; i != fs->getFileArchiveCount(); ++i )
            //{
            //    snprintf ( buf, 128, "archive %s\n",stringc ( fs->getFileArchive(i)->getFileList()->getPath() ).c_str () );
            //    file->write ( buf, (s32) strlen ( buf ) );
            //}

            //file->drop ();
            //return 1;
            //    }
            //    public int load(string filename)
            //    {
            //        if (Device == null)
            //        {
            //            return 0;
            //        }

            //        //! the quake3 mesh loader can also handle *.shader and *.cfg file
            //        IQ3LevelMesh mesh = (IQ3LevelMesh)Device.getSceneManager().getMesh(filename);
            //        if (mesh == null)
            //        {
            //            return 0;
            //        }

            //        tQ3EntityList entityList = mesh.getEntityList();

            //        string s;
            //        uint pos;

            //        for (uint e = 0; e != entityList.size(); ++e)
            //        {
            //            //dumpShader ( s, &entityList[e], false );
            //            //printf ( s.c_str () );

            //            for (uint g = 0; g != entityList[e].getGroupSize(); ++g)
            //            {
            //                SVarGroup group = entityList[e].getGroup(g);

            //                for (uint index = 0; index < group.Variable.size(); ++index)
            //                {
            //                    SVariable v = group.Variable[index];
            //                    pos = 0;
            //                    if (v.name.ToString() == "playerposition")
            //                    {
            //                        PlayerPosition = IrrlichtNet.getAsVector3df(v.content, ref pos);
            //                    }
            //                    else
            //                        if (v.name.ToString() == "playerrotation")
            //                        {
            //                            PlayerRotation = IrrlichtNet.getAsVector3df(v.content, ref pos);
            //                        }
            //                }
            //            }
            //        }

            //        return 1;
        }
        public int load(string filename)
        {
            if (Device == null)
                return 0;

            //! the quake3 mesh loader can also handle *.shader and *.cfg file
            IQ3LevelMesh mesh = IQ3LevelMesh.cast(Device.getSceneManager().getMesh(filename));
            if (mesh == null)
                return 0;

            tQ3EntityList entityList = mesh.getEntityList();

            string s;
            uint pos;

            for (uint e = 0; e != entityList.size(); ++e)
            {
                //dumpShader ( s, &entityList[e], false );
                //printf ( s.c_str () );

                for (uint g = 0; g != entityList[e].getGroupSize(); ++g)
                {
                    SVarGroup group = entityList[e].getGroup(g);

                    for (uint index = 0; index < group.Variable.size(); ++index)
                    {
                        SVariable v = group.Variable[index];
                        pos = 0;
                        if (v.name.ToString() == "playerposition")
                        {
                            PlayerPosition = IrrlichtNet.getAsVector3df(v.content,ref pos);
                        }
                        else
                            if (v.name.ToString() == "playerrotation")
                            {
                                PlayerRotation = IrrlichtNet.getAsVector3df(v.content, ref pos);
                            }
                    }
                }
            }

            return 1;
        }

        public int debugState;
        public int gravityState;
        public int flyTroughState;
        public int wireFrame;
        public int guiActive;
        public int guiInputActive;
        public float GammaValue;
        public int retVal;
        public int sound;

        public string StartupDir;
        public string CurrentMapName;
        public arrayStringc CurrentArchiveList = new arrayStringc();

        public vector3df PlayerPosition = new vector3df();
        public vector3df PlayerRotation = new vector3df();

        public tQ3EntityList Variable = new tQ3EntityList();

        public Q3LevelLoadParameter loadParam = new Q3LevelLoadParameter();
        public SIrrlichtCreationParameters deviceParam = new SIrrlichtCreationParameters();
        //public funcptr_createDeviceEx createExDevice;
        public IrrlichtDevice Device;
    }


    /*!

        Representing a player
    */
    class Q3Player : IAnimationEndCallBack
    {
        internal Q3Player()
        {
            Device = null;
            MapParent = null;
            Mesh = null;
            WeaponNode = null;
            StartPositionCurrent = 0;
            animation = "";
            Anim = new TimeFire[4];
        }

        public override void OnAnimationEnd(IAnimatedMeshSceneNode node)
        {
            setAnim(null);
        }

        public void create(IrrlichtDevice device,
                        IQ3LevelMesh mesh,
                        ISceneNode mapNode,
                        IMetaTriangleSelector meta
                    )
        {
            Q3Factory.setTimeFire(Anim[0], 200, (uint)eTimeFireFlag.FIRED);
            Q3Factory.setTimeFire(Anim[1], 5000, 0);

            // load FPS weapon to Camera
            Device = device;
            Mesh = mesh;
            MapParent = mapNode;

            ISceneManager smgr = device.getSceneManager();
            IVideoDriver driver = device.getVideoDriver();

            ICameraSceneNode camera = null;

            SKeyMap[] keyMap = new SKeyMap[10];
            for (int i = 0; i < keyMap.Length; i++)
            {
                keyMap[i] = new SKeyMap();
            }
            keyMap[0].Action = EKEY_ACTION.EKA_MOVE_FORWARD;
            keyMap[0].KeyCode = EKEY_CODE.KEY_UP;
            keyMap[1].Action = EKEY_ACTION.EKA_MOVE_FORWARD;
            keyMap[1].KeyCode = EKEY_CODE.KEY_KEY_W;

            keyMap[2].Action = EKEY_ACTION.EKA_MOVE_BACKWARD;
            keyMap[2].KeyCode = EKEY_CODE.KEY_DOWN;
            keyMap[3].Action = EKEY_ACTION.EKA_MOVE_BACKWARD;
            keyMap[3].KeyCode = EKEY_CODE.KEY_KEY_S;

            keyMap[4].Action = EKEY_ACTION.EKA_STRAFE_LEFT;
            keyMap[4].KeyCode = EKEY_CODE.KEY_LEFT;
            keyMap[5].Action = EKEY_ACTION.EKA_STRAFE_LEFT;
            keyMap[5].KeyCode = EKEY_CODE.KEY_KEY_A;

            keyMap[6].Action = EKEY_ACTION.EKA_STRAFE_RIGHT;
            keyMap[6].KeyCode = EKEY_CODE.KEY_RIGHT;
            keyMap[7].Action = EKEY_ACTION.EKA_STRAFE_RIGHT;
            keyMap[7].KeyCode = EKEY_CODE.KEY_KEY_D;

            keyMap[8].Action = EKEY_ACTION.EKA_JUMP_UP;
            keyMap[8].KeyCode = EKEY_CODE.KEY_KEY_J;

            keyMap[9].Action = EKEY_ACTION.EKA_CROUCH;
            keyMap[9].KeyCode = EKEY_CODE.KEY_KEY_C;

            camera = smgr.addCameraSceneNodeFPS(null, 100.0f, 0.6f, -1, keyMap, 10, false, 0.6f);
            camera.setName("First Person Camera");
            //camera->setFOV ( 100.f * core::DEGTORAD );
            camera.setFarValue(20000.0f);

            IAnimatedMeshMD2 weaponMesh = IAnimatedMeshMD2.cast(smgr.getMesh("gun.md2"));
            if (null == weaponMesh)
            {
                return;
            }

            if (weaponMesh.getMeshType() == E_ANIMATED_MESH_TYPE.EAMT_MD2)
            {
                int count = weaponMesh.getAnimationCount();
                for (int i = 0; i != count; ++i)
                {
                    buf = string.Format("Animation: {0}", weaponMesh.getAnimationName(i));
                    device.getLogger().log(buf, ELOG_LEVEL.ELL_INFORMATION);
                }
            }

            WeaponNode = smgr.addAnimatedMeshSceneNode(
                                weaponMesh,
                                smgr.getActiveCamera(),
                                10,
                                new vector3df(0, 0, 0),
                                new vector3df(-90, -90, 90)
                                );
            WeaponNode.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            WeaponNode.setMaterialTexture(0, driver.getTexture("gun.jpg"));
            WeaponNode.setLoopMode(false);
            WeaponNode.setName("tommi the gun man");

            //create a collision auto response animator
            ISceneNodeAnimator anim =
                smgr.createCollisionResponseAnimator(meta, camera,
                    new vector3df(30, 45, 30),
                    Q3Factory.getGravity("earth"),
                    new vector3df(0, 40, 0),
                    0.0005f
                );

            camera.addAnimator(anim);
            //anim.drop();

            if (meta != null)
            {
                //meta.drop ();
            }

            respawn();
            setAnim("idle");
        }

        public void shutdown()
        {
            setAnim("");

            Q3Factory.dropElement(WeaponNode);

            if (Device != null)
            {
                ICameraSceneNode camera = Device.getSceneManager().getActiveCamera();
                Q3Factory.dropElement(camera);
                Device = null;
            }

            MapParent = null;
            Mesh = null;
        }
        public void setAnim(string name)
        {
            if (name != null)
            {
                animation = name;
                if (WeaponNode != null)
                {
                    WeaponNode.setAnimationEndCallback(this);
                    WeaponNode.setMD2Animation(animation);
                }
            }
            else
            {
                animation = "";
                if (WeaponNode != null)
                {
                    WeaponNode.setAnimationEndCallback(null);
                }
            }
        }
        public void respawn()
        {
            ICameraSceneNode camera = Device.getSceneManager().getActiveCamera();

            Device.getLogger().log("respawn");

            if (StartPositionCurrent >= Q3Factory.Q3StartPosition(
                    Mesh, camera, StartPositionCurrent++,
                    cam().getEllipsoidTranslation())
                )
            {
                StartPositionCurrent = 0;
            }
        }
        public void setpos(vector3df pos, vector3df rotation)
        {
            Device.getLogger().log("setpos");

            ICameraSceneNode camera = Device.getSceneManager().getActiveCamera();
            if (camera != null)
            {
                camera.setPosition(pos);
                camera.setRotation(rotation);
                //! New. FPSCamera and animators catches reset on animate 0
                camera.OnAnimate(0);
            }
        }

        public ISceneNodeAnimatorCollisionResponse cam()
        {
            return Q3Factory.camCollisionResponse(Device);
        }

        IrrlichtDevice Device;
        ISceneNode MapParent;
        IQ3LevelMesh Mesh;
        public IAnimatedMeshSceneNode WeaponNode;
        int StartPositionCurrent;
        public TimeFire[] Anim;
        public string animation;
        string buf;
    }


    //! GUIElements
    class GUI
    {
        internal GUI()
        {
        }

        internal void drop()
        {
            Q3Factory.dropElement(Window);
            Q3Factory.dropElement(Logo);
        }

        internal IGUIComboBox VideoDriver = null;
        internal IGUIComboBox VideoMode = null;
        internal IGUICheckBox FullScreen = null;
        internal IGUICheckBox Bit32 = null;
        internal IGUIScrollBar MultiSample = null;
        internal IGUIButton SetVideoMode = null;

        internal IGUIScrollBar Tesselation = null;
        internal IGUIScrollBar Gamma = null;
        internal IGUICheckBox Collision = null;
        internal IGUICheckBox Visible_Map = null;
        internal IGUICheckBox Visible_Shader = null;
        internal IGUICheckBox Visible_Fog = null;
        internal IGUICheckBox Visible_Unresolved = null;
        internal IGUICheckBox Visible_Skydome = null;
        internal IGUIButton Respawn = null;

        internal IGUITable ArchiveList = null;
        internal IGUIButton ArchiveAdd = null;
        internal IGUIButton ArchiveRemove = null;
        internal IGUIFileOpenDialog ArchiveFileOpen = null;
        internal IGUIButton ArchiveUp = null;
        internal IGUIButton ArchiveDown = null;

        internal IGUIListBox MapList = null;
        internal IGUITreeView SceneTree = null;
        internal IGUIStaticText StatusLine = null;
        internal IGUIImage Logo = null;
        internal IGUIWindow Window = null;

    }
    
/*!
	CQuake3EventHandler controls the game
*/
    class CQuake3EventHandler : IEventReceiver
    {
        public CQuake3EventHandler(GameData gameData)
        {
            Game = gameData;
            Mesh = null;
            MapParent = null;
            ShaderParent = null;
            ItemParent = null;
            UnresolvedParent = null;
            BulletParent = null;
            FogParent = null;
            SkyNode = null;
            Meta = null;

            buf = "";
            //! Also use 16 Bit Textures for 16 Bit RenderDevice
            if (Game.deviceParam.Bits == 16)
            {
                gameData.Device.getVideoDriver().setTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_ALWAYS_16_BIT, true);
            }

            // Quake3 Shader controls Z-Writing
            gameData.Device.getSceneManager().getParameters().setAttribute(IrrlichtNet.ALLOW_ZWRITE_ON_TRANSPARENT, true);

            // create internal textures
            createTextures();

            Sound.sound_init(gameData.Device);

            Game.Device.setEventReceiver(this);
        }
        ~CQuake3EventHandler()
        {
            Player[0].shutdown();
            Sound.sound_shutdown();

            Game.save("explorer.cfg");

            //Game.Device.drop();
        }
        public void Animate()
        {
            uint now = Game.Device.getTimer().getTime();

            Q3Player player = Player[0];

            Q3Factory.checkTimeFire(player.Anim, 4, now);

            // Query Scene Manager attributes
            if ((player.Anim[0].flags & (uint)eTimeFireFlag.FIRED) != 0)
            {
                ISceneManager smgr = Game.Device.getSceneManager();
                string msg;
                IVideoDriver driver = Game.Device.getVideoDriver();

                IAttributes attr = smgr.getParameters();
                msg = string.Format("Q3 {0} [{1}], FPS:{2:D3} Tri:{3:f3}m Cull {4}/{5} nodes ({6},{7},{8})",
                    Game.CurrentMapName,
                    driver.getName(),
                    driver.getFPS(),
                    (float)driver.getPrimitiveCountDrawn(0) * (1.0f / 1000000.0f),
                    attr.getAttributeAsInt("culled"),
                    attr.getAttributeAsInt("calls"),
                    attr.getAttributeAsInt("drawn_solid"),
                    attr.getAttributeAsInt("drawn_transparent"),
                    attr.getAttributeAsInt("drawn_transparent_effect")
                    );
                Game.Device.setWindowCaption(msg);

                msg = string.Format(
                            "{0:D3} fps, F1 GUI on/off, F2 respawn, F3-F6 toggle Nodes, F7 Collision on/off F8 Gravity on/off, Right Mouse Toggle GUI",
                            Game.Device.getVideoDriver().getFPS()
                        );
                if (gui.StatusLine != null)
                    gui.StatusLine.setText(msg);
                player.Anim[0].flags &= ~(uint)eTimeFireFlag.FIRED;
            }

            // idle..
            if ((player.Anim[1].flags & (uint)eTimeFireFlag.FIRED) != 0)
            {
                if (player.animation == "idle")
                    player.setAnim("idle");

                player.Anim[1].flags &= ~(uint)eTimeFireFlag.FIRED;
            }

            createParticleImpacts(now);

        }
        public void Render() 
        {
            IVideoDriver driver = Game.Device.getVideoDriver();
            if (null == driver)
                return;

            // TODO: This does not work, yet.
            bool anaglyph = false;
            if (anaglyph)
            {
                ICameraSceneNode cameraOld = Game.Device.getSceneManager().getActiveCamera();
                driver.beginScene(true, true, new SColor(0, 0, 0, 0));
                driver.getOverrideMaterial().Material.ColorMask = (byte)E_COLOR_PLANE.ECP_NONE;
                driver.getOverrideMaterial().EnableFlags = (uint)E_MATERIAL_FLAG.EMF_COLOR_MASK;
                driver.getOverrideMaterial().EnablePasses =(ushort)( E_SCENE_NODE_RENDER_PASS.ESNRP_SKY_BOX |
                                                            E_SCENE_NODE_RENDER_PASS.ESNRP_SOLID |
                                                            E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT |
                                                            E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT_EFFECT |
                                                            E_SCENE_NODE_RENDER_PASS.ESNRP_SHADOW);
                Game.Device.getSceneManager().drawAll();
                driver.clearZBuffer();
                
                vector3df oldPosition = cameraOld.getPosition();
                vector3df oldTarget = cameraOld.getTarget();
                CMatrix4f startMatrix = cameraOld.getAbsoluteTransformation();
                vector3df focusPoint = (oldTarget -
                        cameraOld.getAbsolutePosition()).setLength(10000) +
                        cameraOld.getAbsolutePosition();

                ICameraSceneNode camera = cameraOld; //Game.Device.getSceneManager().addCameraSceneNode();

                //Left eye...
                vector3df pos;
                CMatrix4f move = new CMatrix4f();

                move.setTranslation(new vector3df(-1.5f, 0.0f, 0.0f));
                pos = (startMatrix * move).getTranslation();

                driver.getOverrideMaterial().Material.ColorMask = (byte)E_COLOR_PLANE.ECP_RED;
                driver.getOverrideMaterial().EnableFlags = (uint)E_MATERIAL_FLAG.EMF_COLOR_MASK;
                driver.getOverrideMaterial().EnablePasses =
                        (ushort)(E_SCENE_NODE_RENDER_PASS.ESNRP_SKY_BOX |
                        E_SCENE_NODE_RENDER_PASS.ESNRP_SOLID |
                        E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT |
                        E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT_EFFECT |
                        E_SCENE_NODE_RENDER_PASS.ESNRP_SHADOW);

                camera.setPosition(pos);
                camera.setTarget(focusPoint);

                Game.Device.getSceneManager().drawAll();
                driver.clearZBuffer();

                //Right eye...
                move.setTranslation(new vector3df(1.5f, 0.0f, 0.0f));
                pos = (startMatrix * move).getTranslation();

                driver.getOverrideMaterial().Material.ColorMask = (byte)(E_COLOR_PLANE.ECP_GREEN | E_COLOR_PLANE.ECP_BLUE);
                driver.getOverrideMaterial().EnableFlags = (uint)E_MATERIAL_FLAG.EMF_COLOR_MASK;
                driver.getOverrideMaterial().EnablePasses =
                        (ushort)(E_SCENE_NODE_RENDER_PASS.ESNRP_SKY_BOX | 
                        E_SCENE_NODE_RENDER_PASS.ESNRP_SOLID | 
                        E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT |
                        E_SCENE_NODE_RENDER_PASS.ESNRP_TRANSPARENT_EFFECT | 
                        E_SCENE_NODE_RENDER_PASS.ESNRP_SHADOW);

                camera.setPosition(pos);
                camera.setTarget(focusPoint);

                Game.Device.getSceneManager().drawAll();

                driver.getOverrideMaterial().Material.ColorMask = (byte)E_COLOR_PLANE.ECP_ALL;
                driver.getOverrideMaterial().EnableFlags = 0;
                driver.getOverrideMaterial().EnablePasses = 0;

                if (camera != cameraOld)
                {
                    Game.Device.getSceneManager().setActiveCamera(cameraOld);
                    camera.remove();
                }
                else
                {
                    camera.setPosition(oldPosition);
                    camera.setTarget(oldTarget);
                }
            }
            else
            {
                driver.beginScene(true, true, new SColor(0, 0, 0, 0));
                Game.Device.getSceneManager().drawAll();
            }

            Game.Device.getGUIEnvironment().drawAll();
            driver.endScene();
        }

        public void AddArchive(string archiveName)
        {
            IFileSystem fs = Game.Device.getFileSystem();
            uint i;

            if (archiveName.Length != 0)
            {
                bool exists = false;
                for (i = 0; i != fs.getFileArchiveCount(); ++i)
                {
                    if (fs.getFileArchive(i).getFileList().getPath().ToString() == archiveName)
                    {
                        exists = true;
                        break;
                    }
                }

                if (!exists)
                {
                    fs.addFileArchive(archiveName, true, false);
                }
            }

            // store the current archives in game data
            // show the attached Archive in proper order
            if (gui.ArchiveList != null)
            {
                gui.ArchiveList.clearRows();

                for (i = 0; i != fs.getFileArchiveCount(); ++i)
                {
                    IFileArchive archive = fs.getFileArchive(i);

                    uint index = gui.ArchiveList.addRow(i);

                    string typeName;
                    switch (archive.getType())
                    {
                        case (int)E_FILE_ARCHIVE_TYPE.EFAT_ZIP:
                            typeName = "ZIP";
                            break;
                        case (int)E_FILE_ARCHIVE_TYPE.EFAT_GZIP:
                            typeName = "gzip";
                            break;
                        case (int)E_FILE_ARCHIVE_TYPE.EFAT_FOLDER:
                            typeName = "Mount";
                            break;
                        case (int)E_FILE_ARCHIVE_TYPE.EFAT_PAK:
                            typeName = "PAK";
                            break;
                        case (int)E_FILE_ARCHIVE_TYPE.EFAT_TAR:
                            typeName = "TAR";
                            break;
                        default:
                            typeName = "archive";
                            break;
                    }

                    gui.ArchiveList.setCellText(index, 0, typeName);
                    gui.ArchiveList.setCellText(index, 1, archive.getFileList().getPath().ToString());
                }
            }


            // browse the archives for maps
            if (gui.MapList != null)
            {
                gui.MapList.clear();

                IGUISpriteBank bank = Game.Device.getGUIEnvironment().getSpriteBank("sprite_q3map");
                if (null == bank)
                    bank = Game.Device.getGUIEnvironment().addEmptySpriteBank("sprite_q3map");

                SGUISprite sprite = new SGUISprite();
                SGUISpriteFrame frame = new SGUISpriteFrame();
                recti r = new recti();

                bank.getSprites().clear();
                bank.getPositions().clear();
                gui.MapList.setSpriteBank(bank);

                uint g = 0;
                string s;

                //! browse the attached file system
                fs.setFileListSystem(EFileSystemType.FILESYSTEM_VIRTUAL);
                fs.changeWorkingDirectoryTo("/maps/");
                IFileList fileList = fs.createFileList();
                fs.setFileListSystem(EFileSystemType.FILESYSTEM_NATIVE);

                for (i = 0; i < fileList.getFileCount(); ++i)
                {
                    s = fileList.getFullFileName(i).ToString();
                    if (s.IndexOf(".bsp") >= 0)
                    {
                        // get level screenshot. reformat texture to 128x128
                        string c = s;
                        c = System.IO.Path.GetFileNameWithoutExtension(c);
                        //c = IrrlichtNet.deletePathFromFilename((IrrStringc)c);
                        //c = IrrlichtNet.cutFilenameExtension((IrrStringc)c, (IrrStringc)c);
                        c = "levelshots/" + c;

                        dimension2dui dim = new dimension2dui(128, 128);
                        IVideoDriver driver = Game.Device.getVideoDriver();
                        IImage image = null;
                        ITexture tex = null;
                        string filename;

                        filename = c + ".jpg";
                        if (fs.existFile(filename))
                            image = driver.createImageFromFile(filename);
                        if (null == image)
                        {
                            filename = c + ".tga";
                            if (fs.existFile(filename))
                                image = driver.createImageFromFile(filename);
                        }

                        if (image != null)
                        {
                            IImage filter = driver.createImage(ECOLOR_FORMAT.ECF_R8G8B8, dim);
                            image.copyToScalingBoxFilter(filter, 0);
                            //image.drop();
                            image = filter;
                        }

                        if (image != null)
                        {
                            tex = driver.addTexture(filename, image);
                            //image.drop();
                        }


                        bank.setTexture(g, tex);

                        r.LowerRightCorner.X = (int)dim.Width;
                        r.LowerRightCorner.Y = (int)dim.Height;
                        gui.MapList.setItemHeight(r.LowerRightCorner.Y + 4);
                        frame.rectNumber = bank.getPositions().size();
                        frame.textureNumber = g;

                        bank.getPositions().push_back(r);

                        sprite.Frames.set_used(0);
                        sprite.Frames.push_back(frame);
                        sprite.frameTime = 0;
                        bank.getSprites().push_back(sprite);

                        gui.MapList.addItem(s, (int)g);
                        g += 1;
                    }
                }
                //fileList.drop();

                gui.MapList.setSelected(-1);
                IGUIScrollBar bar = IGUIScrollBar.cast(gui.MapList.getElementFromId(0));
                if (bar != null)
                    bar.setPos(0);

            }

        }
        public void LoadMap(string mapName, int collision)
        {
            if (0 == mapName.Length)
                return;

            dropMap();

            IFileSystem fs = Game.Device.getFileSystem();
            ISceneManager smgr = Game.Device.getSceneManager();

            System.IO.BinaryWriter bw = new System.IO.BinaryWriter(new System.IO.FileStream("levelparameter.cfg", System.IO.FileMode.Create));
            bw.Write((int)Game.loadParam.defaultLightMapMaterial);
            bw.Write((int)Game.loadParam.defaultModulate);
            bw.Write((int)Game.loadParam.defaultFilter);
            bw.Write(Game.loadParam.patchTesselation);
            bw.Write(Game.loadParam.verbose);
            bw.Write(Game.loadParam.startTime);
            bw.Write(Game.loadParam.endTime);
            bw.Write(Game.loadParam.mergeShaderBuffer);
            bw.Write(Game.loadParam.cleanUnResolvedMeshes);
            bw.Write(Game.loadParam.loadAllShaders);
            bw.Write(Game.loadParam.loadSkyShader);
            bw.Write(Game.loadParam.alpharef);
            bw.Write(Game.loadParam.swapLump);
            bw.Write(Game.loadParam.swapHeader);
            bw.Write(Game.loadParam.scriptDir.ToCharArray());
            for (int i = Game.loadParam.scriptDir.Length; i < 64; i++)
            {
                bw.Write((byte)0);
            }
            bw.Close();
            //IReadFile file = fs.createMemoryReadFile ( &Game.loadParam, sizeof ( Game.loadParam ),
            //                                                "levelparameter.cfg", false);
            IReadFile file = null;

            smgr.getMesh("levelparameter.cfg");
            //file.drop();

            Mesh = IQ3LevelMesh.cast(smgr.getMesh(mapName));
            if (null == Mesh)
                return;

            /*
                add the geometry mesh to the Scene ( polygon & patches )
                The Geometry mesh is optimised for faster drawing
            */

            IMesh geometry = Mesh.getMesh((int)eQ3MeshIndex.E_Q3_MESH_GEOMETRY);
            if (null == geometry || geometry.getMeshBufferCount() == 0)
                return;

            Game.CurrentMapName = mapName;

            //create a collision list
            Meta = null;

            ITriangleSelector selector = null;
            if (collision != 0)
                Meta = smgr.createMetaTriangleSelector();

            //IMeshBuffer b0 = geometry.getMeshBuffer(0);
            //s32 minimalNodes = b0 ? core::s32_max ( 2048, b0.getVertexCount() / 32 ) : 2048;
            int minimalNodes = 2048;

            MapParent = smgr.addMeshSceneNode(geometry);

            //MapParent = smgr.addOctTreeSceneNode(geometry, 0, -1, minimalNodes);
            MapParent.setName(mapName);
            if (Meta != null)
            {
                selector = smgr.createOctTreeTriangleSelector(geometry, MapParent, minimalNodes);
                //selector = smgr.createTriangleSelector ( geometry, MapParent );
                Meta.addTriangleSelector(selector);
                //selector.drop();
            }

            // logical parent for the items
            ItemParent = smgr.addEmptySceneNode();
            if (ItemParent != null)
                ItemParent.setName("Item Container");

            ShaderParent = smgr.addEmptySceneNode();
            if (ShaderParent != null)
                ShaderParent.setName("Shader Container");

            UnresolvedParent = smgr.addEmptySceneNode();
            if (UnresolvedParent != null)
                UnresolvedParent.setName("Unresolved Container");

            FogParent = smgr.addEmptySceneNode();
            if (FogParent != null)
                FogParent.setName("Fog Container");

            // logical parent for the bullets
            BulletParent = smgr.addEmptySceneNode();
            if (BulletParent != null)
                BulletParent.setName("Bullet Container");


            /*
                now construct SceneNodes for each Shader
                The Objects are stored in the quake mesh E_Q3_MESH_ITEMS
                and the Shader ID is stored in the MaterialParameters
                mostly dark looking skulls and moving lava.. or green flashing tubes?
            */
            Q3Factory.Q3ShaderFactory(Game.loadParam, Game.Device, Mesh, eQ3MeshIndex.E_Q3_MESH_ITEMS, ShaderParent, Meta, false);
            Q3Factory.Q3ShaderFactory(Game.loadParam, Game.Device, Mesh, eQ3MeshIndex.E_Q3_MESH_FOG, FogParent, null, false);
            Q3Factory.Q3ShaderFactory(Game.loadParam, Game.Device, Mesh, eQ3MeshIndex.E_Q3_MESH_UNRESOLVED, UnresolvedParent, Meta, true);


            /*
                Now construct Models from Entity List
            */
            Q3Factory.Q3ModelFactory(Game.loadParam, Game.Device, Mesh, ItemParent, false);

        }
        public void CreatePlayers() 
        {
            Player[0].create(Game.Device, Mesh, MapParent, Meta);
        }

        // irrlicht order
		//static const c8*p[] = { "ft", "lf", "bk", "rt", "up", "dn" };
		// quake3 order
		string[] p = { "ft", "rt", "bk", "lf", "up", "dn" };
        public void AddSky(uint dome, string texture)
        {
            ISceneManager smgr = Game.Device.getSceneManager();
            IVideoDriver driver = Game.Device.getVideoDriver();

            bool oldMipMapState = driver.getTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_CREATE_MIP_MAPS);
            driver.setTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_CREATE_MIP_MAPS, false);

            if (0 == dome)
            {


                uint i = 0;
                buf = string.Format("{0}_{1}.jpg", texture, p[i]);
                SkyNode = smgr.addSkyBoxSceneNode(driver.getTexture(buf), null, null, null, null, null);

                for (i = 0; i < 6; ++i)
                {
                    buf = string.Format("{0}_{1}.jpg", texture, p[i]);
                    SkyNode.getMaterial(i).setTexture(0, driver.getTexture(buf));
                }
            }
            else
                if (1 == dome)
                {
                    buf = string.Format("{0}.jpg", texture);
                    SkyNode = smgr.addSkyDomeSceneNode(
                        driver.getTexture(buf),
                        32, 32,
                        1.0f,
                        1.0f,
                        1000.0f,
                        null,
                        11
                        );
                }
                else
                    if (2 == dome)
                    {
                        buf = string.Format("{0}.jpg", texture);
                        SkyNode = smgr.addSkyDomeSceneNode(
                            driver.getTexture(buf),
                            16, 8,
                            0.95f,
                            2.0f,
                            1000.9f,
                            null,
                            11
                            );
                    }

            SkyNode.setName("Skydome");
            //SkyNode.getMaterial(0).ZBuffer = video::EMDF_DEPTH_LESS_EQUAL;

            driver.setTextureCreationFlag(E_TEXTURE_CREATION_FLAG.ETCF_CREATE_MIP_MAPS, oldMipMapState);

        }
        public Q3Player GetPlayer(uint index) { return Player[index]; }

        public void CreateGUI()
        {
            IGUIEnvironment env = Game.Device.getGUIEnvironment();
            IVideoDriver driver = Game.Device.getVideoDriver();

            gui.drop();

            // set skin font
            IGUIFont font = env.getFont("fontlucida.png");
            if (font != null)
            {
                env.getSkin().setFont(font);
            }
            env.getSkin().setColor(EGUI_DEFAULT_COLOR.EGDC_BUTTON_TEXT, new SColor(240, 0xAA, 0xAA, 0xAA));
            env.getSkin().setColor(EGUI_DEFAULT_COLOR.EGDC_3D_HIGH_LIGHT, new SColor(240, 0x22, 0x22, 0x22));
            env.getSkin().setColor(EGUI_DEFAULT_COLOR.EGDC_3D_FACE, new SColor(240, 0x44, 0x44, 0x44));
            env.getSkin().setColor(EGUI_DEFAULT_COLOR.EGDC_WINDOW, new SColor(240, 0x66, 0x66, 0x66));

            // minimal gui size 800x600
            dimension2dui dim = new dimension2dui(800, 600);
            dimension2dui tmp = Game.Device.getVideoDriver().getScreenSize();
            dimension2dui vdim = new dimension2dui(tmp.Width, tmp.Height);

            if (vdim.Height >= dim.Height && vdim.Width >= dim.Width)
            {
                //dim = vdim;
            }
            else
            {
            }

            gui.Window = env.addWindow(new recti(0, 0, (int)dim.Width, (int)dim.Height), false, "Quake3 Explorer");
            gui.Window.setToolTipText("Quake3Explorer. Loads and show various BSP File Format and Shaders.");
            gui.Window.getCloseButton().setToolTipText("Quit Quake3 Explorer");

            // add a status line help text
            gui.StatusLine = env.addStaticText("", new recti(5, (int)dim.Height - 30, (int)dim.Width - 5, (int)dim.Height - 10),
                                        false, false, gui.Window, -1, true);


            env.addStaticText("VideoDriver:", new recti((int)dim.Width - 400, 24, (int)dim.Width - 310, 40), false, false, gui.Window, -1, false);
            gui.VideoDriver = env.addComboBox(new recti((int)dim.Width - 300, 24, (int)dim.Width - 10, 40), gui.Window);
            gui.VideoDriver.addItem("Direct3D 9.0c", (uint)E_DRIVER_TYPE.EDT_DIRECT3D9);
            gui.VideoDriver.addItem("Direct3D 8.1", (uint)E_DRIVER_TYPE.EDT_DIRECT3D8);
            gui.VideoDriver.addItem("OpenGL 1.5", (uint)E_DRIVER_TYPE.EDT_OPENGL);
            gui.VideoDriver.addItem("Software Renderer", (uint)E_DRIVER_TYPE.EDT_SOFTWARE);
            gui.VideoDriver.addItem("Burning's Video (TM) Thomas Alten", (uint)E_DRIVER_TYPE.EDT_BURNINGSVIDEO);
            gui.VideoDriver.setSelected(gui.VideoDriver.getIndexForItemData((uint)Game.deviceParam.DriverType));
            gui.VideoDriver.setToolTipText("Use a VideoDriver");

            env.addStaticText("VideoMode:", new recti((int)dim.Width - 400, 44, (int)dim.Width - 310, 60), false, false, gui.Window, -1, false);
            gui.VideoMode = env.addComboBox(new recti((int)dim.Width - 300, 44, (int)dim.Width - 10, 60), gui.Window);
            gui.VideoMode.setToolTipText("Supported Screenmodes");
            IVideoModeList modeList = Game.Device.getVideoModeList();
            if (modeList != null)
            {
                int i;
                for (i = 0; i != modeList.getVideoModeCount(); ++i)
                {
                    int d = modeList.getVideoModeDepth(i);
                    if (d < 16)
                    {
                        continue;
                    }

                    uint w = modeList.getVideoModeResolution(i).Width;
                    uint h = modeList.getVideoModeResolution(i).Height;
                    uint val = w << 16 | h;

                    if (gui.VideoMode.getIndexForItemData(val) >= 0)
                    {
                        continue;
                    }

                    float aspect = (float)w / (float)h;
                    string a = "";
                    if (IrrlichtNet.equals(aspect, 1.3333333333f)) { a = "4:3"; }
                    else if (IrrlichtNet.equals(aspect, 1.6666666f)) { a = "15:9 widescreen"; }
                    else if (IrrlichtNet.equals(aspect, 1.7777777f)) { a = "16:9 widescreen"; }
                    else if (IrrlichtNet.equals(aspect, 1.6f)) { a = "16:10 widescreen"; }
                    else if (IrrlichtNet.equals(aspect, 2.133333f)) { a = "20:9 widescreen"; }

                    buf = string.Format("{0} x {1}, {2}", w, h, a);
                    gui.VideoMode.addItem(buf, val);
                }
            }
            gui.VideoMode.setSelected(gui.VideoMode.getIndexForItemData(
                                            Game.deviceParam.WindowSize.Width << 16 |
                                            Game.deviceParam.WindowSize.Height));

            gui.FullScreen = env.addCheckBox(Game.deviceParam.Fullscreen, new recti((int)dim.Width - 400, 64, (int)dim.Width - 300, 80), gui.Window, -1, "Fullscreen");
            gui.FullScreen.setToolTipText("Set Fullscreen or Window Mode");

            gui.Bit32 = env.addCheckBox(Game.deviceParam.Bits == 32, new recti((int)dim.Width - 300, 64, (int)dim.Width - 240, 80), gui.Window, -1, "32Bit");
            gui.Bit32.setToolTipText("Use 16 or 32 Bit");

            env.addStaticText("MultiSample:", new recti((int)dim.Width - 235, 64, (int)dim.Width - 150, 80), false, false, gui.Window, -1, false);
            gui.MultiSample = env.addScrollBar(true, new recti((int)dim.Width - 150, 64, (int)dim.Width - 70, 80), gui.Window, -1);
            gui.MultiSample.setMin(0);
            gui.MultiSample.setMax(8);
            gui.MultiSample.setSmallStep(1);
            gui.MultiSample.setLargeStep(1);
            gui.MultiSample.setPos(Game.deviceParam.AntiAlias);
            gui.MultiSample.setToolTipText("Set the MultiSample (disable, 1x, 2x, 4x, 8x )");

            gui.SetVideoMode = env.addButton(new recti((int)dim.Width - 60, 64, (int)dim.Width - 10, 80), gui.Window, -1, "set");
            gui.SetVideoMode.setToolTipText("Set Video Mode with current values");

            env.addStaticText("Gamma:", new recti((int)dim.Width - 400, 104, (int)dim.Width - 310, 120), false, false, gui.Window, -1, false);
            gui.Gamma = env.addScrollBar(true, new recti((int)dim.Width - 300, 104, (int)dim.Width - 10, 120), gui.Window, -1);
            gui.Gamma.setMin(50);
            gui.Gamma.setMax(350);
            gui.Gamma.setSmallStep(1);
            gui.Gamma.setLargeStep(10);
            gui.Gamma.setPos((int)Math.Floor(Game.GammaValue * 100.0f));
            gui.Gamma.setToolTipText("Adjust Gamma Ramp ( 0.5 - 3.5)");
            Game.Device.setGammaRamp(Game.GammaValue, Game.GammaValue, Game.GammaValue, 0.0f, 0.0f);


            env.addStaticText("Tesselation:", new recti((int)dim.Width - 400, 124, (int)dim.Width - 310, 140), false, false, gui.Window, -1, false);
            gui.Tesselation = env.addScrollBar(true, new recti((int)dim.Width - 300, 124, (int)dim.Width - 10, 140), gui.Window, -1);
            gui.Tesselation.setMin(2);
            gui.Tesselation.setMax(12);
            gui.Tesselation.setSmallStep(1);
            gui.Tesselation.setLargeStep(1);
            gui.Tesselation.setPos(Game.loadParam.patchTesselation);
            gui.Tesselation.setToolTipText("How smooth should curved surfaces be rendered");

            gui.Collision = env.addCheckBox(true, new recti((int)dim.Width - 400, 150, (int)dim.Width - 300, 166), gui.Window, -1, "Collision");
            gui.Collision.setToolTipText("Set collision on or off ( flythrough ). \nPress F7 on your Keyboard");
            gui.Visible_Map = env.addCheckBox(true, new recti((int)dim.Width - 300, 150, (int)dim.Width - 240, 166), gui.Window, -1, "Map");
            gui.Visible_Map.setToolTipText("Show or not show the static part the Level. \nPress F3 on your Keyboard");
            gui.Visible_Shader = env.addCheckBox(true, new recti((int)dim.Width - 240, 150, (int)dim.Width - 170, 166), gui.Window, -1, "Shader");
            gui.Visible_Shader.setToolTipText("Show or not show the Shader Nodes. \nPress F4 on your Keyboard");
            gui.Visible_Fog = env.addCheckBox(true, new recti((int)dim.Width - 170, 150, (int)dim.Width - 110, 166), gui.Window, -1, "Fog");
            gui.Visible_Fog.setToolTipText("Show or not show the Fog Nodes. \nPress F5 on your Keyboard");
            gui.Visible_Unresolved = env.addCheckBox(true, new recti((int)dim.Width - 110, 150, (int)dim.Width - 10, 166), gui.Window, -1, "Unresolved");
            gui.Visible_Unresolved.setToolTipText("Show the or not show the Nodes the Engine can't handle. \nPress F6 on your Keyboard");
            gui.Visible_Skydome = env.addCheckBox(true, new recti((int)dim.Width - 110, 180, (int)dim.Width - 10, 196), gui.Window, -1, "Skydome");
            gui.Visible_Skydome.setToolTipText("Show the or not show the Skydome.");

            //Respawn = env.addButton ( new recti( dim.Width - 260, 90, dim.Width - 10, 106 ), 0,-1, "Respawn" );

            env.addStaticText("Archives:", new recti(5, (int)dim.Height - 530, (int)dim.Width - 600, (int)dim.Height - 514), false, false, gui.Window, -1, false);

            gui.ArchiveAdd = env.addButton(new recti((int)dim.Width - 725, (int)dim.Height - 530, (int)dim.Width - 665, (int)dim.Height - 514), gui.Window, -1, "add");
            gui.ArchiveAdd.setToolTipText("Add an archive, usually packed zip-archives (*.pk3) to the Filesystem");
            gui.ArchiveRemove = env.addButton(new recti((int)dim.Width - 660, (int)dim.Height - 530, (int)dim.Width - 600, (int)dim.Height - 514), gui.Window, -1, "del");
            gui.ArchiveRemove.setToolTipText("Remove the selected archive from the FileSystem.");
            gui.ArchiveUp = env.addButton(new recti((int)dim.Width - 575, (int)dim.Height - 530, (int)dim.Width - 515, (int)dim.Height - 514), gui.Window, -1, "up");
            gui.ArchiveUp.setToolTipText("Arrange Archive Look-up Hirachy. Move the selected Archive up");
            gui.ArchiveDown = env.addButton(new recti((int)dim.Width - 510, (int)dim.Height - 530, (int)dim.Width - 440, (int)dim.Height - 514), gui.Window, -1, "down");
            gui.ArchiveDown.setToolTipText("Arrange Archive Look-up Hirachy. Move the selected Archive down");


            gui.ArchiveList = env.addTable(new recti(5, (int)dim.Height - 510, (int)dim.Width - 450, (int)dim.Height - 410), gui.Window);
            gui.ArchiveList.addColumn("Type", 0);
            gui.ArchiveList.addColumn("Real File Path", 1);
            gui.ArchiveList.setColumnWidth(0, 60);
            gui.ArchiveList.setColumnWidth(1, 284);
            gui.ArchiveList.setToolTipText("Show the attached Archives");


            env.addStaticText("Maps:", new recti(5, (int)dim.Height - 400, (int)dim.Width - 450, (int)dim.Height - 380), false, false, gui.Window, -1, false);
            gui.MapList = env.addListBox(new recti(5, (int)dim.Height - 380, (int)dim.Width - 450, (int)dim.Height - 40), gui.Window, -1, true);
            gui.MapList.setToolTipText("Show the current Maps in all Archives.\n Double-Click the Map to start the level");


            // create a visible Scene Tree
            env.addStaticText("Scenegraph:", new recti((int)dim.Width - 400, (int)dim.Height - 400, (int)dim.Width - 5, (int)dim.Height - 380), false, false, gui.Window, -1, false);
            gui.SceneTree = env.addTreeView(new recti((int)dim.Width - 400, (int)dim.Height - 380, (int)dim.Width - 5, (int)dim.Height - 40),
                                            gui.Window, -1, true, true, false);
            gui.SceneTree.setToolTipText("Show the current Scenegraph");
            gui.SceneTree.getRoot().clearChilds();
            addSceneTreeItem(Game.Device.getSceneManager().getRootSceneNode(), gui.SceneTree.getRoot());


            IGUIImageList imageList = env.createImageList(driver.getTexture("iconlist.png"),
                                                new dimension2di(32, 32), true);

            if (imageList != null)
            {
                gui.SceneTree.setImageList(imageList);
                //imageList.drop ();
            }


            // load the engine logo
            gui.Logo = env.addImage(driver.getTexture("irrlichtlogo3.png"), new position2di(5, 16), true, null);
            gui.Logo.setToolTipText("The great Irrlicht Engine");

            AddArchive("");
        }
        public void SetGUIActive(int command)
        {
            bool inputState = false;

            ICameraSceneNode camera = Game.Device.getSceneManager().getActiveCamera();

            switch (command)
            {
                case 0: Game.guiActive = 0; inputState = Game.guiActive == 0; break;
                case 1: Game.guiActive = 1; inputState = Game.guiActive == 0; ; break;
                case 2: Game.guiActive ^= 1; inputState = Game.guiActive == 0; break;
                case 3:
                    if (camera != null)
                        inputState = !camera.isInputReceiverEnabled();
                    break;
            }

            if (camera != null)
            {
                camera.setInputReceiverEnabled(inputState);
                Game.Device.getCursorControl().setVisible(!inputState);
            }

            if (gui.Window != null)
            {
                gui.Window.setVisible(Game.guiActive != 0);
            }

            if (Game.guiActive != 0 &&
                    gui.SceneTree != null && Game.Device.getGUIEnvironment().getFocus() != gui.SceneTree
                )
            {
                gui.SceneTree.getRoot().clearChilds();
                addSceneTreeItem(Game.Device.getSceneManager().getRootSceneNode(), gui.SceneTree.getRoot());
            }

            Game.Device.getGUIEnvironment().setFocus(Game.guiActive != 0 ? gui.Window : null);
        }

        public override bool OnEvent(SEvent eve)
        {
            if (eve.EventType == EEVENT_TYPE.EET_LOG_TEXT_EVENT)
            {
                return false;
            }

            if (Game.guiActive != 0 && eve.EventType == EEVENT_TYPE.EET_GUI_EVENT)
            {
                if (eve.GUIEvent.Caller.isSameObject(gui.MapList) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_LISTBOX_SELECTED_AGAIN)
                {
                    int selected = gui.MapList.getSelected();
                    if (selected >= 0)
                    {
                        string loadMap = gui.MapList.getListItem((uint)selected);
                        if (null == MapParent || loadMap != Game.CurrentMapName)
                        {
                            Console.Write("Loading map {0}\n", loadMap);
                            LoadMap(loadMap, 1);
                            if (0 == Game.loadParam.loadSkyShader)
                            {
                                AddSky(1, "skydome2");
                            }
                            CreatePlayers();
                            CreateGUI();
                            SetGUIActive(0);
                            return true;
                        }
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.ArchiveRemove) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
                {
                    Game.Device.getFileSystem().removeFileArchive((uint)gui.ArchiveList.getSelected());
                    Game.CurrentMapName = "";
                    AddArchive("");
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.ArchiveAdd) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
                {
                    if (null == gui.ArchiveFileOpen)
                    {
                        Game.Device.getFileSystem().setFileListSystem(EFileSystemType.FILESYSTEM_NATIVE);
                        gui.ArchiveFileOpen = Game.Device.getGUIEnvironment().addFileOpenDialog("Add Game Archive", false, gui.Window);
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.ArchiveFileOpen) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_FILE_SELECTED)
                {
                    AddArchive(gui.ArchiveFileOpen.getFileName());
                    gui.ArchiveFileOpen = null;
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.ArchiveFileOpen) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_DIRECTORY_SELECTED)
                {
                    AddArchive(gui.ArchiveFileOpen.getDirectoryName().ToString());
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.ArchiveFileOpen) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_FILE_CHOOSE_DIALOG_CANCELLED)
                {
                    gui.ArchiveFileOpen = null;
                }
                else if ((eve.GUIEvent.Caller.isSameObject(gui.ArchiveUp) || eve.GUIEvent.Caller == gui.ArchiveDown) &&
                            eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
                {
                    int rel = eve.GUIEvent.Caller == gui.ArchiveUp ? -1 : 1;
                    if (Game.Device.getFileSystem().moveFileArchive((uint)gui.ArchiveList.getSelected(), rel))
                    {
                        int newIndex = Q3Factory.clamp(gui.ArchiveList.getSelected() + rel, 0, gui.ArchiveList.getRowCount() - 1);
                        AddArchive("");
                        gui.ArchiveList.setSelected(newIndex);
                        Game.CurrentMapName = "";
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.VideoDriver) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_COMBO_BOX_CHANGED)
                {
                    Game.deviceParam.DriverType = (E_DRIVER_TYPE)gui.VideoDriver.getItemData((uint)gui.VideoDriver.getSelected());
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.VideoMode) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_COMBO_BOX_CHANGED)
                {
                    uint val = gui.VideoMode.getItemData((uint)gui.VideoMode.getSelected());
                    Game.deviceParam.WindowSize.Width = val >> 16;
                    Game.deviceParam.WindowSize.Height = val & 0xFFFF;
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.FullScreen) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    Game.deviceParam.Fullscreen = gui.FullScreen.isChecked();
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Bit32) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    Game.deviceParam.Bits = gui.Bit32.isChecked() ? (byte)32 : (byte)16;
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.MultiSample) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_SCROLL_BAR_CHANGED)
                {
                    Game.deviceParam.AntiAlias = (byte)gui.MultiSample.getPos();
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Tesselation) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_SCROLL_BAR_CHANGED)
                {
                    Game.loadParam.patchTesselation = gui.Tesselation.getPos();
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Gamma) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_SCROLL_BAR_CHANGED)
                {
                    Game.GammaValue = gui.Gamma.getPos() * 0.01f;
                    Game.Device.setGammaRamp(Game.GammaValue, Game.GammaValue, Game.GammaValue, 0.0f, 0.0f);
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.SetVideoMode) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
                {
                    Game.retVal = 2;
                    Game.Device.closeDevice();
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Window) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_ELEMENT_CLOSED)
                {
                    Game.Device.closeDevice();
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Collision) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    // set fly through active
                    Game.flyTroughState ^= 1;
                    Player[0].cam().setAnimateTarget(Game.flyTroughState == 0);

                    Console.Write("collision {0}\n", Game.flyTroughState == 0);
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Visible_Map) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    bool v = gui.Visible_Map.isChecked();

                    if (MapParent != null)
                    {
                        Console.Write("static node set visible {0}\n", v);
                        MapParent.setVisible(v);
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Visible_Shader) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    bool v = gui.Visible_Shader.isChecked();

                    if (ShaderParent != null)
                    {
                        Console.Write("shader node set visible {0}\n", v);
                        ShaderParent.setVisible(v);
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Visible_Skydome) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_CHECKBOX_CHANGED)
                {
                    if (SkyNode != null)
                    {
                        bool v = !SkyNode.isVisible();
                        Console.Write("skynode set visible {0}\n", v);
                        SkyNode.setVisible(v);
                    }
                }
                else if (eve.GUIEvent.Caller.isSameObject(gui.Respawn) && eve.GUIEvent.EventType == EGUI_EVENT_TYPE.EGET_BUTTON_CLICKED)
                {
                    Player[0].respawn();
                }

                return false;
            }

            // fire
            if ((eve.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT && eve.KeyInput.Key == EKEY_CODE.KEY_SPACE &&
                eve.KeyInput.PressedDown == false) ||
                (eve.EventType == EEVENT_TYPE.EET_MOUSE_INPUT_EVENT && eve.MouseInput.Event == EMOUSE_INPUT_EVENT.EMIE_LMOUSE_LEFT_UP)
                )
            {
                ICameraSceneNode camera = Game.Device.getSceneManager().getActiveCamera();
                if (camera != null && camera.isInputReceiverEnabled())
                {
                    useItem(Player[0]);
                }
            }

            // gui active
            if ((eve.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT && eve.KeyInput.Key == EKEY_CODE.KEY_F1 &&
                eve.KeyInput.PressedDown == false) ||
                (eve.EventType == EEVENT_TYPE.EET_MOUSE_INPUT_EVENT && eve.MouseInput.Event == EMOUSE_INPUT_EVENT.EMIE_RMOUSE_LEFT_UP)
                )
            {
                SetGUIActive(2);
            }

            // check if user presses the key
            if (eve.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT && eve.KeyInput.PressedDown == false)
            {
                // Escape toggles camera Input
                if (eve.KeyInput.Key == EKEY_CODE.KEY_ESCAPE)
                {
                    SetGUIActive(3);
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F11)
                {
                    //! screenshot are taken without gamma!
                    IImage image = Game.Device.getVideoDriver().createScreenShot();
                    if (image != null)
                    {
                        vector3df pos = new vector3df();
                        vector3df rot = new vector3df();
                        ICameraSceneNode cam = Game.Device.getSceneManager().getActiveCamera();
                        if (cam != null)
                        {
                            pos = cam.getPosition();
                            rot = cam.getRotation();
                        }

                        string[] dName = { "null", "software", "burning",
					"d3d8", "d3d9", "opengl" };

                        buf = string.Format("{0}_{1}_{2:f0}_{3:f0}_{4:f0}_{5:f0}_{6:f0}_{7:f0}.jpg",
                                dName[(int)Game.Device.getVideoDriver().getDriverType()],
                                Game.CurrentMapName,
                                pos.X, pos.Y, pos.Z,
                                rot.X, rot.Y, rot.Z
                                );
                        string filename = buf;
                        filename = filename.Replace('/', '_');
                        Console.Write("screenshot : {0}\n", filename);
                        Game.Device.getVideoDriver().writeImageToFile(image, filename, 100);
                        //image.drop();
                    }
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F9)
                {
                    int value = (int)E_DEBUG_SCENE_TYPE.EDS_OFF;

                    Game.debugState = (Game.debugState + 1) & 3;

                    switch (Game.debugState)
                    {
                        case 1:
                            value = (int)E_DEBUG_SCENE_TYPE.EDS_NORMALS |
                                (int)E_DEBUG_SCENE_TYPE.EDS_MESH_WIRE_OVERLAY |
                                (int)E_DEBUG_SCENE_TYPE.EDS_BBOX_ALL;
                            break;
                        case 2:
                            value = (int)E_DEBUG_SCENE_TYPE.EDS_NORMALS |
                                (int)E_DEBUG_SCENE_TYPE.EDS_MESH_WIRE_OVERLAY |
                                (int)E_DEBUG_SCENE_TYPE.EDS_SKELETON;
                            break;
                    }
                    /*
                                // set debug map data on/off
                                debugState = debugState == EDS_OFF ?
                                    EDS_NORMALS | EDS_MESH_WIRE_OVERLAY | EDS_BBOX_ALL:
                                    EDS_OFF;
                    */
                    if (ItemParent != null)
                    {
                        foreach (var it in ItemParent.getChildren())
                        {
                            it.setDebugDataVisible(value);
                        }
                    }

                    if (ShaderParent != null)
                    {
                        foreach (var it in ShaderParent.getChildren())
                        {
                            it.setDebugDataVisible(value);
                        }
                    }

                    if (UnresolvedParent != null)
                    {
                        foreach (var it in UnresolvedParent.getChildren())
                        {
                            it.setDebugDataVisible(value);
                        }
                    }

                    if (FogParent != null)
                    {
                        foreach (var it in FogParent.getChildren())
                        {
                            it.setDebugDataVisible(value);
                        }
                    }

                    if (SkyNode != null)
                    {
                        SkyNode.setDebugDataVisible(value);
                    }

                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F8)
                {
                    // set gravity on/off
                    Game.gravityState ^= 1;
                    Player[0].cam().setGravity(Q3Factory.getGravity(Game.gravityState != 0 ? "earth" : "none"));
                    Console.Write("gravity {0}\n", Game.gravityState != 0 ? "earth" : "none");
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F7)
                {
                    // set fly through active
                    Game.flyTroughState ^= 1;
                    Player[0].cam().setAnimateTarget(Game.flyTroughState == 0);
                    if (gui.Collision != null)
                        gui.Collision.setChecked(Game.flyTroughState == 0);

                    Console.Write("collision {0}\n", Game.flyTroughState == 0);
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F2)
                {
                    Player[0].respawn();
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F3)
                {
                    if (MapParent != null)
                    {
                        bool v = !MapParent.isVisible();
                        Console.Write("static node set visible {0}\n", v);
                        MapParent.setVisible(v);
                        if (gui.Visible_Map != null)
                            gui.Visible_Map.setChecked(v);
                    }
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F4)
                {
                    if (ShaderParent != null)
                    {
                        bool v = !ShaderParent.isVisible();
                        Console.Write("shader node set visible {0}\n", v);
                        ShaderParent.setVisible(v);
                        if (gui.Visible_Shader != null)
                            gui.Visible_Shader.setChecked(v);
                    }
                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F5)
                {
                    if (FogParent != null)
                    {
                        bool v = !FogParent.isVisible();
                        Console.Write("fog node set visible {0}\n", v);
                        FogParent.setVisible(v);
                        if (gui.Visible_Fog != null)
                            gui.Visible_Fog.setChecked(v);
                    }

                }
                else if (eve.KeyInput.Key == EKEY_CODE.KEY_F6)
                {
                    if (UnresolvedParent != null)
                    {
                        bool v = !UnresolvedParent.isVisible();
                        Console.Write("unresolved node set visible {0}\n", v);
                        UnresolvedParent.setVisible(v);
                        if (gui.Visible_Unresolved != null)
                            gui.Visible_Unresolved.setChecked(v);
                    }
                }
            }

            // check if user presses the key C ( for crouch)
            if (eve.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT && eve.KeyInput.Key == EKEY_CODE.KEY_KEY_C)
            {
                // crouch
                ISceneNodeAnimatorCollisionResponse anim = Player[0].cam();
                if (anim != null && 0 == Game.flyTroughState)
                {
                    if (false == eve.KeyInput.PressedDown)
                    {
                        // stand up
                        anim.setEllipsoidRadius(new vector3df(30, 45, 30));
                        anim.setEllipsoidTranslation(new vector3df(0, 40, 0));

                    }
                    else
                    {
                        // on your knees
                        anim.setEllipsoidRadius(new vector3df(30, 20, 30));
                        anim.setEllipsoidTranslation(new vector3df(0, 20, 0));
                    }
                    return true;
                }
            }
            return false;
        }



        GameData Game;

        IQ3LevelMesh Mesh;
        ISceneNode MapParent;
        ISceneNode ShaderParent;
        ISceneNode ItemParent;
        ISceneNode UnresolvedParent;
        ISceneNode BulletParent;
        ISceneNode FogParent;
        ISceneNode SkyNode;
        IMetaTriangleSelector Meta;

        string buf;

        Q3Player[] Player = new Q3Player[2] { new Q3Player(), new Q3Player() };

        class SParticleImpact
        {
            public uint when;
            public vector3df pos;
            public vector3df outVector;
        };
        List<SParticleImpact> Impacts = new List<SParticleImpact>();
        void useItem(Q3Player player)
        {
            ISceneManager smgr = Game.Device.getSceneManager();
            ICameraSceneNode camera = smgr.getActiveCamera();

            if (camera == null)
                return;

            SParticleImpact imp = new SParticleImpact();
            imp.when = 0;

            // get line of camera

            vector3df start = camera.getPosition();

            if (player.WeaponNode != null)
            {
                start.X += 0.0f;
                start.Y += 0.0f;
                start.Z += 0.0f;
            }

            vector3df end = (camera.getTarget() - start);
            end.normalize();
            start += end * 20.0f;

            end = start + (end * camera.getFarValue());

            triangle3df triangle = new triangle3df();
            line3df line = new line3df(start, end);

            // get intersection point with map
            ISceneNode hitNode;
            if (smgr.getSceneCollisionManager().getCollisionPoint(
                line, Meta, end, triangle, out hitNode))
            {
                // collides with wall
                vector3df out1 = triangle.getNormal();
                out1.setLength(0.03f);

                imp.when = 1;
                imp.outVector = out1;
                imp.pos = end;

                player.setAnim("pow");
                player.Anim[1].next += player.Anim[1].delta;
            }
            else
            {
                // doesnt collide with wall
                vector3df start2 = camera.getPosition();
                if (player.WeaponNode != null)
                {
                    //start.X += 10.0f;
                    //start.Y += -5.0f;
                    //start.Z += 1.0f;
                }

                vector3df end2 = (camera.getTarget() - start2);
                end2.normalize();
                start2 += end2 * 20.0f;
                end2 = start2 + (end2 * camera.getFarValue());
            }

            // create fire ball
            ISceneNode node = null;
            node = smgr.addBillboardSceneNode(BulletParent, new dimension2df(10, 10), start);

            node.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
            node.setMaterialTexture(0, Game.Device.getVideoDriver().getTexture("fireball.bmp"));
            node.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZWRITE_ENABLE, false);
            node.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_ADD_COLOR);

            float length = (float)(end - start).getLength();
            const float speed = 5.8f;
            uint time = (uint)(length / speed);

            ISceneNodeAnimator anim = null;

            // set flight line

            anim = smgr.createFlyStraightAnimator(start, end, time);
            node.addAnimator(anim);
            //anim.drop();

            buf = string.Format("bullet: {0} on {1:f1},{2:f1},{3:f1}",
                        imp.when != 0 ? "hit" : "nohit", end.X, end.Y, end.Z);
            node.setName(buf);


            anim = smgr.createDeleteAnimator(time);
            node.addAnimator(anim);
            //anim.drop();

            if (imp.when != 0)
            {
                // create impact note
                imp.when = (uint)(Game.Device.getTimer().getTime() +
                    (time + (int)((1.0f + Noiser.get()) * 250.0f)));
                Impacts.Add(imp);
            }

            // play sound
        }
        struct smokeLayer
        {
            public string texture;
            public float scale;
            public float minparticleSize;
            public float maxparticleSize;
            public float boxSize;
            public uint minParticle;
            public uint maxParticle;
            public uint fadeout;
            public uint lifetime;
        }
        void createParticleImpacts(uint now)
        {
            ISceneManager sm = Game.Device.getSceneManager();



            smokeLayer[] smoke = new smokeLayer[2]{
        new smokeLayer(){ 
            texture = "smoke2.jpg",
            scale = 0.4f,
            minparticleSize = 1.5f,
            maxparticleSize = 18.0f,
            boxSize = 20.0f,
            minParticle = 20,
            maxParticle = 50,
            fadeout = 2000,
            lifetime = 10000
        },
        new smokeLayer(){
            texture = "smoke3.jpg",
            scale = 0.2f,
            minparticleSize = 1.2f,
            maxparticleSize = 15.0f,
            boxSize = 20.0f,
            minParticle = 10,
            maxParticle = 30,
            fadeout = 1000,
            lifetime = 12000
       }
    };

            int i;
            uint g;
            uint factor = 1;
            for (g = 0; g != 2; ++g)
            {
                smoke[g].minParticle *= factor;
                smoke[g].maxParticle *= factor;
                smoke[g].lifetime *= factor;
                smoke[g].boxSize *= Noiser.get() * 0.5f;
            }

            for (i = 0; i < Impacts.Count; ++i)
            {
                if (now < Impacts[i].when)
                    continue;

                // create smoke particle system
                IParticleSystemSceneNode pas = null;

                for (g = 0; g != 2; ++g)
                {
                    pas = sm.addParticleSystemSceneNode(false, BulletParent, -1, Impacts[i].pos);

                    buf = string.Format("bullet impact smoke at {0}, {1}, {2}",
                        Impacts[i].pos.X, Impacts[i].pos.Y, Impacts[i].pos.Z);
                    pas.setName(buf);

                    // create a flat smoke
                    vector3df direction = Impacts[i].outVector;
                    direction *= smoke[g].scale;
                    IParticleEmitter em = pas.createBoxEmitter(
                        new aabbox3df(-4.0f, 0.0f, -4.0f, 20.0f, smoke[g].minparticleSize, 20.0f),
                        direction, smoke[g].minParticle, smoke[g].maxParticle,
                        new SColor(0, 0, 0, 0), new SColor(0, 128, 128, 128),
                        250, 4000, 60);

                    em.setMinStartSize(new dimension2df(smoke[g].minparticleSize, smoke[g].minparticleSize));
                    em.setMaxStartSize(new dimension2df(smoke[g].maxparticleSize, smoke[g].maxparticleSize));

                    pas.setEmitter(em);
                    //em.drop();

                    // particles get invisible
                    IParticleAffector paf = pas.createFadeOutParticleAffector(
                        new SColor(0, 0, 0, 0), smoke[g].fadeout);
                    pas.addAffector(paf);
                    //paf.drop();

                    // particle system life time
                    ISceneNodeAnimator anim = sm.createDeleteAnimator(smoke[g].lifetime);
                    pas.addAnimator(anim);
                    //anim.drop();

                    pas.setMaterialFlag(E_MATERIAL_FLAG.EMF_LIGHTING, false);
                    pas.setMaterialFlag(E_MATERIAL_FLAG.EMF_ZWRITE_ENABLE, false);
                    pas.setMaterialType(E_MATERIAL_TYPE.EMT_TRANSPARENT_VERTEX_ALPHA);
                    pas.setMaterialTexture(0, Game.Device.getVideoDriver().getTexture(smoke[g].texture));
                }


                // delete entry
                Impacts.RemoveAt(i);
                i--;
            }
        }

        unsafe void createTextures() 
        {
        	IVideoDriver driver = Game.Device.getVideoDriver();

            dimension2dui dim = new dimension2dui ( 64, 64 );

            ITexture texture;
            IImage image;
            uint i;
            uint x;
            uint y;
            uint * data;
            for ( i = 0; i != 8; ++i )
            {
	            image = driver.createImage ( ECOLOR_FORMAT.ECF_A8R8G8B8, dim);
	            data = (uint*) image.Lock();
	            for ( y = 0; y != dim.Height; ++y )
	            {
		            for ( x = 0; x != dim.Width; ++x )
		            {
			            data [x] = 0xFFFFFFFF;
		            }
		            data = (uint*) ( (byte*) data + image.getPitch() );
	            }
	            image.unlock();
	            buf = string.Format("smoke_{0:D2}", i );
	            texture = driver.addTexture( buf, image );
	            //image.drop ();
            }

            // fog
            for ( i = 0; i != 1; ++i )
            {
	            image = driver.createImage ( ECOLOR_FORMAT.ECF_A8R8G8B8, dim);
	            data = (uint*) image.Lock ();
	            for ( y = 0; y != dim.Height; ++y )
	            {
		            for ( x = 0; x != dim.Width; ++x )
		            {
			            data [x] = 0xFFFFFFFF;
		            }
		            data = (uint*) ( (byte*) data + image.getPitch() );
	            }
	            image.unlock();
	            buf = string.Format("fog_{0:D2}", i );
	            texture = driver.addTexture( buf, image );
	            //image.drop ();
            }
        }
        void addSceneTreeItem(ISceneNode parent, IGUITreeViewNode nodeParent)
        {
            IGUITreeViewNode node;
            string msg;

            int imageIndex;
            foreach (var it in parent.getChildren())
            {
                switch (it.getType())
                {
                    case (int)ESCENE_NODE_TYPE.ESNT_Q3SHADER_SCENE_NODE: imageIndex = 0; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_CAMERA: imageIndex = 1; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_EMPTY: imageIndex = 2; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_MESH: imageIndex = 3; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_OCT_TREE: imageIndex = 3; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_ANIMATED_MESH: imageIndex = 4; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_SKY_BOX: imageIndex = 5; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_BILLBOARD: imageIndex = 6; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_PARTICLE_SYSTEM: imageIndex = 7; break;
                    case (int)ESCENE_NODE_TYPE.ESNT_TEXT: imageIndex = 8; break;
                    default: imageIndex = -1; break;
                }

                if (imageIndex < 0)
                {
                    msg = string.Format("{0},{1}",
                        Game.Device.getSceneManager().getSceneNodeTypeName(it.getType()),
                        it.getName()
                        );
                }
                else
                {
                    msg = string.Format("{0}", it.getName());
                }

                node = nodeParent.addChildBack(msg, null, imageIndex);

                // Add all Animators
                foreach (ISceneNodeAnimator ait in it.getAnimators())
                {
                    imageIndex = -1;
                    msg = string.Format("{0}",
                        Game.Device.getSceneManager().getAnimatorTypeName(ait.getType())
                        );

                    switch (ait.getType())
                    {
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_FLY_CIRCLE:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_FLY_STRAIGHT:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_FOLLOW_SPLINE:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_ROTATION:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_TEXTURE:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_DELETION:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_COLLISION_RESPONSE:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_CAMERA_FPS:
                        case ESCENE_NODE_ANIMATOR_TYPE.ESNAT_CAMERA_MAYA:
                        default:
                            break;
                    }
                    node.addChildBack(msg, null, imageIndex);
                }

                addSceneTreeItem(it, node);
            }


        }

        GUI gui = new GUI();
        void dropMap() 
        {
            IVideoDriver driver = Game.Device.getVideoDriver();

            driver.removeAllHardwareBuffers();
            driver.removeAllTextures();

            Player[0].shutdown();


            Q3Factory.dropElement(ItemParent);
            Q3Factory.dropElement(ShaderParent);
            Q3Factory.dropElement(UnresolvedParent);
            Q3Factory.dropElement(FogParent);
            Q3Factory.dropElement(BulletParent);


            Impacts.Clear();

            if (Meta != null)
            {
                Meta = null;
            }

            Q3Factory.dropElement(MapParent);
            Q3Factory.dropElement(SkyNode);

            // clean out meshes, because textures are invalid
            // TODO: better texture handling;-)
            IMeshCache cache = Game.Device.getSceneManager().getMeshCache();
            cache.clear();
            Mesh = null;
        }
    }
}
