﻿using System;
using System.Collections.Generic;
using System.Text;
using IrrlichtNetSwig;

namespace SMeshHandling
{
    class Program
    {

        /*
        Much of this is code taken from some of the examples. We merely set
        up a mesh from a heightmap, light it with a moving light, and allow
        the user to navigate around it.
        */
        static void Main(string[] args)
        {
            // ask user for driver
            E_DRIVER_TYPE driverType = IrrlichtNet.driverChoiceConsole();
            if (driverType == E_DRIVER_TYPE.EDT_COUNT)
                return;

            MyEventReceiver receiver = new MyEventReceiver();
            IrrlichtDevice device = IrrlichtNet.createDevice(driverType,
                    new dimension2dui(800, 600), 32, false, false, false,
                    receiver);

            if (device == null)
                return;

            IVideoDriver driver = device.getVideoDriver();
            ISceneManager smgr = device.getSceneManager();
            device.setWindowCaption("Irrlicht Example for SMesh usage.");

            /*
            Create the custom mesh and initialize with a heightmap
            */
            TMesh mesh = new TMesh();
            HeightMap hm = new HeightMap(255, 255);
            hm.generate(eggbox);
            mesh.init(hm, 50.0f, grey, driver);

            // Add the mesh to the scene graph
            IMeshSceneNode meshnode = smgr.addMeshSceneNode(mesh.Mesh);
            meshnode.setMaterialFlag(E_MATERIAL_FLAG.EMF_BACK_FACE_CULLING, false);

            // light is just for nice effects
            ILightSceneNode node = smgr.addLightSceneNode(null, new vector3df(0, 100, 0),
                new SColorf(1.0f, 0.6f, 0.7f, 1.0f), 500.0f);
            if (node != null)
            {
                node.getLightData().Attenuation.set(0.0f, 1.0f / 500.0f, 0.0f);
                ISceneNodeAnimator anim = smgr.createFlyCircleAnimator(new vector3df(0, 150, 0), 250.0f);
                if (anim != null)
                {
                    node.addAnimator(anim);
                    //anim.drop();
                }
            }

            ICameraSceneNode camera = smgr.addCameraSceneNodeFPS();
            if (camera != null)
            {
                camera.setPosition(new vector3df(-20.0f, 150.0f, -20.0f));
                camera.setTarget(new vector3df(200.0f, -80.0f, 150.0f));
                camera.setFarValue(20000.0f);
            }

            /*
            Just a usual render loop with event handling. The custom mesh is
            a usual part of the scene graph which gets rendered by drawAll.
            */
            while (device.run())
            {
                if (!device.isWindowActive())
                {
                    device.sleep(100);
                    continue;
                }

                if (receiver.IsKeyDown(EKEY_CODE.KEY_KEY_W))
                {
                    meshnode.setMaterialFlag(E_MATERIAL_FLAG.EMF_WIREFRAME, !meshnode.getMaterial(0).Wireframe);
                }
                else if (receiver.IsKeyDown(EKEY_CODE.KEY_KEY_1))
                {
                    hm.generate(eggbox);
                    mesh.init(hm, 50.0f, grey, driver);
                }
                else if (receiver.IsKeyDown(EKEY_CODE.KEY_KEY_2))
                {
                    hm.generate(moresine);
                    mesh.init(hm, 50.0f, yellow, driver);
                }
                else if (receiver.IsKeyDown(EKEY_CODE.KEY_KEY_3))
                {
                    hm.generate(justexp);
                    mesh.init(hm, 50.0f, yellow, driver);
                }

                driver.beginScene(true, true, new SColor(0xff000000));
                smgr.drawAll();
                driver.endScene();
            }

            //device.drop();

            return;
        }


        /* Here comes a set of functions which can be used for coloring the nodes while
        creating the mesh. */

        // Greyscale, based on the height.
        static SColor grey(float x, float y, float z)
        {
            uint n = (uint)(255.0f * z);
            return new SColor(255, n, n, n);
        }

        // Interpolation between blue and white, with red added in one
        // direction and green in the other.
        static SColor yellow(float x, float y, float z)
        {
            return new SColor(255, 128 + (uint)(127.0f * x), 128 + (uint)(127.0f * y), 255);
        }

        // Pure white.
        static SColor white(float x, float y, float z) { return new SColor(255, 255, 255, 255); }


        // An interesting sample function :-)
        static float eggbox(short x, short y, float s)
        {
            float r = (float)(4.0f * Math.Sqrt((double)(x * x + y * y)) / s);
            float z = (float)(Math.Exp(-r * 2) * (Math.Cos(0.2f * x) + Math.Cos(0.2f * y)));
            return 0.25f + 0.25f * z;
        }

        // A rather dumb sine function :-/
        static float moresine(short x, short y, float s)
        {
            float xx = 0.3f * (float)x / s;
            float yy = 12 * y / s;
            float z = (float)(Math.Sin(xx * xx + yy) * Math.Sin(xx + yy * yy));
            return 0.25f + 0.25f * z;
        }

        // A simple function
        static float justexp(short x, short y, float s)
        {
            float xx = 6 * x / s;
            float yy = 6 * y / s;
            float z = (xx * xx + yy * yy);
            return (float)(0.3f * z * Math.Cos(xx * yy));
        }
    }

    /* This is the type of the functions which work out the colour. */
    delegate SColor colour_func(float x, float y, float z);

    /* The type of the functions which generate the heightmap. x and y
    range between -0.5 and 0.5, and s is the scale of the heightmap. */
    delegate float generate_func(short x, short y, float s);

    /* A simple class for representing heightmaps. Most of this should be obvious. */

    class HeightMap
    {
        ushort Width;
        ushort Height;
        float s;
        arrayFloat data = new arrayFloat();

        public HeightMap(ushort _w, ushort _h)
        {
            Width = _w;
            Height = _h;
            s = 0.0f;
            s = (float)Math.Sqrt((float)(Width * Width + Height * Height));
            data.set_used((uint)Width * Height);
        }

        // Fill the heightmap with values generated from f.
        public void generate(generate_func f)
        {
            int i = 0;
            for (ushort y = 0; y < Height; ++y)
                for (ushort x = 0; x < Width; ++x)
                    set(i++, calc(f, x, y));
        }

        public ushort height() { return Height; }
        public ushort width() { return Width; }

        public float calc(generate_func f, ushort x, ushort y)
        {
            float xx = (float)x - Width * 0.5f;
            float yy = (float)y - Height * 0.5f;
            return (float)f((short)xx, (short)yy, s);
        }

        // The height at (x, y) is at position y * Width + x.

        public void set(ushort x, ushort y, float z) { data[(uint)(y * Width + x)] = z; }
        public void set(int i, float z) { data[(uint)i] = z; }
        public float get(ushort x, ushort y) { return data[(uint)(y * Width + x)]; }

        /* The only difficult part. This considers the normal at (x, y) to
        be the cross product of the vectors between the adjacent points
        in the horizontal and vertical directions.

        s is a scaling factor, which is necessary if the height units are
        different from the coordinate units; for example, if your map has
        heights in metres and the coordinates are in units of a
        kilometer. */

        public vector3df getnormal(ushort x, ushort y, float s)
        {
            float zc = get(x, y);
            float zl, zr, zu, zd;

            if (x == 0)
            {
                zr = get((ushort)(x + 1), y);
                zl = zc + zc - zr;
            }
            else if (x == Width - 1)
            {
                zl = get((ushort)(x - 1), y);
                zr = zc + zc - zl;
            }
            else
            {
                zr = get((ushort)(x + 1), y);
                zl = get((ushort)(x - 1), y);
            }

            if (y == 0)
            {
                zd = get(x, (ushort)(y + 1));
                zu = zc + zc - zd;
            }
            else if (y == Height - 1)
            {
                zu = get(x, (ushort)(y - 1));
                zd = zc + zc - zu;
            }
            else
            {
                zd = get(x, (ushort)(y + 1));
                zu = get(x, (ushort)(y - 1));
            }

            return new vector3df(s * 2 * (zl - zr), 4, s * 2 * (zd - zu)).normalize();
        }
    };

    /* A class which generates a mesh from a heightmap. */
    class TMesh
    {
        ushort Width;
        ushort Height;
        float Scale;

        public SMesh Mesh;

        public TMesh()
        {
            Mesh = null;
            Width = 0;
            Height = 0;
            Scale = 1.0f;
            Mesh = new SMesh();
        }

        // Unless the heightmap is small, it won't all fit into a single
        // SMeshBuffer. This function chops it into pieces and generates a
        // buffer from each one.

        public void init(HeightMap hm, float scale, colour_func cf, IVideoDriver driver)
        {
            Scale = scale;

            int mp = (int)driver.getMaximalPrimitiveCount();
            Width = hm.width();
            Height = hm.height();

            int sw = mp / (6 * Height); // the width of each piece

            int i = 0;
            for (int y0 = 0; y0 < Height; y0 += sw)
            {
                ushort y1 = (ushort)(y0 + sw);
                if (y1 >= Height)
                    y1 = (ushort)(Height - 1); // the last one might be narrower
                addstrip(hm, cf, (ushort)y0, y1, i);
                ++i;
            }
            if (i < Mesh.getMeshBufferCount())
            {
                // clear the rest
                for (int j = i; j < Mesh.getMeshBufferCount(); ++j)
                {
                    //Mesh.getMeshBuffer(j)->drop();
                }
                Mesh.MeshBuffers.erase((uint)i, (int)Mesh.getMeshBufferCount() - i);
            }

            Mesh.recalculateBoundingBox();
        }

        // Generate a SMeshBuffer which represents all the vertices and
        // indices for values of y between y0 and y1, and add it to the
        // mesh.
        public void addstrip(HeightMap hm, colour_func cf, ushort y0, ushort y1, int bufNum)
        {
            SMeshBuffer buf = null;
            if (bufNum < Mesh.getMeshBufferCount())
            {
                buf = SMeshBuffer.cast(Mesh.getMeshBuffer((uint)bufNum));
            }
            else
            {
                // create new buffer
                buf = new SMeshBuffer();
                Mesh.addMeshBuffer(buf);
                // to simplify things we drop here but continue using buf
                //buf.drop();
            }
            buf.Vertices.set_used((uint)((1 + y1 - y0) * Width));

            int i = 0;
            for (ushort y = y0; y <= y1; ++y)
            {
                for (ushort x = 0; x < Width; ++x)
                {
                    float z = hm.get(x, y);
                    float xx = (float)x / (float)Width;
                    float yy = (float)y / (float)Height;

                    var vertices = buf.Vertices;
                    S3DVertex v = vertices[(uint)(i++)];
                    v.Pos.set(x, Scale * z, y);
                    v.Normal.set(hm.getnormal(x, y, Scale));
                    v.Color = cf(xx, yy, z);
                    v.TCoords.set(xx, yy);
                    //buf.Vertices[(uint)i] = v;
                }
            }

            buf.Indices.set_used((uint)(6 * (Width - 1) * (y1 - y0)));
            i = 0;
            for (ushort y = y0; y < y1; ++y)
            {
                for (ushort x = 0; x < Width - 1; ++x)
                {
                    ushort n = (ushort)((y - y0) * Width + x);
                    buf.Indices[(uint)i] = n;
                    buf.Indices[(uint)++i] = (ushort)(n + Height);
                    buf.Indices[(uint)++i] = (ushort)(n + Height + 1);
                    buf.Indices[(uint)++i] = (ushort)(n + Height + 1);
                    buf.Indices[(uint)++i] = (ushort)(n + 1);
                    buf.Indices[(uint)++i] = n;
                    ++i;
                }
            }

            buf.recalculateBoundingBox();
        }
    }

    /*
    Our event receiver implementation, taken from tutorial 4.
    */
    class MyEventReceiver : IEventReceiver
    {
        // This is the one method that we have to implement
        public override bool OnEvent(SEvent ev)
        {
            // Remember whether each key is down or up
            if (ev.EventType == EEVENT_TYPE.EET_KEY_INPUT_EVENT)
                KeyIsDown[(int)ev.KeyInput.Key] = ev.KeyInput.PressedDown;

            return false;
        }

        // This is used to check whether a key is being held down
        public virtual bool IsKeyDown(EKEY_CODE keyCode)
        {
            return KeyIsDown[(int)keyCode];
        }

        public MyEventReceiver()
        {
            for (int i = 0; i < (int)EKEY_CODE.KEY_KEY_CODES_COUNT; ++i)
                KeyIsDown[i] = false;
        }

        // We use this array to store the current state of each key
        bool[] KeyIsDown = new bool[(int)EKEY_CODE.KEY_KEY_CODES_COUNT];
    }

}
