//
// Inclusion of standard header file
//
#include <cerrno>
#include <cmath>
#include <cstdio>
#include <cstddef>
#include <ctime>
#include <string>
#include <string.h>
#include <sstream>
#include <stdexcept>

//
// Inclusion of system header file
//
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <utime.h>

//
// Inclusion of library header file
//
#include <boost/filesystem/path.hpp>

//
// Inclusion of local header file
//
#include "fatal_error.hpp"
#include "islibfunc.hpp"
#include "system_error.hpp"

namespace islib
  {
    //
    // Definition of function
    //
    void
      islib_link
      (
        boost::filesystem::path const &source_file_path,
        boost::filesystem::path const &destination_file_path
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &local_source_file_path,
                boost::filesystem::path const &local_destination_file_path
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::local_link, local_source_file_path.empty () );
                islib_throw_if_with_message ( std::invalid_argument, islib::local_link, local_destination_file_path.empty () );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( source_file_path, destination_file_path );

        int link_return_value = link ( source_file_path.string ().c_str (), destination_file_path.string ().c_str () );

        if ( link_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_link: link: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_localtime_r
      (
        std::time_t const * const time,
        struct std::tm * const time_information
      )
      {
        class local
          {
          public:
            local
              (
                std::time_t const * const local_time,
                struct std::tm * const local_time_information
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_localtime_r, !local_time );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_localtime_r, !local_time_information );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( time, time_information );

        struct tm * const localtime_r_return_value = localtime_r ( time, time_information );

        if ( localtime_r_return_value == 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_localtime_r: std::localtime_r: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_mkdir
      (
        boost::filesystem::path const &directory_path,
        mode_t mode
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &local_directory_path
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_mkdir, local_directory_path.empty () );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( directory_path );

        int mkdir_return_value = mkdir ( directory_path.string ().c_str (), mode );

        if ( mkdir_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_mkdir: mkdir: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_raise
      (
        int const signal_number
      )
      {
        class local
          {
          public:
            local
              (
                int const local_signal_number
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_raise, local_signal_number < 1 || _NSIG <= local_signal_number );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( signal_number );

        int const raise_return_value = raise ( signal_number );

        if ( raise_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_raise: raise: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_raise_fatal
      (
        int const signal_number
      )
      {
        class local
          {
          public:
            local
              (
                int const local_signal_number
              )
              {
                islib_throw_if_with_message ( fatal_error,  islib::islib_raise_fatal, local_signal_number < 1 || _NSIG <= local_signal_number );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( signal_number );

        int const raise_return_value = raise ( signal_number );

        if ( raise_return_value != 0 )
          islib_throw_with_message ( fatal_error, islib::islib_raise_fatal, "raise: fatal error." );
      }

    //
    // Definition of function
    //
    void
      islib_sigaction
      (
        int const signal_number,
        struct sigaction const * const signal_action,
        struct sigaction * const old_signal_action
      )
      {
        class local
          {
          public:
            local
              (
                int const local_signal_number,
                struct sigaction const * const local_signal_action
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_sigaction, local_signal_number < 1 || _NSIG <= local_signal_number );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_sigaction, !local_signal_action );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( signal_number, signal_action );

        int const sigaction_return_value = sigaction ( signal_number, signal_action, old_signal_action );

        if ( sigaction_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_sigaction: sigaction: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_sigaction_fatal
      (
        int const signal_number,
        struct sigaction const * const signal_action,
        struct sigaction * const old_signal_action
      )
      {
        class local
          {
          public:
            local
              (
                int const local_signal_number,
                struct sigaction const * const local_signal_action
              )
              {
                islib_throw_if_with_message ( fatal_error, islib::islib_sigaction_fatal, local_signal_number < 1 || _NSIG <= local_signal_number );
                islib_throw_if_with_message ( fatal_error, islib::islib_sigaction_fatal, !local_signal_action );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( signal_number, signal_action );

        int const sigaction_return_value = sigaction ( signal_number, signal_action, old_signal_action );

        if ( sigaction_return_value != 0 )
          islib_throw_with_message ( fatal_error, islib::islib_sigaction_fatal, "sigaction: fatal error." );
      }

    //
    // Definition of function
    //
    void
      islib_sigfillset
      (
        sigset_t * const signal_set
      )
      {
        class local
          {
          public:
            local
              (
                sigset_t * const local_signal_set
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_sigfillset, !local_signal_set );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( signal_set );

        int const sigfillset_return_value = sigfillset ( signal_set );

        if ( sigfillset_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_sigfillset: sigfillset: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_sigfillset_fatal
      (
        sigset_t * const signal_set
      )
      {
        class local
          {
          public:
            local
              (
                sigset_t * const local_signal_set
              )
              {
                islib_throw_if_with_message ( fatal_error, islib::islib_sigfillset_fatal, !local_signal_set );
              }

            ~local
              ()
              throw
                ()
              {
              }
          }
        a_local ( signal_set );

        int const sigfillset_return_value = sigfillset ( signal_set );

        if ( sigfillset_return_value != 0 )
          islib_throw_with_message ( fatal_error, islib::islib_sigfillset_fatal, "sigfillset: fatal error." );
      }

    //
    // Definition of function
    //
    void
      islib_sigprocmask
      (
        int const mode,
        sigset_t const * const signal_set,
        sigset_t * const old_signal_set
      )
      {
        class local
          {
          public:
            local
              (
                int const local_mode,
                sigset_t const * const local_signal_set
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_sigprocmask, local_mode != SIG_BLOCK && local_mode != SIG_UNBLOCK && local_mode != SIG_SETMASK );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_sigprocmask, !local_signal_set );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( mode, signal_set );

        int const sigprocmask_return_value = sigprocmask ( mode, signal_set, old_signal_set );

        if ( sigprocmask_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_sigprocmask: sigprocmask: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_sigprocmask_fatal
      (
        int const mode,
        sigset_t const * const signal_set,
        sigset_t * const old_signal_set
      )
      {
        class local
          {
          public:
            local
              (
                int const local_mode,
                sigset_t const * const local_signal_set
              )
              {
                islib_throw_if_with_message ( fatal_error, islib::islib_sigprocmask_fatal, local_mode != SIG_BLOCK && local_mode != SIG_UNBLOCK && local_mode != SIG_SETMASK );
                islib_throw_if_with_message ( fatal_error, islib::islib_sigprocmask_fatal, !local_signal_set );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( mode, signal_set );

        int const sigprocmask_return_value = sigprocmask ( mode, signal_set, old_signal_set );

        if ( sigprocmask_return_value != 0 )
          islib_throw_with_message ( fatal_error, islib::islib_sigprocmask_fatal, "sigprocmask: fatal error." );
      }

    //
    // Definition of function
    //
    void
      islib_stat
      (
        boost::filesystem::path const &path,
        struct stat * const status_information
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &local_path,
                struct stat * const local_status_information
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_stat, local_path.empty () );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_stat, !local_status_information );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( path, status_information );

        int const stat_return_value = stat ( path.string ().c_str (), status_information );

        if ( stat_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_stat: stat: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    std::time_t
      islib_time
      (
      )
      {
        std::time_t time;

        std::time_t time_return_value = std::time ( &time );

        if ( time_return_value == -1 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_time: std::time: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }

        return time;
      }

    //
    // Definition of function
    //
    void
      islib_unlink
      (
        boost::filesystem::path const &file_path
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &local_file_path
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_unlink, local_file_path.empty () );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( file_path );

        int unlink_return_value = unlink ( file_path.string ().c_str () );

        if ( unlink_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_unlink: unlink: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_utime
      (
        boost::filesystem::path const &path,
        struct utimbuf const * const utime_information
      )
      {
        class local
          {
          public:
            local
              (
                boost::filesystem::path const &local_path,
                struct utimbuf const * const local_utime_information
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_utime, local_path.empty () );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_utime, !local_utime_information );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( path, utime_information );

        int utime_return_value = utime ( path.string ().c_str (), utime_information );

        if ( utime_return_value != 0 )
          {
            int const saved_errno = errno;
            char buffer [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_utime: utime: " << strerror_r ( saved_errno, buffer, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_write
      (
        int const file_descriptor,
        void const *buffer,
        std::size_t const number_of_bytes
      )
      {
        class local
          {
          public:
            local
              (
                int const local_file_descriptor,
                void const *local_buffer
              )
              {
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_write, local_file_descriptor < 0 );
                islib_throw_if_with_message ( std::invalid_argument, islib::islib_write, !local_buffer );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( file_descriptor, buffer );

        ssize_t const write_return_value = write ( file_descriptor, buffer, number_of_bytes );

        if ( write_return_value == -1 )
          {
            int const saved_errno = errno;
            char buffer_ [ BUFSIZ ];
            std::stringstream system_error_message_stream;

            system_error_message_stream << "islib::islib_write: write: " << strerror_r ( saved_errno, buffer_, BUFSIZ );

            islib_throw ( system_error ( system_error_message_stream.str () ) );
          }
      }

    //
    // Definition of function
    //
    void
      islib_write_assert
      (
        int const file_descriptor,
        void const *buffer,
        std::size_t const number_of_bytes
      )
      {
        class local
          {
          public:
            local
              (
                int const local_file_descriptor,
                void const *local_buffer
              )
              {
                islib_throw_if_with_message ( fatal_error, islib::islib_write_assert, local_file_descriptor < 0 );
                islib_throw_if_with_message ( fatal_error, islib::islib_write_assert, !local_buffer );
              }

            ~local
              (
              )
              throw
              (
              )
              {
              }
          }
        a_local ( file_descriptor, buffer );

        ssize_t const write_return_value = write ( file_descriptor, buffer, number_of_bytes );

        islib_assert_with_message ( write_return_value != -1, islib::islib_write_assert );
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return std::fabs ( x - y ) <= permissible_error;
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      does_not_equal_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return std::fabs ( x - y ) > permissible_error;
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      is_less_than
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return y - x > permissible_error;
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      is_greater_than
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return x - y > permissible_error;
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      is_less_than_or_equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return x - y <= permissible_error;
      }

    //
    // Definition of specialized function template
    //
    template
      <
      >
      bool
      is_greater_than_or_equals_to
      <
        double,
        double
      >
      (
        double const &x,
        double const &y,
        double const &permissible_error
      )
      {
        return y - x <= permissible_error;
      }
  }

//
// End of file
//
