#ifndef DEFINE_ISLIB_ISLIBFUNC_HPP

#define DEFINE_ISLIB_ISLIBFUNC_HPP

//
// Inclusion of standard header file
//
#include <cassert>
#include <cstddef>
#include <ctime>
#include <iostream>
#include <stdexcept>
#include <string>
#include <utility>

//
// Inclusion of system header file
//
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <utime.h>

//
// Inclusion of library header file
//
#include <boost/filesystem/path.hpp>

//
// Definition of macro
//
#define islib_assert_with_message(expression,where) islib::islib_assert ( expression, #where ": `" #expression "' has failed." )

//
// Definition of macro
//
#define islib_throw_with_message(exception,where,what) islib::islib_throw ( exception ( #where ": " what ) )

//
// Definition of macro
//
#define islib_throw_if_with_message(exception,where,predicate) islib::islib_throw_if ( exception ( #where ": " #predicate ), predicate )

namespace islib
  {
    //
    // Declaration of function
    //
    void islib_link ( boost::filesystem::path const &, boost::filesystem::path const & );
    void islib_localtime_r ( std::time_t const *, struct std::tm * );
    void islib_mkdir ( boost::filesystem::path const &, mode_t );
    void islib_raise ( int );
    void islib_raise_fatal ( int );
    void islib_sigaction ( int, struct sigaction const *, struct sigaction * );
    void islib_sigaction_fatal ( int, struct sigaction const *, struct sigaction * );
    void islib_sigfillset ( sigset_t * );
    void islib_sigfillset_fatal ( sigset_t * );
    void islib_sigprocmask ( int, sigset_t const *, sigset_t * );
    void islib_sigprocmask_fatal ( int, sigset_t const *, sigset_t * );
    std::time_t islib_time ();
    void islib_stat ( boost::filesystem::path const &, struct stat * );
    void islib_unlink ( boost::filesystem::path const & );
    void islib_utime ( boost::filesystem::path const &, struct utimbuf const * );
    void islib_write ( int, void const *, std::size_t );
    void islib_write_assert ( int, void const *, std::size_t );

    //
    // Declaration of function template
    //
    template < typename Value_Type, typename Permissible_Error_Type > bool equals_to ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );
    template < typename Value_Type, typename Permissible_Error_Type > bool does_not_equal_to ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );
    template < typename Value_Type, typename Permissible_Error_Type > bool is_less_than ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );
    template < typename Value_Type, typename Permissible_Error_Type > bool is_greater_than ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );
    template < typename Value_Type, typename Permissible_Error_Type > bool is_less_than_or_equals_to ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );
    template < typename Value_Type, typename Permissible_Error_Type > bool is_greater_than_or_equals_to ( Value_Type const &, Value_Type const &, Permissible_Error_Type const & );

    //
    // Definition of inline function
    //
    inline
      void
      islib_assert
      (
        bool const expression,
        std::string const &cause
      )
      {

#ifndef NDEBUG

        if ( !expression )
          std::cerr << cause << std::endl;

#endif // NDEBUG

        assert ( expression );
      }

    //
    // Definition of function template
    //
    template
      <
        typename Exception_Type
      >
      void
      islib_throw
      (
        Exception_Type const &exception
      )
      {

#ifndef DO_NOT_USE_ISLIB_THROW

        throw exception;

#else // DO_NOT_USE_ISLIB_THROW

        islib_assert ( false, exception.what () );

#endif // DO_NOT_USE_ISLIB_THROW

      }

    //
    // Definition of function template
    //
    template
      <
        typename Exception_Type
      >
      void
      islib_throw_if
      (
        Exception_Type const &exception,
        bool const predicate
      )
      {

#ifndef DO_NOT_USE_ISLIB_THROW_IF

        if ( predicate )
          throw exception;

#else // DO_NOT_USE_ISLIB_THROW_IF
        
        if ( predicate )
          islib_assert ( false, exception.what () );

#endif // DO_NOT_USE_ISLIB_THROW_IF

      }

    //
    // Definition of function template
    //
    template
      <
        typename Container_Type,
        typename Value_Type
      >
      void
      insert
      (
        Container_Type &container,
        Value_Type const &value
      )
      {
        std::pair < typename Container_Type::iterator, bool > const insert_return_value = container.insert ( value );

        islib_throw_if_with_message ( std::logic_error, islib::insert, !insert_return_value.second );
      }
  }

#endif // DEFINE_ISLIB_ISLIBFUNC_HPP

//
// End of file
//
