/*
 * Copyright (C) 2002 Robert Ernst <robert.ernst@linux-solutions.at>
 *
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file LICENSE.GPL included in the
 * packaging of this file.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See COPYING for GPL licensing information.
 *
 */
#ifdef DESKTOP
#include <qapplication.h>
#else
#include <qpe/qpeapplication.h>
#endif
#include <qmessagebox.h>
#include <qwidget.h>
#include <qpainter.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include "PlayField.h"
#include "Engine.h"
#include "Game.h"
#include "Score.h"
#include "Move.h"

#include "config.h"
#include "TitleView.h"
#include "enemySelectView.h"
#include "GameoverView.h"
#include "GameoverVsView.h"
#include "StringTable.h"
#include <stdlib.h>
#include <unistd.h>


#define ANIMATION_NO    0
#define ANIMATION_FAST  1
#define ANIMATION_SLOW  2
#define ANIMATION_SLOW_DELTA  4   // 1 is show every chip
#define ANIMATION_FAST_DELTA 12

#define VS_BLACK_IMG() m_sprites->GetChipImg(3)
#define VS_WHITE_IMG() m_sprites->GetChipImg(23)
#define VS_BACK_IMG()  m_sprites->GetChipImg(0)

#define GET_DELTA() (m_animations == ANIMATION_SLOW ? ANIMATION_SLOW_DELTA : ANIMATION_FAST_DELTA )

#define GET_VSCOLOR(state) (state == PLAYER_MOVE ? Score::BLACK : Score::WHITE ) 

PlayField::PlayField( QWidget *pParent, QWidget *pMainWidget )
{
    int row;
    int col;

    m_pParent = pParent;
    m_pMainWidget = pMainWidget;
    m_sprites = new SpriteCollection();
    for (row = 0; row < rows; row++) {
	for (col = 0; col < cols; col++) {
	    int id = col + row * cols;
	    m_tile[id] = new PlayTile(id);
	    m_tile[id]->setState(empty);
	    m_tile[id]->setPosition(col * tilesize + PLAYFIELD_POS_X, 
				    row * tilesize + PLAYFIELD_POS_Y,
				    tilesize, tilesize);
	}
    }
    m_game_running = false;
    m_viewState = BOOT;
    m_strength = 4;
    m_animations = m_Config.GetSpeed();
    m_computerBegins = false;

    m_pViews = 0;
    m_bDrawSkip = false;
    m_bVsMode = false;

}

PlayField::~PlayField()
{

    int row;
    int col;

    for (row = 0; row < rows; row++) {
	for (col = 0; col < cols; col++) {
	    int id = col + row * cols;
	    if (m_tile[id]) {
		delete m_tile[id];
	    }
	}
    }
    if (m_sprites) {
	delete m_sprites;
    }

}

void PlayField::Click(int x, int y)
{
  int tilenum = cols*rows;
  int id;
  

  if( m_viewState == TITLE ){

    int ret = m_pViews->click( x,y );

    if( ret == 1 ){
      m_bVsMode = false;
      CTitleView *pView =(CTitleView *)m_pViews;
      m_animations = pView->GetAnimation();
      m_Config.SetSpeed(m_animations);
      ChangeView(ENEMY_SELECT);
    } else if( ret == 2 ){
       m_bVsMode = true;
       ChangeView(UNDERPLAY);
       start();
       m_state = PLAYER_MOVE;
    }

    return;
  }

  if( m_viewState == ENEMY_SELECT ){

    int id = m_pViews->click( x,y );

    if( id ){
      m_strength = m_EnemyMng.GetSelectedEnemy()->getStrong();
      ChangeView(UNDERPLAY);
      start();
      m_state = PLAYER_MOVE;
    }

    return;
  }

  if( m_viewState == GAME_OVER ){
    ChangeView(TITLE);
    return;
  }

  //check tile click
  for( id = 0; id < tilenum; id++ ){

    if( m_tile[id]->isClick(x,y) ){

      if(!m_bVsMode){
	tileClicked(id);
      } else {
	tileClickedVs(id);
      }

      break;
    }

  }
  
}
void PlayField::tileClicked(int id)
{

    if (!m_game_running || (m_state != PLAYER_MOVE && m_state != HINT)) {
	return;
    }
    if (m_state == HINT) {
	m_tile[id]->setState(empty);
	m_state = PLAYER_MOVE;
    } else if (m_state == PLAYER_MOVE) {
	Move move((id % cols) + 1, (id / cols) + 1, PlayerColor());
	if (game.MakeMove(move)) {
	    m_state = PLAYER_TURN;
	    turnTiles(move);
	    if (!game.MoveIsAtAllPossible()) {
		gameOver();
	    } else {
		computerMove();
	    }
	}
    }

}
void PlayField::tileClickedVs(int id)
{

    State oldstate;

    if (!m_game_running) {
	return;
    }

    if (m_state == HINT) {

	m_tile[id]->setState(empty);
	m_state = PLAYER_MOVE;

    } else if ((m_state == PLAYER_MOVE)||(m_state == COMPUTER_MOVE)) {

      int color = (m_state == PLAYER_MOVE) ? PlayerColor(): ComputerColor();
	Move move((id % cols) + 1, (id / cols) + 1, color);

	if (game.MakeMove(move)) {
	  oldstate = m_state; //restore old status
	  m_state = (m_state==PLAYER_MOVE) ? PLAYER_TURN : COMPUTER_TURN;
	  turnTiles(move);
	  if (!game.MoveIsAtAllPossible()) {
	    gameOver();
	  } else {
	    m_state = (m_state==PLAYER_TURN) ? COMPUTER_MOVE : PLAYER_MOVE;
	    if(!game.MoveIsPossible(GET_VSCOLOR(m_state))){
	      // pass
	      m_state = oldstate;
	    }
	  }
	}

    }

}
void PlayField::gameOver(void)
{
  m_state = GAME_OVER;
  ChangeView( RESULT );
}

int PlayField::PlayerColor(void)
{
    return m_computerBegins ? Score::WHITE : Score::BLACK;
}

int PlayField::ComputerColor(void)
{
    return m_computerBegins ? Score::BLACK : Score::WHITE;
}

void PlayField::computerMove(void)
{

    if (game.GetWhoseTurn() != ComputerColor() || !game.MoveIsPossible(ComputerColor())) {
	m_state = PLAYER_MOVE;
	return;
    }

    m_state = COMPUTER_MOVE;

    do {

	if (!game.MoveIsAtAllPossible()) {
	    gameOver();
	    return;
	}

	m_bDrawSkip = true;
	Move move = engine.ComputeMove(game);
	m_bDrawSkip = false;

	if (move.GetX() == -1) {
	    m_state = PLAYER_MOVE;
	    return;
	}

	m_state = COMPUTER_TURN;
	game.MakeMove(move);
	turnTiles(move);
    } while (!game.MoveIsPossible(PlayerColor()));

    m_state = PLAYER_MOVE;
    if (!game.MoveIsAtAllPossible()) {
	gameOver();
    }

}

void PlayField::drawTile(int col, int row, int state)
{

    int id = col + row * cols;
    
    if (id >= 0 && id < (rows * cols) && state >= empty && state <= red) {
	m_tile[id]->setState(state);
	m_pMainWidget->repaint(false);
    }

}
void PlayField::drawTile( QPainter *pPainter )
{

  int row;
  int col;

    for (row = 0; row < rows; row++) {
	for (col = 0; col < cols; col++) {

	    int id = col + row * cols;
	    int square =  game.GetSquare(col+1,row+1);
	    int state = m_tile[id]->state();

	    if(!state){
	      if( square == Score::BLACK){
		state = 1;
	      } else if( square ==  Score::WHITE ){
		state =24;
	      } else {
		state = 0;
	      }
	    }
	      

	 m_sprites->drawChip(*pPainter,m_tile[id]->getPosX(),
			     m_tile[id]->getPosY(),state);
	}
  }

}
void PlayField::turnTilesRow(int row, int col, int dy, int dx)
{

    row = row + dy;
    col = col + dx;

    while (row >= 0 && row < rows && col >= 0 && col < cols) {

	if (game.wasTurned(col + 1, row + 1)) {

	    int color = game.GetSquare(col + 1, row + 1);
	    int from;
	    int to;
	    int delta;
	    int i;

	    if (color == PlayerColor()) {
		from = red - 2;
		to = blue + 2;
		delta = -1*GET_DELTA();
		for (i = from; i > to; i = i + delta) {
		  drawTile(col, row, i);
		}

	    } else if (color == ComputerColor()) {
		from = blue + 2;
		to = red - 2;
		delta = GET_DELTA();
		for (i = from; i < to; i = i + delta) {
		  drawTile(col, row, i);
		}
	    } else {
		break;
	    }
	    drawTile(col, row, color == PlayerColor() ? blue : red);
	} else {
	    break;
	}

	row = row + dy;
	col = col + dx;
    }


}

void PlayField::turnTiles(Move &move)
{

    if (m_animations) {
	int row = move.GetY() - 1;
	int col = move.GetX() - 1;
	int dx;
	int dy;
	drawTile(col, row, move.GetPlayer() == PlayerColor() ? blue : red);
	for (dx = -1; dx <= 1; dx++) {
	    for (dy = -1; dy <= 1; dy++) {
		if (dx || dy) {
		    turnTilesRow(row, col, dy, dx);
		}
	    }
	}
    } else {
	updateBoard();
    }

}

void PlayField::updateBoard(void)
{

    int row;
    int col;

    for (row = 0; row < rows; row++) {

	for (col = 0; col < cols; col++) {
	    int id = col + row * cols;
	    int color = game.GetSquare(col + 1, row + 1);
	    if (color == PlayerColor()) {
		m_tile[id]->setState(blue);
	    } else if (color == ComputerColor()) {
		m_tile[id]->setState(red);
	    } else {
		m_tile[id]->setState(empty);
	    }
	}

    }

}

void PlayField::updateAnimations(int animations)
{
    if ((animations != 0) != m_animations) {
	m_animations = animations ? true : false;
    }
}

void PlayField::start( bool bReset, int state )
{
    m_game_running = false;
    engine.SetStrength(m_strength);
    if(bReset){
      game.Reset();
      m_state = m_computerBegins ? COMPUTER_MOVE : PLAYER_MOVE;
    } else {
      m_state = (enum State)state;
    }
    m_game_running = true;
    updateBoard();
    if ( m_state == COMPUTER_MOVE ) {
	computerMove();
    }
}

void PlayField::back(void)
{
#if 0
    if (!m_game_running || m_state != PLAYER_MOVE || game.GetMoveNumber() == 0) {
	return;
    }

    game.TakeBackMove();
    game.TakeBackMove();
    updateBoard();
#endif
}

void PlayField::hint(void)
{
#if 0
    if (!m_game_running || m_state != PLAYER_MOVE) {
	return;
    }

    Move move = engine.ComputeMove(game);
    int row = move.GetY() - 1;
    int col = move.GetX() - 1;
    if (row >= 0 && col >= 0) {
	int id = col + row * cols;
	int i;
	m_state = HINT;
	repaint(0, 0, width(), 22);
	for (i = 0; i < 240 && m_state == HINT; i++) {
	    if (m_tile[id]->state() == blue) {
		m_tile[id]->setState(empty);
	    } else {
		m_tile[id]->setState(blue);
	    }
	    m_tile[id]->repaint(false);
	    for (i = 0; i < 5; i++) {
		usleep(50000);
		qApp->processEvents();
	    }
	}
    }
    m_state = PLAYER_MOVE;
    repaint(0, 0, width(), 22);
#endif
}
void PlayField::draw( QPainter *pPainter )
{
  
  if(m_viewState == BOOT){

    m_sprites->DrawBootLogo(pPainter, SCREENW, SCREENH );
    ChangeView( UNDER_BOOT );

  } else if( m_viewState == UNDER_BOOT){

    m_sprites->DrawBootLogo(pPainter, SCREENW, SCREENH );
    sleep(2);
    ChangeView( TITLE );

  } else if( (m_viewState == TITLE) || (m_viewState == ENEMY_SELECT)){

    m_pViews->draw(pPainter, SCREENW, SCREENH);

  } else if( m_viewState == RESULT){

    m_pViews->draw(pPainter, SCREENW, SCREENH);

  } else {

    DrawFiled(pPainter);

  }

}
void PlayField::getChipCount( int *black, int *white )
{

    int row;
    int col;

    *black = 0;
    *white = 0;
    for (row = 0; row < rows; row++) {
	for (col = 0; col < cols; col++) {
	    int id = col + row * cols;
	    int state;
	    state = m_tile[id]->state();
	    if( state == blue ){
	      (*black)++;
	    } else if( state == red ){
	      (*white)++;
	    }
	}
    }
  
}
void PlayField::DrawFiled( QPainter *pPainter )
{
    if(!m_bDrawSkip){

      int player = game.GetScore(PlayerColor());
      int computer = game.GetScore(ComputerColor());
      m_sprites->setBackGround(pPainter, SCREENW, SCREENH );
      drawTile(pPainter);

      if(!m_bVsMode){
	// Single Mode
	m_sprites->setStrings(pPainter,
			    m_EnemyMng.GetSelectedEnemy()->getName(),
			    getStateStr(),player,computer);
	pPainter->drawPixmap( PLAYFIELD_ENEMY_POS_X, PLAYFIELD_ENEMY_POS_Y, 
			  *(m_EnemyMng.GetSelectedEnemy()->getImage()));
      } else {
	// VS Mode
	QString str;
	QPixmap *pImg;
	if( game.GetWhoseTurn() != ComputerColor()){
	  str = STRING_UTF8(ID_STRING_22);
	  pImg = VS_BLACK_IMG();
	} else {
	  str = STRING_UTF8(ID_STRING_23);
	  pImg = VS_WHITE_IMG();
	}
	m_sprites->setStrings(pPainter,str,"",player,computer);
	pPainter->drawPixmap( PLAYFIELD_ENEMY_POS_X, PLAYFIELD_ENEMY_POS_Y, *pImg );	
      }

    }

}
void PlayField::ChangeView( ViewState view )
{

  CCustomView *pOldView = m_pViews;
  
  if( view == TITLE ){
    m_pViews = (CCustomView*)(new CTitleView(m_pMainWidget,m_animations));
  } else if( view == ENEMY_SELECT ){

    if(loadLog()){
      view = UNDERPLAY;
      load();
    } else {
      m_pViews = (CCustomView*)(new CEnemySelectView(&m_EnemyMng,m_sprites->GetPanelImg()));
    }

  } else if( view == RESULT){
    if(!m_bVsMode){
      // Single Game
      game.clearLog();
      m_pViews = (CCustomView*)(new CGameOverView(m_EnemyMng.GetSelectedEnemy(),
						  game.GetScore(PlayerColor()),
						  game.GetScore(ComputerColor()))
				);
    } else {
      // Vs Game
      m_pViews = (CCustomView*)(new CGameOverVsView(VS_BLACK_IMG(),VS_WHITE_IMG(),VS_BACK_IMG(),
						  game.GetScore(PlayerColor()),
						  game.GetScore(ComputerColor()))
				);      
    }
    m_EnemyMng.SaveResult();
  }
  
  if( view == UNDERPLAY ){
    m_pViews = 0;
  }

  m_viewState = view;
  if( pOldView ){
    delete pOldView;
  }
}
QString PlayField::getStateStr()
{
    QString label("");

    switch (m_state) {
    case PLAYER_MOVE:
	label = STRING_UTF8(ID_STRING_15);
	break;
    case PLAYER_TURN:
    case COMPUTER_TURN:
	label = STRING_UTF8(ID_STRING_17);
	break;
    case COMPUTER_MOVE:
	label = STRING_UTF8(ID_STRING_16);
	break;
    case HINT:
	label = "Hint...";
	break;
    }

    return label;

}
void PlayField::quit()
{

  if( (m_viewState == UNDERPLAY) && (!m_bVsMode) 
      && (game.GetScore(Score::WHITE)+game.GetScore(Score::BLACK))!=4 ){
    int state;
    if( (m_state == PLAYER_MOVE ) || (m_state == COMPUTER_TURN) ){
      state = PLAYER_MOVE;
    } else {
      state = COMPUTER_MOVE;
    }
    game.Save( state, m_EnemyMng.GetSelectEnemyType() );
  }
}
bool PlayField::loadLog()
{
 
  bool ret = false;

  if( game.checkLog() ){

    QMessageBox mb( STRING_UTF8(ID_STRING_1200),
		    STRING_UTF8(ID_STRING_1201),
		    QMessageBox::NoIcon,
		    QMessageBox::Yes | QMessageBox::Default,
		    QMessageBox::No  | QMessageBox::Escape,
		    QMessageBox::NoButton , m_pMainWidget);

    if ( mb.exec() == QMessageBox::Yes ){
      ret = true;
    }

  }

  return ret;

}
void PlayField::load()
{
  int state,type;

  game.Load( &state, &type );
  m_EnemyMng.SelectEnemy( type );
  m_strength = m_EnemyMng.GetSelectedEnemy()->getStrong();
  start( false, state );

}
