/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */
package jp.sourceforge.javacpt.impl;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.logging.Logger;

import jp.sourceforge.javacpt.ComparativePerformanceTest;
import jp.sourceforge.javacpt.Printer;

/**
 * ComparativePerformanceTestImpl<br>
 * <br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class ComparativePerformanceTestImpl implements ComparativePerformanceTest
{

	/**
	 * test name
	 */
	private String testName;

	/**
	 * start millisec
	 */
	private long startMillis;

	/**
	 * end millisec
	 */
	private long endMillis;

	/**
	 * print result flag
	 */
	private boolean printResultFlag = true;

	/**
	 * printer
	 */
	private Printer printer = new PrinterImpl();

	/**
	 * default execute times
	 */
	public static final int DEFAULT_EXECUTE_TIMES = 1000;

	private int executeTimes = DEFAULT_EXECUTE_TIMES;

	private Object target = null;

	/**
	 * Default Constructor
	 */
	public ComparativePerformanceTestImpl()
	{
	}

	/**
	 * Constructor
	 * 
	 * @param testName
	 *            test name
	 */
	public ComparativePerformanceTestImpl(String testName)
	{
		this.testName = testName;
	}

	/**
	 * Record test start time(millisec)
	 */
	public void recordTestStart()
	{
		this.startMillis = System.currentTimeMillis();
	}

	/**
	 * Record test end time(millisec)
	 */
	public void recordTestEnd()
	{
		this.endMillis = System.currentTimeMillis();
	}

	/**
	 * Get result value
	 */
	public long getResult()
	{
		return this.endMillis - this.startMillis;
	}

	/**
	 * Set printResultFlag
	 */
	public ComparativePerformanceTest setPrintResultFlag(boolean printResultFlag)
	{
		this.printResultFlag = printResultFlag;
		return this;
	}

	/**
	 * Print result value
	 * 
	 * @see PrinterImpl
	 */
	public void printResult()
	{
		if (this.printResultFlag)
		{
			String result = this.testName + " : "
					+ String.valueOf(this.endMillis - this.startMillis) + " milsec.";
			printer.infoPrint(result);
		}
	}

	/**
	 * Print in ERROR level
	 * 
	 * @param str
	 *            String message to print
	 */
	public void errorPrint(String str)
	{
		printer.errorPrint(str);
	}

	/**
	 * Execute loop
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param target
	 *            target instance or class
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 */
	public void repeat(int executeTimes, Object target, Method executeMethod,
			Object... args)
	{
		if (target == null)
			throw new IllegalStateException(
					"Target instance or class to invoke method is required!");

		for (int i = 0; i < executeTimes; i++)
		{
			try
			{
				executeMethod.invoke(target, args);
			} catch (IllegalArgumentException e)
			{
				String argsStr = "args : ";
				for (Object each : args)
				{
					argsStr += each.toString() + ",";
				}
				errorPrint("exception catched! : " + String.valueOf(i + 1) + "/"
						+ executeTimes + "(" + argsStr + ")");
				e.printStackTrace();
				break;
			} catch (IllegalAccessException e)
			{
				errorPrint("exception catched! : " + String.valueOf(i + 1) + "/"
						+ executeTimes);
				e.printStackTrace();
				break;
			} catch (InvocationTargetException e)
			{
				errorPrint("exception catched! : " + String.valueOf(i + 1) + "/"
						+ executeTimes);
				e.printStackTrace();
				break;
			}
		}
	}

	/**
	 * Execute loop test.
	 * 
	 * @param executeTimes
	 *            loop execute times
	 * @param instance
	 *            target instance(if execute instance method)
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 * @return result millisec
	 */
	public long execute(int executeTimes, Object instance, Method executeMethod,
			Object... args)
	{
		recordTestStart();
		repeat(executeTimes, instance, executeMethod, args);
		recordTestEnd();
		printResult();
		return getResult();
	}

	/**
	 * Execute loop test.
	 * 
	 * @param executeMethod
	 *            execute method object
	 * @param args
	 *            method parameters
	 * @return result millisec
	 */
	public long execute(Method execuMethod, Object... args)
	{
		return execute(this.executeTimes, this.target, execuMethod, args);
	}

	/**
	 * Get Logger
	 * 
	 * @return logger
	 */
	public Logger getLogger()
	{
		return printer.getLogger();
	}

	/**
	 * Set Logger
	 * 
	 * @param logger
	 */
	public ComparativePerformanceTest setLogger(Logger logger)
	{
		printer.setLogger(logger);
		return this;
	}

	/**
	 * Get execute times
	 * 
	 * @return execute times
	 */
	public int getExecuteTimes()
	{
		return executeTimes;
	}

	/**
	 * Set execute times
	 * 
	 * @param executeTimes
	 *            execute times
	 */
	public ComparativePerformanceTest setExecuteTimes(int executeTimes)
	{
		this.executeTimes = executeTimes;
		return this;
	}

	/**
	 * Get target instance or class.
	 * 
	 * @return target instance
	 */
	public Object getTarget()
	{
		return target;
	}

	/**
	 * Set target instance or class.
	 * 
	 * @param targetInstance
	 *            target instance
	 */
	public ComparativePerformanceTest setTarget(Object target)
	{
		this.target = target;
		return this;
	}

	/**
	 * Get test name.
	 * 
	 * @return test name
	 */
	public String getTestName()
	{
		return testName;
	}

	/**
	 * Set test name
	 * 
	 * @param testName
	 *            test name
	 */
	public ComparativePerformanceTest setTestName(String testName)
	{
		this.testName = testName;
		return this;
	}

	/**
	 * Get Execute Method
	 * 
	 * @param targetInstance
	 * @param methodName
	 * @param methodParamTypes
	 * @return method object
	 */
	public Method getExecuteMethod(String methodName, Class<?>[] methodParamTypes)
			throws NoSuchMethodException, IllegalStateException
	{
		Method method = null;
		if (target == null)
		{
			throw new IllegalStateException(
					"Target instance or class to invoke method is required!");
		} else if (target instanceof Class)
		{
			// class method
			Class<?> targetClass = (Class<?>) target;
			method = targetClass.getMethod(methodName, methodParamTypes);
		} else
		{
			// instance method
			method = target.getClass().getMethod(methodName, methodParamTypes);
		}
		return method;
	}

}
