/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh;

import java.io.File;
import java.io.IOException;

/**
 * Simple file accessor interface.<br>
 * <br>
 * It's too much of a bother to treat file resources in Java.<br>
 * This module offers simple interface to treat file resources.<br>
 * <br>
 * It is able to access os file system resource(using string file path value) or
 * access ClassPathResource object.
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public interface SimpleFileAccessor
{

	/**
	 * Get dirname from path simply. <br>
	 * <br>
	 * It is compatible with UNIX command.<br>
	 * 
	 * @param path
	 *            target file path
	 * @return dirname string
	 */
	public String dirname(String path);

	/**
	 * Get dirname from path simply. <br>
	 * <br>
	 * It is compatible with UNIX command.<br>
	 * 
	 * @param file
	 *            target file object
	 * @return dirname string
	 */
	public String dirname(File file);

	/**
	 * Get basename from path simply. <br>
	 * <br>
	 * It is compatible with UNIX command.<br>
	 * 
	 * @param path
	 *            target path
	 * @return basename string
	 */
	public String basename(String path);

	/**
	 * Get basename from path simply. <br>
	 * <br>
	 * It is compatible with UNIX command.<br>
	 * 
	 * @param file
	 *            target file object
	 * @return basename string
	 */
	public String basename(File file);

	/**
	 * Copy file simply.<br>
	 * 
	 * @param src
	 *            src file path
	 * @param dest
	 *            dest file path
	 * @throws IOException
	 *             cannot copy file<br>
	 *             ex. source file does not exist
	 */
	public void copy(String src, String dest) throws IOException;

	/**
	 * Copy file simply. <br>
	 * <br>
	 * If destination file was in archive path, <br>
	 * it's impossible to copy it and this method throws IOException.<br>
	 * 
	 * @param src
	 *            {@link ClassPathResource} object
	 * @param dest
	 *            {@link ClassPathResource} object
	 * @throws IOException
	 *             cannot access resources
	 */
	public void copy(ClassPathResource src, ClassPathResource dest) throws IOException;

	/**
	 * Copy file simply. <br>
	 * 
	 * @param srcFile
	 *            src file
	 * @param dest
	 *            dest file path
	 * @throws IOException
	 *             cannot copy file<br>
	 *             ex. source file does not exist
	 */
	public void copy(File srcFile, String dest) throws IOException;

	/**
	 * Delete file simply.<br>
	 * 
	 * @param path
	 *            target file path
	 * @throws IOException
	 *             cannot delete file<br>
	 *             ex. file does not exist, locked
	 */
	public void delete(String path) throws IOException;

	/**
	 * Delete file simply. <br>
	 * <br>
	 * If target file was in archive path, <br>
	 * it's impossible to delete it and this method throws IOException.<br>
	 * 
	 * @param resourcePath
	 *            {@link ClassPathResource} object
	 * @throws IOException
	 *             cannot access resources
	 */
	public void delete(ClassPathResource resourcePath) throws IOException;

	/**
	 * Delete file simply.<br>
	 * 
	 * @param targetFile
	 *            target file object
	 * @throws IOException
	 *             cannot delete file<br>
	 *             ex. file does not exist, locked
	 */
	public void delete(File targetFile) throws IOException;

	/**
	 * Delete file simply if target file exists.<br>
	 * 
	 * @param path
	 *            target file path
	 * @throws IOException
	 *             cannot delete file<br>
	 *             ex. file does not exist, locked
	 */
	public void deleteIfExists(String path) throws IOException;

	/**
	 * Delete file simply if target file exists. <br>
	 * <br>
	 * If target file was in archive path, <br>
	 * it's impossible to delete it and this method throws IOException.<br>
	 * 
	 * @param resourcePath
	 *            {@link ClassPathResource} object
	 * @throws IOException
	 *             cannot access resources
	 */
	public void deleteIfExists(ClassPathResource resourcePath) throws IOException;

	/**
	 * Delete file simply if target file exists.<br>
	 * 
	 * @param targetFile
	 *            target file object
	 * @throws IOException
	 *             cannot delete file<br>
	 *             ex. file does not exist, locked
	 */
	public void deleteIfExists(File targetFile) throws IOException;

	/**
	 * Move file simply.<br>
	 * 
	 * @param fromPath
	 *            from file path
	 * @param toPath
	 *            to file path
	 * @throws IOException
	 *             cannot move file<br>
	 *             ex. from file not found, locked
	 */
	public void move(String fromPath, String toPath) throws IOException;

	/**
	 * Move file simply. <br>
	 * <br>
	 * If target file or destination file was in archive path,<br>
	 * it's impossible to move it and this method throws IOException.<br>
	 * 
	 * @param from
	 *            {@link ClassPathResource} from resource
	 * @param to
	 *            {@link ClassPathResource} to resource
	 * @throws IOException
	 *             cannot access resources
	 */
	public void move(ClassPathResource from, ClassPathResource to) throws IOException;

	/**
	 * Move file simply.<br>
	 * 
	 * @param targetFile
	 *            from file object
	 * @param toPath
	 *            to file path
	 * @throws IOException
	 *             cannot move file<br>
	 *             ex. from file not found, locked
	 */
	public void move(File targetFile, String toPath) throws IOException;

	/**
	 * Rename file simply.<br>
	 * 
	 * @param path
	 *            target file path
	 * @param newFilename
	 *            new file name
	 * @throws IOException
	 *             cannot rename file<br>
	 *             ex. from file not found, locked
	 */
	public void rename(String path, String newFilename) throws IOException;

	/**
	 * Rename file simply. <br>
	 * <br>
	 * If target file or destination file was in archive path,<br>
	 * it's impossible to rename it and this method throws IOException.<br>
	 * 
	 * @param resource
	 *            {@link ClassPathResource} resource
	 * @param newFilename
	 *            new file name
	 * @throws IOException
	 *             cannot access resources
	 */
	public void rename(ClassPathResource resource, String newFilename) throws IOException;

	/**
	 * Rename file simply.<br>
	 * 
	 * @param file
	 *            target file object
	 * @param newFilename
	 *            new file name
	 * @throws IOException
	 *             cannot rename file<br>
	 *             ex. from file not found, locked
	 */
	public void rename(File file, String newFilename) throws IOException;

	/**
	 * Touch file simply.<br>
	 * 
	 * @param path
	 *            target file path
	 * @throws IOException
	 *             cannot touch file<br>
	 */
	public void touch(String path) throws IOException;

	/**
	 * Touch file simply. <br>
	 * <br>
	 * If target file was in archive path,<br>
	 * it's impossible to touch it and this method throws IOException.<br>
	 * 
	 * @param resourcePath
	 *            {@link ClassPathResource} resource
	 * @throws IOException
	 *             cannot access resources
	 */
	public void touch(ClassPathResource resourcePath) throws IOException;

	/**
	 * Touch file simply.<br>
	 * 
	 * @param file
	 *            target file object
	 * @throws IOException
	 *             cannot touch file<br>
	 */
	public void touch(File file) throws IOException;

	/**
	 * Check file exist.<br>
	 * 
	 * @param path
	 *            target file path
	 * @return result boolean value
	 */
	public boolean fileExist(String path);

	/**
	 * Check file exist.<br>
	 * 
	 * @param resourcePath
	 *            {@link ClassPathResource} resource
	 * @return result boolean value
	 */
	public boolean fileExist(ClassPathResource resourcePath);

	/**
	 * Check file exist.<br>
	 * 
	 * @param file
	 *            target file object
	 * @return result boolean value
	 */
	public boolean fileExist(File file);

}
