/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * Simple file accessor interface.<br>
 * <br>
 * It's too much of a bother to treat file resources in Java.<br>
 * This module offers simple interface to treat file resources.<br>
 * <br>
 * It is able to access os file system resource(using string file path value) or
 * access ClassPathResource object.
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public interface SimpleFileWriter
{
	/**
	 * Write file with String array.
	 * 
	 * @param lines
	 *            String array contents
	 * @throws IOException
	 *             cannot write file
	 */
	public void write(String[] lines) throws IOException;

	/**
	 * Write file with String List.
	 * 
	 * @param lines
	 *            String List contents
	 * @throws IOException
	 *             cannot write file
	 */
	public void write(List<String> lines) throws IOException;

	/**
	 * Write file with String.
	 * 
	 * @param strContent
	 *            String contents
	 * @throws IOException
	 *             cannot write file
	 */
	public void write(String strContent) throws IOException;

	/**
	 * Write file with byte array.
	 * 
	 * @param content
	 *            byte array contents
	 * @throws IOException
	 *             cannot write file
	 */
	public void write(byte[] content) throws IOException;

	/**
	 * Get write target absolute file path.
	 * 
	 * @return write target file path
	 */
	public String getWriteTarget();

	/**
	 * Set write target.
	 * 
	 * @param path
	 *            write target file path
	 */
	public SimpleFileWriter setWriteTarget(String path);

	/**
	 * Set write target. <br>
	 * <br>
	 * If destination file was in archive path,<br>
	 * it's impossible to write and all write methods will throw IOException.<br>
	 * 
	 * @param resourcePath
	 *            write target resource path
	 */
	public SimpleFileWriter setWriteTarget(ClassPathResource resourcePath);

	/**
	 * Set write target.
	 * 
	 * @param file
	 *            write target file
	 */
	public SimpleFileWriter setWriteTarget(File file);

	/**
	 * Get line break.
	 * 
	 * @return line break
	 */
	public String getLineBreak();

	/**
	 * Set line break.
	 * 
	 * @param lineBreak
	 *            line break
	 */
	public SimpleFileWriter setLineBreak(String lineBreak);

	/**
	 * Get encoding.
	 * 
	 * @return encoding
	 */
	public String getEncoding();

	/**
	 * Set encoding.
	 * 
	 * @param encoding
	 *            specified encoding
	 */
	public SimpleFileWriter setEncoding(String encoding);
}
