/* 
 * Copyright 2009 Kazuhiro Sera. 
 import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;

import jp.sourceforge.simplefh.ClassPathResource;
import jp.sourceforge.simplefh.SimpleFileAccessor;
import jp.sourceforge.simplefh.SimpleFileWriter;
import jp.sourceforge.simplefh.constant.FixedValue;
import jp.sourceforge.simplefh.util.FileResourceUtil;
ONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */
package jp.sourceforge.simplefh.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;

import jp.sourceforge.simplefh.ClassPathResource;
import jp.sourceforge.simplefh.SimpleFileAccessor;
import jp.sourceforge.simplefh.SimpleFileReader;
import jp.sourceforge.simplefh.SimpleFileWriter;
import jp.sourceforge.simplefh.constant.FixedValue;
import jp.sourceforge.simplefh.util.FileResourceUtil;

/**
 * @see jp.sourceforge.simplefh.SimpleFileAccessor
 * @author Kazuhiro Sera
 * @version 1.0
 */
public class SimpleFileAccessorImpl implements SimpleFileAccessor
{
	SimpleFileWriter writer = new SimpleFileWriterImpl();

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#dirname(java.lang.String)
	 */
	public String dirname(String path)
	{
		String[] tmpArr = path.split(FixedValue.DIR_SEPARATOR_REGEXP);
		StringBuffer sb = new StringBuffer();
		int len = tmpArr.length - 1;
		for (int i = 0; i < len; i++)
		{
			sb.append(tmpArr[i]);
			sb.append(FixedValue.SLASH);
		}
		return sb.toString();
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#dirname(java.io.File)
	 */
	public String dirname(File file)
	{
		String path = file.getPath();
		return dirname(path);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#basename(java.lang.String)
	 */
	public String basename(String path)
	{
		String[] tmpArr = path.split(FixedValue.DIR_SEPARATOR_REGEXP);
		int basenameIdx = tmpArr.length - 1;
		return tmpArr[basenameIdx];
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#basename(java.io.File)
	 */
	public String basename(File file)
	{
		String path = file.getPath();
		return basename(path);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#copy(java.lang.String,
	 *      java.io.String)
	 */
	public void copy(String src, String dest) throws IOException
	{
		FileChannel srcChannel = new FileInputStream(src).getChannel();
		FileChannel destChannel = new FileOutputStream(dest).getChannel();
		srcChannel.transferTo(0, srcChannel.size(), destChannel);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#copy(java.io.File,
	 *      java.lang.String)
	 */
	public void copy(File srcFile, String dest) throws IOException
	{
		FileChannel srcChannel = new FileInputStream(srcFile).getChannel();
		FileChannel destChannel = new FileOutputStream(dest).getChannel();
		srcChannel.transferTo(0, srcChannel.size(), destChannel);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#copy(jp.sourceforge.simplefh
	 *      .ClassPathResource, jp.sourceforge.simplefh.ClassPathResource)
	 */
	public void copy(ClassPathResource src, ClassPathResource dest) throws IOException
	{
		String srcPath = src.getResourcePath();
		// stream read and write
		SimpleFileReader reader = new SimpleFileReaderImpl();
		srcPath = srcPath.replaceAll(FixedValue.BACK_SLASH_ESCAPED, FixedValue.SLASH);
		byte[] srcByteArr = reader.read2ByteArray(srcPath);
		SimpleFileWriter writer = new SimpleFileWriterImpl();
		try
		{
			writer.setWriteTarget(dest).write(srcByteArr);
		} catch (IOException e)
		{
			// cannot write destination file
			throw new IOException("Cannot copy resource file (destination file locked : "
			        + dest.getResourcePath() + ")");
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#delete(java.lang.String)
	 */
	public void delete(String filepath) throws IOException
	{
		File targetFile = new File(filepath);
		delete(targetFile);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#delete(jp.sourceforge.simplefh
	 *      .ClassPathResource)
	 */
	public void delete(ClassPathResource resourcePath) throws IOException
	{
		String path = resourcePath.getResourcePath();
		String abPath = FileResourceUtil.getFileSystemAbsolutePath(path);
		if (FileResourceUtil.isFileAccessAvailable(abPath))
		{
			// file access available
			delete(abPath);
		} else
		{
			// cannot remove because src file is included in jar file
			throw new IOException("Cannot delete resource file from archive file : "
			        + abPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#delete(java.io.File)
	 */
	public void delete(File targetFile) throws IOException
	{
		if (!targetFile.delete())
		{
			throw new IOException("Delete execute error : "
			        + targetFile.getAbsolutePath());
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#deleteIfExists(java.lang.String)
	 */
	public void deleteIfExists(String path) throws IOException
	{
		File targetFile = new File(path);
		deleteIfExists(targetFile);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#deleteIfExists(jp.sourceforge.simplefh.ClassPathResource)
	 */
	public void deleteIfExists(ClassPathResource resourcePath) throws IOException
	{
		String path = resourcePath.getResourcePath();
		String abPath = FileResourceUtil.getFileSystemAbsolutePath(path);
		if (FileResourceUtil.isFileAccessAvailable(abPath))
		{
			// file access available
			deleteIfExists(new File(abPath));
		} else
		{
			// cannot remove because src file is included in jar file
			throw new IOException("Cannot delete resource file from archive file : "
			        + abPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#deleteIfExists(java.io.File)
	 */
	public void deleteIfExists(File targetFile) throws IOException
	{
		targetFile.deleteOnExit();
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#move(java.lang.String,
	 *      java.lang.String)
	 */
	public void move(String fromPath, String toPath) throws IOException
	{
		File targetFile = new File(fromPath);
		try
		{
			move(targetFile, toPath);
		} catch (IOException e)
		{
			throw new IOException("Cannot move file from : " + fromPath + ", to : "
			        + toPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#move(jp.sourceforge.simplefh
	 *      .ClassPathResource, jp.sourceforge.simplefh.ClassPathResource)
	 */
	public void move(ClassPathResource from, ClassPathResource to) throws IOException
	{
		String fromPath = from.getResourcePath();
		String abFromPath = FileResourceUtil.getFileSystemAbsolutePath(fromPath);
		String rootPath = FileResourceUtil.getResourceRootPath(fromPath, abFromPath);
		String abToPath = rootPath + to.getResourcePath();
		if (FileResourceUtil.isFileAccessAvailable(abFromPath))
		{
			// file access available
			move(abFromPath, abToPath);
		} else
		{
			// cannot remove because src file is included in jar file
			throw new IOException("Cannot move resource file from archive file : "
			        + abFromPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#move(java.io.File,
	 *      java.lang.String)
	 */
	public void move(File targetFile, String toPath) throws IOException
	{
		// no src file
		if (!targetFile.exists())
			throw new IOException("Cannot find move target file : "
			        + targetFile.getAbsolutePath());

		// if dest file exists, try to overwrite it
		File dest = new File(toPath);
		if (dest.exists())
		{
			try
			{
				delete(dest);
			} catch (IOException e)
			{
				throw new IOException(
				        "The destination file has already existed and cannot remove it(maybe locked) : "
				                + toPath);
			}
		}
		// execute move
		if (!targetFile.renameTo(new File(toPath)))
		{
			throw new IOException("Cannot move file from : " + targetFile.getPath()
			        + ", to : " + toPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#rename(java.lang.String,
	 *      java.lang.String)
	 */
	public void rename(String path, String newFilename) throws IOException
	{
		File targetFile = new File(path);
		rename(targetFile, newFilename);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#rename(jp.sourceforge.simplefh
	 *      .ClassPathResource, java.lang.String)
	 */
	public void rename(ClassPathResource resource, String newFilename) throws IOException
	{
		String path = resource.getResourcePath();
		String abPath = FileResourceUtil.getFileSystemAbsolutePath(path);
		if (FileResourceUtil.isFileAccessAvailable(abPath))
		{
			// file access available
			rename(abPath, newFilename);
		} else
		{
			// cannot remove because src file is included in jar file
			throw new IOException("Cannot rename resource file from archive file : "
			        + abPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#rename(java.io.File,
	 *      java.lang.String)
	 */
	public void rename(File file, String newFilename) throws IOException
	{
		// no src file
		if (!file.exists())
			throw new IOException("Cannot find rename target file : "
			        + file.getAbsolutePath());

		File dest = new File(dirname(file) + newFilename);
		// if dest file existed, try to overwrite it
		if (dest.exists())
		{
			try
			{
				delete(dest);
			} catch (IOException e)
			{
				throw new IOException(
				        "The destination file has already existed and cannot remove it(maybe locked) : "
				                + dest.getAbsolutePath());
			}
		}

		// execute rename
		if (!file.renameTo(dest))
		{
			throw new IOException("Cannot rename file from : " + file.getPath()
			        + ", to : " + newFilename);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#touch(java.lang.String)
	 */
	public void touch(String path) throws IOException
	{
		writer.setWriteTarget(path);
		byte[] bytes = new byte[0];
		writer.write(bytes);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#touch(jp.sourceforge.simplefh
	 *      .ClassPathResource)
	 */
	public void touch(ClassPathResource resourcePath) throws IOException
	{
		String path = resourcePath.getResourcePath();
		String abPath = FileResourceUtil.getFileSystemAbsolutePath(path);
		try
		{
			touch(abPath);
		} catch (IOException e)
		{
			// cannot write destination file
			throw new IOException("Cannot touch file in archive file : " + abPath);
		}
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#touch(java.io.File)
	 */
	public void touch(File file) throws IOException
	{
		String path = file.getPath();
		writer.setWriteTarget(path);
		byte[] bytes = new byte[0];
		writer.write(bytes);
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#fileExist(java.lang.String)
	 */
	public boolean fileExist(String path)
	{
		File targetFile = new File(path);
		return targetFile.exists();
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#fileExist(java.io.File)
	 */
	public boolean fileExist(File file)
	{
		return file.exists();
	}

	/**
	 * @see jp.sourceforge.simplefh.SimpleFileAccessor#fileExist(jp.sourceforge.simplefh
	 *      .ClassPathResource)
	 */
	public boolean fileExist(ClassPathResource resourcePath)
	{
		String path = resourcePath.getResourcePath();
		String abPath = FileResourceUtil.getFileSystemAbsolutePath(path);
		return fileExist(abPath);
	}

}
