/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.simplefh;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * Simple file reader interface.<br>
 * <br>
 * It's too much of a bother to treat file resources in Java.<br>
 * This module offers simple interface to treat file resources.<br>
 * <br>
 * It is able to access os file system resource(using string file path value) or
 * access ClassPathResource object.
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public interface SimpleFileReader
{

	/**
	 * Read file content(convert to char array). <br>
	 * <br>
	 * This method go through the following steps to read target path.<br>
	 * (1) try to read the path in class path resource<br>
	 * (2) try to read from full path on os file system<br>
	 * 
	 * @param path
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(String path) throws IOException;

	/**
	 * Read resource content(convert to char array).
	 * 
	 * @param resourcePath
	 *            readable class path resource
	 * @return file content
	 * @throws IOException
	 */
	public char[] read2CharArray(ClassPathResource resourcePath) throws IOException;

	/**
	 * Read file content(convert to char array).
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(File file) throws IOException;

	/**
	 * Read file content(convert to byte array). <br>
	 * <br>
	 * This method go through the following steps to read target path.<br>
	 * (1) try to read the path in class path resource<br>
	 * (2) try to read from full path on os file system<br>
	 * 
	 * @param path
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(String path) throws IOException;

	/**
	 * Read resource content(convert to byte array).
	 * 
	 * @param resourcePath
	 *            readable class path resource
	 * @return file content
	 * @throws IOException
	 */
	public byte[] read2ByteArray(ClassPathResource resourcePath) throws IOException;

	/**
	 * Read file content(convert to byte array).
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(File file) throws IOException;

	/**
	 * Read file content(convert to String List). <br>
	 * <br>
	 * This method go through the following steps to read target path.<br>
	 * (1) try to read the path in class path resource<br>
	 * (2) try to read from full path on os file system<br>
	 * 
	 * @param path
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2StringLineList(String path) throws IOException;

	/**
	 * Read resource content(convert to String List).
	 * 
	 * @param resourcePath
	 *            readable class path resource
	 * @return file content
	 * @throws IOException
	 */
	public List<String> read2StringLineList(ClassPathResource resourcePath)
	        throws IOException;

	/**
	 * 
	 * Read file content(convert to String List).
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2StringLineList(File file) throws IOException;

	/**
	 * Read file content(convert to String array). <br>
	 * <br>
	 * This method go through the following steps to read target path.<br>
	 * (1) try to read the path in class path resource<br>
	 * (2) try to read from full path on os file system<br>
	 * 
	 * @param path
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2StringLineArray(String path) throws IOException;

	/**
	 * Read resource content(convert to String array).
	 * 
	 * @param resourcePath
	 *            readable class path resource
	 * @return file content
	 * @throws IOException
	 */
	public String[] read2StringLineArray(ClassPathResource resourcePath)
	        throws IOException;

	/**
	 * Read file content(convert to String array).
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2StringLineArray(File file) throws IOException;

	/**
	 * Get encoding to read.
	 * 
	 * @return encoding
	 */
	public String getEncodingToRead();

	/**
	 * Set encoding to read.
	 * 
	 * @param encodingToRead
	 *            encoding
	 * @return reader
	 */
	public SimpleFileReader setEncodingToRead(String encodingToRead);

	/**
	 * Get default array length to buffering.
	 * 
	 * @return buffering array length
	 */
	public Integer getDefaultArrayLength();

	/**
	 * Set default array length to buffering.
	 * 
	 * @param defaultArrayLength
	 *            default array length to buffering
	 * @return reader
	 */
	public SimpleFileReader setDefaultArrayLength(Integer defaultArrayLength);
}
