/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.ui.IFolderLayout;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveFactory;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.views.MediaPlayerView;
import ch.kuramo.javie.app.views.ProjectView;

public class Perspective implements IPerspectiveFactory {

	public static final String ID = "ch.kuramo.javie.app.perspective";

	public static final String ID_BOTTOM_FOLDER = "ch.kuramo.javie.app.perspective.bottomFolder";

	public static final String ID_LEFT_FOLDER = "ch.kuramo.javie.app.perspective.leftFolder";

	public static final String ID_CENTER_FOLDER = "ch.kuramo.javie.app.perspective.centerFolder";


	public void createInitialLayout(IPageLayout layout) {
		String editorArea = layout.getEditorArea();

		IFolderLayout bottomFolder = layout.createFolder(
				ID_BOTTOM_FOLDER, IPageLayout.BOTTOM, 0.5f, editorArea);

		IFolderLayout leftFolder = layout.createFolder(
				ID_LEFT_FOLDER, IPageLayout.LEFT, 0.3f, editorArea);

		IFolderLayout centerFolder = layout.createFolder(
				ID_CENTER_FOLDER, IPageLayout.LEFT, 0.95f, editorArea);

		leftFolder.addPlaceholder(ProjectView.ID);
		leftFolder.addPlaceholder("org.eclipse.ui.console.ConsoleView");
		bottomFolder.addPlaceholder(LayerCompositionView.ID + ":*");
		centerFolder.addPlaceholder(MediaPlayerView.ID + ":*");

		layout.setEditorAreaVisible(false);
	}

	public static void reset(IWorkbenchWindow window) {
		IPerspectiveRegistry registry = window.getWorkbench().getPerspectiveRegistry();
		IPerspectiveDescriptor defaultPerspective = registry.findPerspectiveWithId(ID);

		IWorkbenchPage page = window.getActivePage();
		page.setPerspective(defaultPerspective);
		page.resetPerspective();
		closeOther(page, defaultPerspective);
	}

	public static void openForProject(ProjectManager pm) {
		IWorkbenchWindow window = pm.getWorkbenchWindow();
		IPerspectiveRegistry registry = window.getWorkbench().getPerspectiveRegistry();

		String id = perspectiveId(pm);
		IPerspectiveDescriptor perspective = registry.findPerspectiveWithId(id);
		if (perspective != null) {
			IWorkbenchPage page = window.getActivePage();
			page.setPerspective(perspective);
			closeOther(page, perspective);
		}
	}

	public static void saveForProject(ProjectManager pm) {
		IWorkbenchWindow window = pm.getWorkbenchWindow();
		IPerspectiveRegistry registry = window.getWorkbench().getPerspectiveRegistry();

		String id = perspectiveId(pm);
		IPerspectiveDescriptor perspective = registry.findPerspectiveWithId(id);
		if (perspective == null) {
			IPerspectiveDescriptor defaultPerspective = registry.findPerspectiveWithId(ID);
			perspective = registry.clonePerspective(id, id, defaultPerspective);
		}

		IWorkbenchPage page = window.getActivePage();
		page.savePerspectiveAs(perspective);
		closeOther(page, perspective);
	}

	private static String perspectiveId(ProjectManager pm) {
		String path;
		try {
			path = pm.getFile().getCanonicalPath();
		} catch (IOException e) {
			path = pm.getFile().getAbsolutePath();
		}
		return ID + "." + pm.getProject().getId() + "." + path;
	}

	private static void closeOther(IWorkbenchPage page, IPerspectiveDescriptor perspective) {
		for (IPerspectiveDescriptor pd : page.getOpenPerspectives()) {
			if (pd != perspective) {
				page.closePerspective(pd, false, false);
			}
		}
	}

	public static void deleteOld(IWorkbench workbench, List<List<String>> retain) {
		Set<String> retainIds = new HashSet<String>();
		for (List<String> idAndPath : retain) {
			retainIds.add(ID + "." + idAndPath.get(0) + "." + idAndPath.get(1));
		}

		IPerspectiveRegistry registry = workbench.getPerspectiveRegistry();
		for (IPerspectiveDescriptor pd : registry.getPerspectives()) {
			if (!retainIds.contains(pd.getId())) {
				registry.deletePerspective(pd);
			}
		}
	}

}
