/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.AbstractOperation;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.swt.widgets.Shell;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;

public abstract class ProjectOperation extends AbstractOperation {

	private final ProjectManager _projectManager;

	protected final String _relation;

	protected boolean _noEffect;


	public ProjectOperation(ProjectManager projectManager, String label) {
		this(projectManager, label, null);
	}

	public ProjectOperation(ProjectManager projectManager, String label, String relation) {
		super(label);
		_projectManager = projectManager;
		_relation = relation;
	}

	boolean projectManagerMatches(ProjectManager projectManager) {
		return (_projectManager == projectManager);
	}

	public String getRelation() {
		return _relation;
	}

	public boolean isNoEffect() {
		return _noEffect;
	}

	protected Shell getShell() {
		return _projectManager.getWorkbenchWindow().getShell();
	}

	public IStatus execute(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
		return redo(monitor, info);
	}

	public IStatus redo(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
		IStatus status = executeOrRedo(monitor, info, _projectManager.getProject(), _projectManager);
		if (status.isOK()) {
			_projectManager.postShadowOperation(new Runnable() {
				public void run() {
					try {
						IStatus status = executeOrRedo(null, null, _projectManager.getShadow(), null);
						if (!status.isOK()) {
							// シャドウオペレーションだけ失敗することは無いはず。あったとしても回復不能なので例外を投げる。
							// (ShadowOperationRunnerは例外をキャッチしたらただちに終了する)
							throw new JavieRuntimeException("error running shadow operation: " + status);
						}
					} catch (ExecutionException e) {
						// 同上
						throw new JavieRuntimeException("error running shadow operation", e);
					}
				}
			});
		}
		return status;
	}

	public IStatus undo(IProgressMonitor monitor, IAdaptable info) throws ExecutionException {
		IStatus status = undo(monitor, info, _projectManager.getProject(), _projectManager);
		if (status.isOK()) {
			_projectManager.postShadowOperation(new Runnable() {
				public void run() {
					try {
						IStatus status = undo(null, null, _projectManager.getShadow(), null);
						if (!status.isOK()) {
							// シャドウオペレーションだけ失敗することは無いはず。あったとしても回復不能なので例外を投げる。
							// (ShadowOperationRunnerは例外をキャッチしたらただちに終了する)
							throw new JavieRuntimeException("error running shadow operation: " + status);
						}
					} catch (ExecutionException e) {
						// 同上
						throw new JavieRuntimeException("error running shadow operation", e);
					}
				}
			});
		}
		return status;
	}

	protected abstract IStatus executeOrRedo(
			IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException;

	protected abstract IStatus undo(
			IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException;

	public boolean tryMerging(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info) {
		if (_relation == null || !_relation.equals(operation.getRelation())) {
			return false;
		}

		if (!merge(operation, monitor, info, _projectManager.getProject(), _projectManager)) {
			return false;
		}

		_projectManager.postShadowOperation(new Runnable() {
			public void run() {
				if (!merge(null, null, null, _projectManager.getShadow(), null)) {
					// シャドウオペレーションだけマージ不可ということは無いはず。あったとしても回復不能なので例外を投げる。
					// (ShadowOperationRunnerは例外をキャッチしたらただちに終了する)
					throw new JavieRuntimeException("error running shadow operation: unable to merge");
				}
			}
		});
		return true;
	}

	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {
		return false;
	}

}
