/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class RenameItemOperation extends ProjectOperation {

	private final String _itemId;

	private final String _newName;

	private final String _oldName;

	private final Set<String> _layerIds = Util.newSet();


	public RenameItemOperation(ProjectManager projectManager, Item item, String newName) {
		super(projectManager, "名前の変更");

		newName = newName.replaceAll("[\\r\\n]+", "");
		_noEffect = (newName.length() == 0 || newName.equals(item.getName()));

		_itemId = item.getId();
		_newName = newName;
		_oldName = item.getName();

		// itemを参照しているItemLayerのうち、名前を変更していないレイヤーはアイテムの名前変更に合わせてレイヤー名も変更する。
		// TODO これをするかどうかをユーザーに確認するUIがあったほうが良いかも。
		//      また、部分一致などの場合でもレイヤー名を変更するオプションがあってもいい。
		for (Composition comp : projectManager.getProject().getCompositions()) {
			if (comp instanceof LayerComposition) {
				for (Layer layer : ((LayerComposition) comp).getLayers()) {
					if (layer instanceof ItemLayer) {
						ItemLayer itemLayer = (ItemLayer) layer;
						if (itemLayer.getItem() == item && itemLayer.getName().equals(_oldName)) {
							_layerIds.add(itemLayer.getId());
						}
					}
				}
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setName(_newName, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setName(_oldName, project, pm);
		return Status.OK_STATUS;
	}

	private void setName(String name, Project project, ProjectManager pm) {
		Item item = project.getItem(_itemId);
		item.setName(name);

		for (Composition comp : project.getCompositions()) {
			if (comp instanceof LayerComposition) {
				for (Layer layer : ((LayerComposition) comp).getLayers()) {
					if (_layerIds.contains(layer.getId())) {
						layer.setName(name);
					}
				}
			}
		}

		if (pm != null) {
			pm.fireItemNameChange(item);
		}
	}

}
