/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;
import java.util.SortedMap;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Scale;
import org.eclipse.swt.widgets.Slider;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.LayerSlipOperation;
import ch.kuramo.javie.app.project.ModifyLayerInPointOperation;
import ch.kuramo.javie.app.project.ModifyLayerOutPointOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ProjectOperation;
import ch.kuramo.javie.app.project.RemoveKeyframesOperation;
import ch.kuramo.javie.app.project.ShiftKeyframesOperation;
import ch.kuramo.javie.app.project.ShiftLayerTimesOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class TimelineManager {

	private static final int FRAME_WIDTH = 50;
	private static final int LEFT_MARGIN = 20;
	private static final int RIGHT_MARGIN = 40;


	private final ProjectManager projectManager;

	private final LayerComposition composition;

	private final LayerCompositionView view;

	private final Composite meter;

	private final Scale zoomScale;

	private final Slider scrollSlider;

	private final TreeColumn timelineColumn;


	private double zoom;

	private Time currentTime;

	private Time fromTime;

	private Time toTime;

	private boolean wrapMode = true;

	private final Map<Keyframe<?>, AnimatableValueElement> keyframeSelection = Util.newMap();

	private AbstractDragGestureEditor dragGestureEditor;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public TimelineManager(
			ProjectManager projectManager, LayerComposition composition, LayerCompositionView view,
			Composite meter, Scale zoomScale, Slider scrollSlider, Tree tree) {

		super();
		InjectorHolder.getInjector().injectMembers(this);

		this.projectManager = projectManager;
		this.composition = composition;
		this.view = view;
		this.meter = meter;
		this.zoomScale = zoomScale;
		this.scrollSlider = scrollSlider;
		this.timelineColumn = tree.getColumn(tree.getColumnCount() - 1);

		zoom = Double.NaN;

		meter.setBackground(ColorUtil.tableBackground());
		meter.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				drawMeter(e);
			}
		});

		meter.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				mouseDownOnMeter(e);
			}
		});

		// TODO これは暫定。
		zoomScale.setMinimum(0);
		zoomScale.setMaximum(10);
		zoomScale.setSelection(5);

		zoomScale.setIncrement(1);
		zoomScale.setPageIncrement(1);

		scrollSlider.setMinimum(0);
		scrollSlider.setIncrement(FRAME_WIDTH);

		update(Time.fromFrameNumber(0, composition.getFrameDuration()), true);
	}

	public void update(Time currentTime, boolean reveal) {
		// キーフレームなどをドラッグ中はrevealしない。
		// TODO しかし、ドラッグしたままタイムラインの領域の端まで来たらスクロールするようにはしたい。
		if (dragGestureEditor != null) {
			reveal = false;
		}

		int oldScroll = scrollSlider.getSelection();
		double oldZoom = this.zoom;

		Time compDuration = composition.getDuration();
		Time frameDuration = composition.getFrameDuration();

		// TODO zoomの値の計算方法はもっとよく考える必要がある。
		double zoom = Math.pow(2.0, zoomScale.getSelection() - zoomScale.getMaximum());

		int totalPixels = timeToPixels(compDuration, frameDuration, zoom) + RIGHT_MARGIN;
		int colWidth = timelineColumn.getWidth();

		scrollSlider.setMaximum(totalPixels);
		scrollSlider.setThumb(colWidth);
		scrollSlider.setPageIncrement(colWidth - LEFT_MARGIN - RIGHT_MARGIN);
		scrollSlider.setEnabled(totalPixels > colWidth);

		boolean wrap = wrapMode;

		if (!reveal && zoom != oldZoom && !Double.isNaN(oldZoom)) {
			Time oldFromTime = pixelsToTime(oldScroll, frameDuration, oldZoom);
			Time oldToTime = pixelsToTime(oldScroll + colWidth, frameDuration, oldZoom);

			if (currentTime.before(oldFromTime) || currentTime.after(oldToTime)) {
				Time tmp = oldFromTime.add(oldToTime);
				Time center = new Time(tmp.timeValue/2, tmp.timeScale);
				scrollSlider.setSelection(timeToPixels(center, frameDuration, zoom) - colWidth/2);

			} else {
				int offset = timeToPixels(currentTime, frameDuration, oldZoom) - timeToPixels(oldFromTime, frameDuration, oldZoom);
				scrollSlider.setSelection(timeToPixels(currentTime, frameDuration, zoom) - offset);

				reveal = true;
				wrap = false;
			}
		}

		if (reveal) {
			Time leftTime = pixelsToTime(scrollSlider.getSelection() + LEFT_MARGIN, frameDuration, zoom);
			Time rightTime = pixelsToTime(scrollSlider.getSelection() + colWidth - RIGHT_MARGIN
														- (int)(FRAME_WIDTH * zoom), frameDuration, zoom);

			if (currentTime.before(leftTime) || (wrap && currentTime.after(rightTime))) {
				scrollSlider.setSelection(timeToPixels(currentTime, frameDuration, zoom) - LEFT_MARGIN);

			} else if (!wrap && currentTime.after(rightTime)) {
				Time t = currentTime.add(leftTime).subtract(rightTime);
				scrollSlider.setSelection(timeToPixels(t, frameDuration, zoom) - LEFT_MARGIN);
			}
		}

		this.zoom = zoom;
		this.currentTime = currentTime;
		this.fromTime = pixelsToTime(scrollSlider.getSelection(), frameDuration, zoom);
		this.toTime = pixelsToTime(scrollSlider.getSelection() + colWidth, frameDuration, zoom);

		redraw();
	}

	public void redraw() {
		int x = timelineColumnX();
		Tree tree = timelineColumn.getParent();
		Rectangle clientArea = tree.getClientArea();
		tree.redraw(clientArea.x+x, clientArea.y, clientArea.width-x, clientArea.height, true);
		meter.redraw();
	}

	private int timeToPixels(Time time, Time frameDuration, double zoom) {
		return (int)Math.round(FRAME_WIDTH * zoom * time.toSecond() / frameDuration.toSecond()) + LEFT_MARGIN;
	}

	private Time pixelsToTime(int pixels, Time frameDuration, double zoom) {
		double frames = (pixels - LEFT_MARGIN) / (FRAME_WIDTH * zoom);
		return new Time(Math.round(frames * frameDuration.timeValue), frameDuration.timeScale);
	}

	private int timelineColumnX() {
		Tree tree = timelineColumn.getParent();
		int x = 0;
		for (int i = 0, n = tree.getColumnCount()-1; i < n; ++i) {
			TreeColumn column = tree.getColumn(i);
			x += column.getWidth();
		}
		return x;
	}

	public void drawCurrentFrameRegion(Event e) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int x1 = timeToPixels(currentTime, frameDuration, zoom) - scroll;
		int x2 = timeToPixels(currentTime.add(frameDuration), frameDuration, zoom) - scroll;

		e.gc.setBackground(e.display.getSystemColor(SWT.COLOR_BLUE));
		e.gc.setAlpha(48);
		e.gc.fillRectangle(e.x+x1, e.y, x2-x1, e.height);
		e.gc.setAlpha(255);
	}

	public void drawTimeIndicatorLine(PaintEvent e) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int left = timeToPixels(fromTime, frameDuration, zoom) - scroll;
		int x = timeToPixels(currentTime, frameDuration, zoom) - scroll;

		if (left <= x) {
			x += timelineColumnX();
			e.gc.setForeground(e.display.getSystemColor(SWT.COLOR_RED));
			e.gc.drawLine(x, e.y, x, e.y+e.height);
		}
	}

	public void drawColumnLeftLine(PaintEvent e) {
		int x = timelineColumnX() - 1;
		e.gc.setForeground(ColorUtil.tableRowLine());
		e.gc.drawLine(x, e.y, x, e.y+e.height);
	}

	private void drawMeter(PaintEvent e) {
		Display display = e.display;
		GC gc = e.gc;

		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();
		int h = meter.getBounds().height;

		gc.setForeground(display.getSystemColor(SWT.COLOR_BLACK));

		long step = (long)(1/zoom);
		long i0 = fromTime.toFrameNumber(frameDuration) - step;
		long i1 = toTime.toFrameNumber(frameDuration) + step;
		i0 = Math.max((i0/step)*step, 0);
		for (long i = i0; i <= i1; i += step) {
			Time t = Time.fromFrameNumber(i, frameDuration);
			int x = timeToPixels(t, frameDuration, zoom) - scroll;

			String text = TimeCode.toTimeCode(t, frameDuration, true, false);
			Point extent = gc.textExtent(text);

			gc.drawLine(x, extent.y, x, h);
			if ((i/step)%3 == 0) {
				gc.drawText(text, x-extent.x/2, 0);
			}
		}

		int x = timeToPixels(currentTime, frameDuration, zoom) - scroll;
		gc.setForeground(display.getSystemColor(SWT.COLOR_RED));
		gc.drawLine(x, 0, x, h);
	}

	private void mouseDownOnMeter(final MouseEvent e) {
		setWrapMode(false);
		updateView(e.x);

		final Control control = (Control) e.widget;
		final Display display = e.display;

		class MeterMouseTracker implements Runnable, Listener {
			private int x = e.x;
			private boolean end;

			public void run() {
				if (!end) {
					updateView(x);
					display.timerExec(30, this);
				}
			}

			public void handleEvent(Event e) {
				switch (e.type) {
					case SWT.MouseMove:
						if (!control.isDisposed()) {
							x = control.toControl(display.getCursorLocation()).x;
							updateView(x);
							break;
						}
						// fall through

					case SWT.MouseUp:
					case SWT.Deactivate:
						end = true;
						display.removeFilter(SWT.MouseMove, this);
						display.removeFilter(SWT.MouseUp, this);
						display.removeFilter(SWT.Deactivate, this);
						setWrapMode(true);
						break;
				}
			}
		}

		MeterMouseTracker tracker = new MeterMouseTracker();
		display.timerExec(30, tracker);
		display.addFilter(SWT.MouseMove, tracker);
		display.addFilter(SWT.MouseUp, tracker);
		display.addFilter(SWT.Deactivate, tracker);
	}

	private void updateView(int x) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		Time time = pixelsToTime(x + scroll, frameDuration, zoom);

		if (time.timeValue < 0) {
			time = new Time(0, time.timeScale);
		} else if (!time.before(composition.getDuration())) {
			time = composition.getDuration().subtract(frameDuration);
		}

		time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);

		if (!time.equals(currentTime)) {
			view.update(time);
		}
	}

	public void drawLayer(Event e, Layer layer) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int x = e.x;
		int y = e.y;
		int h = e.height-1;
		GC gc = e.gc;

		boolean hasSlipBar = false;

		if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					int x1 = x + timeToPixels(startTime, frameDuration, zoom) - scroll;
					int x2 = x + timeToPixels(endTime, frameDuration, zoom) - scroll;

					gc.setBackground(e.display.getSystemColor(SWT.COLOR_RED));
					gc.setAlpha(64);
					gc.fillRectangle(x1, y, x2-x1, h);

					hasSlipBar = true;
				}
			}
		}

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x1 = x + timeToPixels(inPoint, frameDuration, zoom) - scroll;
		int x2 = x + timeToPixels(outPoint, frameDuration, zoom) - scroll;

		if (!hasSlipBar) {
			gc.setBackground(e.display.getSystemColor(SWT.COLOR_RED));
			gc.setAlpha(64);
			gc.fillRectangle(x1, y, x2-x1, h);
		}
		gc.setBackground(e.display.getSystemColor(SWT.COLOR_DARK_RED));
		gc.setAlpha(128);
		gc.fillRectangle(x1, y, x2-x1, h);
		gc.setAlpha(255);


		drawCurrentFrameRegion(e);
	}

	public void updateCursor(MouseEvent e, Layer layer) {
		if (dragGestureEditor != null) {
			return;
		}

		Cursor cursor = null;

		Time frameDuration = composition.getFrameDuration();

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x = e.x - timelineColumnX() + scrollSlider.getSelection();
		int x1 = timeToPixels(inPoint, frameDuration, zoom);
		int x2 = timeToPixels(outPoint, frameDuration, zoom);

		if (x1-3 < x && x < x1+3) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_SIZEE);

		} else if (x2-3 < x && x < x2+3) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_SIZEW);

		} else if (x1 <= x && x < x2) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_HAND);

		} else if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					x1 = timeToPixels(startTime, frameDuration, zoom);
					x2 = timeToPixels(endTime, frameDuration, zoom);

					if (x1 <= x && x < x2) {
						cursor = e.display.getSystemCursor(SWT.CURSOR_HAND);
					}
				}
			}
		}

		Tree tree = timelineColumn.getParent();
		tree.setCursor(cursor);
	}

	public void mouseDown(MouseEvent e, Layer layer) {
		if (dragGestureEditor != null) {
			return;
		}

		Time frameDuration = composition.getFrameDuration();

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x = e.x - timelineColumnX() + scrollSlider.getSelection();
		int x1 = timeToPixels(inPoint, frameDuration, zoom);
		int x2 = timeToPixels(outPoint, frameDuration, zoom);

		if (x1-3 < x && x < x1+3) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureInPointShifter(e, pixelsToTime(x, frameDuration, zoom));

		} else if (x2-3 < x && x < x2+3) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureOutPointShifter(e, pixelsToTime(x, frameDuration, zoom));

		} else if (x1 <= x && x < x2) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureLayerTimeShifter(e, pixelsToTime(x, frameDuration, zoom));

		} else if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					x1 = timeToPixels(startTime, frameDuration, zoom);
					x2 = timeToPixels(endTime, frameDuration, zoom);

					if (x1 <= x && x < x2) {
						dragGestureEditor = new DragGestureSlipEditor(e, pixelsToTime(x, frameDuration, zoom));
					}
				}
			}
			if (dragGestureEditor == null) {
				clearKeyframeSelection();			
			}
		} else {
			clearKeyframeSelection();			
		}
	}

	public int drawKeyframes(Event e, AnimatableValue<?> avalue) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int ex = e.x;
		int y = e.y + e.height/2 - 1;
		GC gc = e.gc;

		for (Keyframe<?> kf : avalue.getKeyframes(fromTime, toTime)) {
			int x = ex + timeToPixels(kf.time, frameDuration, zoom) - scroll;
			Image icon = ImageUtil.getKeyframeIcon(kf.interpolation, keyframeSelection.containsKey(kf));
			if (icon != null) {
				gc.drawImage(icon, x-5, y-5);
			} else {
				gc.setForeground(e.display.getSystemColor(SWT.COLOR_BLACK));
				gc.drawRectangle(x-4, y-4, 8, 8);
			}
		}

		return y;
	}

	private Keyframe<?> findKeyframe(MouseEvent e, int yKeyframe, AnimatableValue<?> avalue) {
		if (e.y-4 > yKeyframe || e.y+4 < yKeyframe) {
			return null;
		}

		int x = e.x - timelineColumnX();
		int scroll = scrollSlider.getSelection();
		Time frameDuration = composition.getFrameDuration();

		Time time1 = pixelsToTime(x-4+scroll, frameDuration, zoom);
		Time time2 = pixelsToTime(x+5+scroll, frameDuration, zoom);	// headKeyframeMap は引数の時刻を「含まない」ので
																	// time2 は x+4 ではなく x+5 の時刻で計算する。 
		SortedMap<Time, ?> head = avalue.headKeyframeMap(time2);
		if (head.isEmpty()) {
			return null;
		}

		Keyframe<?> kf = (Keyframe<?>) head.get(head.lastKey());
		return time1.after(kf.time) ? null : kf;
	}

	public void mouseDown(
			MouseEvent e, int yKeyframe, AnimatableValueElement element, AnimatableValue<?> avalue) {

		Keyframe<?> kf = findKeyframe(e, yKeyframe, avalue);
		if (kf == null) {
			if (keyframeSelection.isEmpty()) return;
			keyframeSelection.clear();

		} else if (!keyframeSelection.containsKey(kf)) {
			if ((e.stateMask & SWT.SHIFT) != SWT.SHIFT) {
				keyframeSelection.clear();
			}
			keyframeSelection.put(kf, element);

		} else if (e.button == 1 && (e.stateMask & SWT.SHIFT) == SWT.SHIFT) {
			keyframeSelection.remove(kf);
		}

		redraw();


		if (kf != null && keyframeSelection.containsKey(kf)
				&& e.button == 1 && dragGestureEditor == null) {

			dragGestureEditor = new DragGestureKeyframeShifter(e, kf.time);
		}
	}

	public void mouseDown(MouseEvent e) {
		if (keyframeSelection.size() > 0 && e.x >= timelineColumnX()) {
			clearKeyframeSelection();
			redraw();
		}
	}

	public void clearKeyframeSelection() {
		keyframeSelection.clear();
	}

	public void addKeyframeSelection(Keyframe<?> keyframe, AnimatableValueElement element) {
		keyframeSelection.put(keyframe, element);
	}

	public boolean hasKeyframeSelection() {
		return keyframeSelection.size() > 0;
	}

	private Object[][] createSelectedKeyframeData() {
		Object[][] data = new Object[keyframeSelection.size()][];

		int i = 0;
		for (Map.Entry<Keyframe<?>, AnimatableValueElement> e : keyframeSelection.entrySet()) {
			Keyframe<?> kf = e.getKey();
			AnimatableValueElement element = e.getValue();

			if (element instanceof LayerAnimatableValueElement) {
				LayerAnimatableValueElement avalueElem = (LayerAnimatableValueElement) element;
				data[i++] = new Object[] { avalueElem.layer.getId(), -1, avalueElem.property, kf };
			} else {
				EffectAnimatableValueElement avalueElem = (EffectAnimatableValueElement) element;
				data[i++] = new Object[] { avalueElem.layer.getId(),
											avalueElem.layer.getEffects().indexOf(avalueElem.effect),
											avalueElem.descriptor.getName(), kf };
			}
		}

		return data;
	}

	private String[] createBaseAnimatableValues(Object[][] keyframeData) {
		Map<AnimatableValue<?>, AnimatableValue<?>> map = Util.newMap();
		AnimatableValue<?>[] baseAvalues = new AnimatableValue[keyframeData.length];

		for (int i = 0, n = keyframeData.length; i < n; ++i) {
			Layer layer = composition.getLayer((String) keyframeData[i][0]);
			Integer effectIndex = (Integer) keyframeData[i][1];
			String property = (String) keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			AnimatableValue<?> baseAvalue = map.get(avalue);
			if (baseAvalue == null) {
				String s = _encoder.encodeElement(avalue);
				try {
					baseAvalue = _decoder.decodeElement(s, avalue.getClass());
				} catch (ProjectDecodeException e) {
					throw new JavieRuntimeException(e);
				}
				map.put(avalue, baseAvalue);
				baseAvalues[i] = baseAvalue;
			}
			baseAvalue.removeKeyframe(kf.time);
		}

		String[] encodedBaseAvalues = new String[baseAvalues.length];
		for (int i = 0, n = baseAvalues.length; i < n; ++i) {
			if (baseAvalues[i] != null) {
				encodedBaseAvalues[i] = _encoder.encodeElement(baseAvalues[i]);
			}
		}
		return encodedBaseAvalues;
	}

	public void removeSelectedKeyframes() {
		projectManager.postOperation(new RemoveKeyframesOperation(
				projectManager, composition, createSelectedKeyframeData()));
	}

	public void setWrapMode(boolean wrapMode) {
		this.wrapMode = wrapMode;
	}

	public Time getCurrentTime() {
		return currentTime;
	}


	// TODO AnimatableValueElementDelegate.DragGestureEditor と基本的には同じ。できればうまく纏めたい。
	private abstract class AbstractDragGestureEditor {

		private final Time baseTime;

		private final String relation = Util.randomId();

		private final long downTime;

		private final int downX;

		private boolean dragDetected;

		private Time prevTime;


		private AbstractDragGestureEditor(MouseEvent event, Time baseTime) {
			Time frameDuration = composition.getFrameDuration();
			this.baseTime = prevTime = Time.fromFrameNumber(baseTime.toFrameNumber(frameDuration), frameDuration);

			Control control = (Control) event.widget;
			downTime = System.currentTimeMillis();
			downX = event.x;

			init(control);
		}

		private void init(final Control control) {
			final Display display = control.getDisplay();

			Listener listener = new Listener() {
				public void handleEvent(Event e) {
					switch (e.type) {
						case SWT.MouseMove:
							if (!control.isDisposed()) {
								int x = control.toControl(display.getCursorLocation()).x;

								if (!dragDetected) {
									dragDetected = (System.currentTimeMillis() - downTime > 100) && (Math.abs(x - downX) > 3);
								}

								if (dragDetected) {
									drag(x);
								}

								break;
							}
							// fall through

						case SWT.MouseUp:
						case SWT.Deactivate:
							dragGestureEditor = null;

							display.removeFilter(SWT.MouseMove, this);
							display.removeFilter(SWT.MouseUp, this);
							display.removeFilter(SWT.Deactivate, this);
							break;
					}
				}
			};

			display.addFilter(SWT.MouseMove, listener);
			display.addFilter(SWT.MouseUp, listener);
			display.addFilter(SWT.Deactivate, listener);
		}

		private void drag(int x) {
			Time frameDuration = composition.getFrameDuration();
			Time time = pixelsToTime(x-timelineColumnX()+scrollSlider.getSelection(), frameDuration, zoom);
			time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
			if (!time.equals(prevTime)) {
				projectManager.postOperation(createOperation(time, prevTime, baseTime, relation));
				prevTime = time;
			}
		}

		protected abstract ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation);
	}


	private class DragGestureInPointShifter extends AbstractDragGestureEditor {

		private final Object[][] layersAndInPoints;


		private DragGestureInPointShifter(MouseEvent event, Time baseTime) {
			super(event, baseTime);
			layersAndInPoints = getLayersAndInPoints();
		}

		private Object[][] getLayersAndInPoints() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getInPoint() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation) {
			return new ModifyLayerInPointOperation(
					projectManager, composition, layersAndInPoints, time.subtract(baseTime), relation);
		}

	}

	private class DragGestureOutPointShifter extends AbstractDragGestureEditor {

		private final Object[][] layersAndOutPoints;


		private DragGestureOutPointShifter(MouseEvent event, Time baseTime) {
			super(event, baseTime);
			layersAndOutPoints = getLayersAndOutPoints();
		}

		private Object[][] getLayersAndOutPoints() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getOutPoint() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation) {
			return new ModifyLayerOutPointOperation(
					projectManager, composition, layersAndOutPoints, time.subtract(baseTime), relation);
		}

	}

	private class DragGestureSlipEditor extends AbstractDragGestureEditor {

		private final Object[][] layersAndStartTimes;

		private final Object[][] selectedKeyframeData;

		private final String[] baseAnimatableValues;


		private DragGestureSlipEditor(MouseEvent event, Time baseTime) {
			super(event, baseTime);
			layersAndStartTimes = getLayersAndStartTimes();
			selectedKeyframeData = createSelectedKeyframeData();
			baseAnimatableValues = createBaseAnimatableValues(selectedKeyframeData);
		}

		private Object[][] getLayersAndStartTimes() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getStartTime() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation) {
			return new LayerSlipOperation(
					projectManager, composition,
					layersAndStartTimes, selectedKeyframeData, baseAnimatableValues,
					time.subtract(baseTime), relation);
		}

	}

	private class DragGestureLayerTimeShifter extends AbstractDragGestureEditor {

		private final List<Layer> layers = Util.newList();

		private DragGestureLayerTimeShifter(MouseEvent event, Time baseTime) {
			super(event, baseTime);
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					layers.add(((LayerElement) element).layer);
				}
			}
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation) {
			return new ShiftLayerTimesOperation(
					projectManager, composition, layers,
					time.subtract(prevTime), relation);
		}

	}

	private class DragGestureKeyframeShifter extends AbstractDragGestureEditor {

		private final Object[][] selectedKeyframeData;

		private final String[] baseAnimatableValues;


		private DragGestureKeyframeShifter(MouseEvent event, Time baseTime) {
			super(event, baseTime);
			selectedKeyframeData = createSelectedKeyframeData();
			baseAnimatableValues = createBaseAnimatableValues(selectedKeyframeData);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation) {
			return new ShiftKeyframesOperation(
					projectManager, composition, selectedKeyframeData, baseAnimatableValues,
					time.subtract(baseTime), relation);
		}

	}

}
