/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;
import java.util.Set;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.NativeJavaArray;
import org.mozilla.javascript.ScriptRuntime;
import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.plugin.PIAnimatableVec2d;
import ch.kuramo.javie.core.exprelems.DoubleProperty;

public class AnimatableVec2d extends AbstractAnimatableValue<Vec2d> implements PIAnimatableVec2d {

	private static final Set<Interpolation> SUPPORTED_INTERPOLATIONS = Util.unmodifiableSet(
			Interpolation.HOLD, Interpolation.LINEAR, Interpolation.CATMULL_ROM);


	public AnimatableVec2d(Vec2d staticValue, Collection<Keyframe<Vec2d>> keyframes, String expression) {
		super(staticValue, keyframes, expression);
	}

	public AnimatableVec2d(Vec2d staticValue) {
		super(staticValue);
	}

	public Vec2d jsToJava(Object jsValue) {
		if (jsValue instanceof NativeJavaArray) {
			// Vec2dの値をそのまま参照した場合 (例: thisComp.layer(3).audioLevels)
			// (Vec3dの値も参照可能)

			Object[] value = (Object[]) ((NativeJavaArray) jsValue).unwrap();
			double[] array = new double[2];
			for (int i = 0, n = Math.min(2, value.length); i < n; ++i) {
				array[i] = (Double) Context.jsToJava(value[i], double.class);
			}
			return valueOf(array);

		} else /*if (jsValue instanceof NativeArray)*/ {
			// 配列表現を用いた場合 (例: [time*10, 200])

			double[] array = (double[]) Context.jsToJava(jsValue, double[].class);
			if (array.length < 2) {
				double[] tmp = array;
				array = new double[2];
				for (int i = 0; i < tmp.length; ++i) {
					array[i] = tmp[i];
				}
			}
			return valueOf(array);
		}
	}

	@Override
	protected double[] toArray(Vec2d value) {
		return new double[] { value.x, value.y };
	}

	@Override
	protected Vec2d valueOf(double[] d) {
		return new Vec2d(d[0], d[1]);
	}

	@Override
	public Set<Interpolation> supportedInterpolations() {
		return SUPPORTED_INTERPOLATIONS;
	}

	public Scriptable[] createExpressionElement(final RenderContext renderContext) {
		DoubleProperty x = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).x; }
			public double valueAtTime(double t)	{ return AnimatableVec2d.this.valueAtTime(t, renderContext).x; }
		};

		DoubleProperty y = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).y; }
			public double valueAtTime(double t)	{ return AnimatableVec2d.this.valueAtTime(t, renderContext).y; }
		};

		return new Scriptable[] {
				renderContext.toNativeJavaObject(x, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(y, ScriptRuntime.NumberClass)
		};
	}

}
