/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Map;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

class CameraImpl implements Camera {

	private final VideoRenderContext _vrContext;

	private final VideoRenderSupport _vrSupport;

	private final CameraLayer _cameraLayer;

	private final Size2i _compSize;

	private final Size2i _viewportSize;

	private final double[] _mvMatrix2D;

	private final double[] _prjMatrix2D;

	private final Map<Time, double[][]> _matrix3DCache;


	CameraImpl(
			VideoRenderContext vrContext,
			VideoRenderSupport vrSupport,
			CameraLayer cameraLayer,
			Size2i compSize,
			Size2i viewportSize) {

		super();

		_vrContext = vrContext;
		_vrSupport = vrSupport;
		_cameraLayer = cameraLayer;
		_compSize = compSize;

		_viewportSize = viewportSize;
		_mvMatrix2D = new double[16];
		_prjMatrix2D = new double[16];
		_matrix3DCache = Util.newMap();

		createMatrix2D();
	}

	private void createMatrix2D() {
		// TODO この処理は VideoRenderSupport 内に持たすべきでは？

		GL gl = _vrContext.getGL();
		GLU glu = _vrContext.getGLU();

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, _viewportSize.width, 0, _viewportSize.height);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glLoadIdentity();

		_vrSupport.getMatrix(_mvMatrix2D, _prjMatrix2D);
	}

	public Size2i getViewportSize() {
		return _viewportSize;
	}

	public double[] getModelView2D() {
		return _mvMatrix2D;
	}

	public double[] getProjection2D() {
		return _prjMatrix2D;
	}

	public double[] getModelView3D() {
		return getMatrix3D()[0];
	}

	public double[] getProjection3D() {
		return getMatrix3D()[1];
	}

	private double[][] getMatrix3D() {
		Time time = _vrContext.getTime();
		double[][] matrix = _matrix3DCache.get(time);
		if (matrix != null) {
			return matrix;
		}

		matrix = _vrSupport.pushMatrixAndExecute(new WrappedOperation<double[][]>() {
			public double[][] execute() {
				Vec3d orientation, rotation, position, pointOfInterest;
				double fovy, near, far;
				double aspect = (double) _viewportSize.width / _viewportSize.height;

				if (_cameraLayer != null) {
					orientation = _cameraLayer.getOrientation().value(_vrContext);
					rotation = new Vec3d(
							_cameraLayer.getRotationX().value(_vrContext),
							_cameraLayer.getRotationY().value(_vrContext),
							_cameraLayer.getRotationZ().value(_vrContext));
					position = _cameraLayer.getPosition().value(_vrContext);
					pointOfInterest = _cameraLayer.getPointOfInterest().value(_vrContext);

					double zoom = _cameraLayer.getZoom().value(_vrContext);
					fovy = Math.toDegrees(2 * Math.atan(_compSize.height / (2*zoom)));
					near = _cameraLayer.getNear().value(_vrContext);
					far = _cameraLayer.getFar().value(_vrContext);

				} else {
					double zoom = getDefaultZoom();
					fovy = Math.toDegrees(2 * Math.atan(_compSize.height / (2*zoom)));

					// TODO デフォルトカメラのニアクリップ面、ファクリップ面はこれで妥当か？
					near = zoom / 2;
					far = zoom * 10;

					orientation = new Vec3d(0, 0, 0);
					rotation = new Vec3d(0, 0, 0);
					position = new Vec3d(_compSize.width/2.0, _compSize.height/2.0, -zoom);
					pointOfInterest = new Vec3d(position.x, position.y, 0);
				}

				RenderResolution resolution = _vrContext.getRenderResolution();
				position = resolution.scale(position);
				pointOfInterest = resolution.scale(pointOfInterest);
				near = resolution.scale(near);
				far = resolution.scale(far);


				GL gl = _vrContext.getGL();
				GLU glu = _vrContext.getGLU();

				gl.glMatrixMode(GL.GL_PROJECTION);
				gl.glLoadIdentity();
				glu.gluPerspective(fovy, aspect, near, far);

				gl.glMatrixMode(GL.GL_MODELVIEW);
				gl.glLoadIdentity();

				_vrSupport.multCameraMatrix(orientation, rotation, position, pointOfInterest, true);

				if (_cameraLayer != null) {
					LayerMatrixUtil.multParentMatrix(_cameraLayer, _vrContext, _vrSupport);
				}

				double[][] matrix3D = new double[2][16];
				_vrSupport.getMatrix(matrix3D[0], matrix3D[1]);

				return matrix3D;
			}
		});

		_matrix3DCache.put(time, matrix);
		return matrix;
	}

	private double getDefaultZoom() {
		// TODO デフォルトカメラの画角はこれで妥当か？ 50mmレンズの画角が39.6。
		double fovx = 39.6;

		// 画角fovxの画面内にコンポジションの横幅がちょうど収まるようにする。
		return _compSize.width / (2 * Math.tan(Math.toRadians(fovx/2)));
	}

	Vec3d getPosition() {
		return (_cameraLayer != null) ? _cameraLayer.getPosition().value(_vrContext)
				: new Vec3d(_compSize.width/2.0, _compSize.height/2.0, -getDefaultZoom());
	}

}
