/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;

import com.google.inject.Inject;

public class JavaSoundInput implements MediaFileInput {

	private static final Logger _logger = LoggerFactory.getLogger(JavaSoundInput.class);


	private File _file;

	private Time _duration;

	private AudioInputStream _stream;

	private long _position;

	private byte[] _buffer;


	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public JavaSoundInput() {
		super();
	}

	public boolean initialize(File file) {
		_file = file;

		try {
			AudioInputStream stream = AudioSystem.getAudioInputStream(file);
			try {
				long numFrames = stream.getFrameLength();
				float frameRate = stream.getFormat().getFrameRate();
				int frameRateInt = (int) frameRate;
				if (frameRate - frameRateInt != 0) {
					_logger.warn("frame rate is not integer: " + frameRate);
				}
				_duration = new Time(numFrames, frameRateInt);
			} finally {
				stream.close();
			}
		} catch (UnsupportedAudioFileException e) {
			return false;
		} catch (IOException e) {
			return false;
		}

		return true;
	}

	public void dispose() {
		closeStream();
	}

	private void closeStream() {
		if (_stream != null) {
			try {
				_stream.close();
			} catch (IOException e) {
				// ログだけ書き出して無視
				_logger.warn("failed to close AudioInputStream", e);
			}
			_stream = null;
		}
		_position = 0;
	}

	public synchronized AudioBuffer getAudioChunk() {
		try {
			Time time = _arContext.getTime();
			AudioMode audioMode = _arContext.getAudioMode();
			long frameNumber = time.toFrameNumber(audioMode.sampleDuration);

			if (frameNumber < _position) {
				closeStream();
			}

			if (_stream == null) {
				_stream = createAudioInputStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioInputStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		AudioMode audioMode = _arContext.getAudioMode();
		int bits = audioMode.sampleSize * 8;
		int ch = audioMode.channels;
		boolean bigEndian = ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN);

		AudioInputStream s1 = AudioSystem.getAudioInputStream(_file);

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, bigEndian);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(audioMode.sampleRate, bits, ch, true, bigEndian);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private AudioBuffer readAudioData(long frameNumber) throws IOException {
		AudioBuffer ab = _arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataLength = ab.getDataLength();
		int lengthInBytes = ab.getDataLengthInBytes();

		// TODO ストリームの末尾を超えてスキップしようとすると例外発生する？
		_stream.skip(audioMode.frameSize * (frameNumber - _position));

		if (_buffer == null || _buffer.length < lengthInBytes) {
			_buffer = new byte[lengthInBytes];
		}

		_stream.read(_buffer, 0, lengthInBytes);

		switch (audioMode.dataType) {
			case SHORT:
				ByteBuffer.wrap(_buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
						.asShortBuffer().get((short[]) data, 0, dataLength);
				break;
			case INT:
				ByteBuffer.wrap(_buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
						.asIntBuffer().get((int[]) data, 0, dataLength);
				break;
			case FLOAT:
				for (int i = 0; i < dataLength; ++i) {
					int value =  (_buffer[i*4  ] & 0xff)
							  | ((_buffer[i*4+1] & 0xff) <<  8)
							  | ((_buffer[i*4+2] & 0xff) << 16)
							  | ((_buffer[i*4+3] & 0xff) << 24);
					((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
				}
				break;
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + audioMode.dataType);
		}

		_position = frameNumber + ab.getFrameCount();

		return ab;
	}

	public boolean isVideoAvailable() {
		return false;
	}

	public boolean isAudioAvailable() {
		return true;
	}

	public Time getDuration() {
		return _duration;
	}

	public Time getVideoFrameDuration() {
		throw new UnsupportedOperationException("video is not available");
	}

	public VideoBounds getVideoFrameBounds() {
		throw new UnsupportedOperationException("video is not available");
	}

	public VideoBuffer getVideoFrameImage() {
		throw new UnsupportedOperationException("video is not available");
	}

}
