/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.SynchronousTaskThread;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;
import ch.kuramo.javie.core.services.SynchronousTaskThread.Task;
import ch.kuramo.javie.core.services.SynchronousTaskThread.TaskWithoutResult;

import com.google.inject.Inject;

public class WindowsDirectShowInput implements MediaFileInput {

	private static final Logger _logger = LoggerFactory.getLogger(WindowsDirectShowInput.class);


	private long _videoInput;

	private long _audioInput;

	private VideoBounds _bounds;

	private Time _duration;

	private Time _videoFrameDuration;

	private AudioMode _audioMode;

	@Inject
	private SynchronousTaskThread _videoThread;

	@Inject
	private SynchronousTaskThread _audioThread;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public WindowsDirectShowInput() {
		super();
	}

	public boolean initialize(final File file) {
		_videoThread.start();
		_videoThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeVideo(file);
			}
		});
		if (_videoInput == 0) {
			disposeVideo();
		}

		_audioThread.start();
		_audioThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeAudio(file);
			}
		});
		if (_audioInput == 0) {
			disposeAudio();
		}

		return (_videoInput != 0 || _audioInput != 0);
	}

	private void initializeVideo(File file) {
		long[] result = openVideo(file.getAbsolutePath());
		if (result != null) {
			_videoInput = result[0];
			_bounds = new VideoBounds((int) result[1], (int) result[2]);
			_duration = new Time(result[4], 10000000);
			_videoFrameDuration = new Time(result[4] / result[3], 10000000);
			_videoThread.setName(getClass().getSimpleName() + " (Video): " + file.getName());
		}
	}

	private void initializeAudio(File file) {
		long[] result = openAudio(file.getAbsolutePath());
		if (result != null) {
			_audioInput = result[0];
			if (_duration == null) {
				_duration = new Time(result[1], 10000000);
			}
			_audioThread.setName(getClass().getSimpleName() + " (Audio): " + file.getName());
		}
	}

	public void dispose() {
		disposeVideo();
		disposeAudio();
	}

	private void disposeVideo() {
		_videoThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_videoInput != 0) {
					closeVideo(_videoInput);
					_videoInput = 0;
					_bounds = null;
				}
			}
		});
	}

	private void disposeAudio() {
		_audioThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_audioInput != 0) {
					closeAudio(_audioInput);
					_audioInput = 0;
					_audioMode = null;
				}
			}
		});
	}

	public boolean isVideoAvailable() {
		return (_videoInput != 0);
	}

	public boolean isAudioAvailable() {
		return (_audioInput != 0);
	}

	public Time getDuration() {
		return _duration;
	}

	public Time getVideoFrameDuration() {
		return _videoFrameDuration;
	}

	public VideoBounds getVideoFrameBounds() {
		return _bounds;
	}

	public VideoBuffer getVideoFrameImage() {
		if (_videoInput == 0) {
			return null;
		}

		VideoBuffer vb = _vrSupport.createVideoBuffer(ColorMode.RGBA8, _bounds);
		try {
			vb.allocateAsArray();

			Time time = Time.fromFrameNumber(_vrContext.getTime().toFrameNumber(_videoFrameDuration), _videoFrameDuration);
			long timeValue = time.timeValue + _videoFrameDuration.timeValue/2;
			int timeScale = time.timeScale;
			final long dsTime = timeValue * 10000000 / timeScale;

			final byte[] array = (byte[]) vb.getArray();

			int error = _videoThread.invoke(new Task<Integer>() {
				public Integer run() throws Exception {
					int error = frameImageAtTime(_videoInput, dsTime, array);
					if (error != 0) {
						_logger.error("frameImageAtTime: error " + error);
					}
					return error;
				}
			});

			if (error != 0) {
				return null;
			}

			RenderResolution resolution = _vrContext.getRenderResolution();
			VideoBuffer vb2 = _vrSupport.createVideoBuffer(
					_vrContext.getColorMode(), resolution.scale(_bounds));

			_vrSupport.scaleAndFlipVertical(vb, vb2, resolution.scale(1));

			return vb2;

		} finally {
			vb.dispose();
		}
	}

	public AudioBuffer getAudioChunk() {
		if (_audioInput == 0) {
			return null;
		}

		Time time = _arContext.getTime();
		final long dsTime = time.timeValue * 10000000 / time.timeScale;
		final AudioBuffer ab = _arSupport.createAudioBuffer();

		int error = _audioThread.invoke(new Task<Integer>() {
			public Integer run() throws Exception {
				AudioMode am = ab.getAudioMode();
				if (am != _audioMode) {
					// TODO オーディオフォーマットの設定
					//int error = setAudioDescription(
					//		_audioInput, am.channels, am.sampleRate, am.sampleSize,
					//		(am.dataType == DataType.FLOAT));
					//if (error != 0) {
					//	_logger.error("setAudioDescription: error " + error);
					//	return error;
					//}
					//_audioMode = am;
				}

				int error = audioChunkFromTime(_audioInput, dsTime, ab.getData(), ab.getDataLengthInBytes());
				if (error != 0) {
					_logger.error("audioChunkFromTime: error " + error);
				}
				return error;
			}
		});

		if (error == 0) {
			return ab;
		} else {
			ab.dispose();
			return null;
		}
	}


	static { System.loadLibrary("DirectShowInput"); }

	private native long[] openVideo(String filename);

	private native long[] openAudio(String filename);

	private native void closeVideo(long videoInput);

	private native void closeAudio(long audioInput);

	private native int frameImageAtTime(
			long videoInput, long dsTime, byte[] buffer);

	private native int audioChunkFromTime(
			long audioInput, long dsTime, Object buffer, int chunkSize);

}
