/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.plugin.PIAnimatableBoolean;
import ch.kuramo.javie.api.plugin.PIAnimatableColor;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIAnimatableEnum;
import ch.kuramo.javie.api.plugin.PIAnimatableInteger;
import ch.kuramo.javie.api.plugin.PIAnimatableString;
import ch.kuramo.javie.api.plugin.PIAnimatableVec2d;
import ch.kuramo.javie.api.plugin.PIAnimatableVec3d;
import ch.kuramo.javie.core.AnimatableBoolean;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

class EffectDescriptorImpl implements EffectDescriptor {

	private final String _type;

	private final Class<?> _interiorClass;

	private final Class<Effect> _exteriorClass;

	private final Class<?> _exprInterface;

	private final ResourceBundle _bundle;

	private final String _category;

	private final String _label;

	private final Map<String, PropertyDescriptor> _propertyDescriptorMap;

	private final List<PropertyDescriptor> _propertyDescriptorList;


	EffectDescriptorImpl(String type, Class<?> interiorClass, Class<Effect> exteriorClass, Class<?> exprInterface) {
		super();
		_type = type;
		_interiorClass = interiorClass;
		_exteriorClass = exteriorClass;
		_exprInterface = exprInterface;
		_bundle = getBundle();
		_category = interiorClass.getAnnotation(ch.kuramo.javie.api.annotations.Effect.class).category().trim();
		_label = getString("label");
		_propertyDescriptorMap = Collections.unmodifiableMap(createPropertyDescriptors());
		_propertyDescriptorList = Collections.unmodifiableList(Util.newList(_propertyDescriptorMap.values()));
	}

	private ResourceBundle getBundle() {
		try {
			// javassistでいじったクラスだからか _interiorClass.getPackage() が
			// nullを返すようなので、クラス名からパッケージ名を切り出す。
			String className = _interiorClass.getName();
			String bundleName = className.substring(0, className.lastIndexOf('.') + 1) + "messages";

			return ResourceBundle.getBundle(
					bundleName, Locale.getDefault(), _interiorClass.getClassLoader());

		} catch (MissingResourceException e) {
			return null;
		}
	}

	private String getString(String key) {
		String str = null;
		if (_bundle != null) {
			try {
				str = _bundle.getString(String.format("%s.%s", _interiorClass.getSimpleName(), key));
			} catch (MissingResourceException e) {
				// ignore
			}
		}
		if (str == null) {
			str = '!' + key + '!';
		}
		return str;
	}

	private Map<String, PropertyDescriptor> createPropertyDescriptors() {
		Map<String, PropertyDescriptor> map = Util.newLinkedHashMap();

		for (Field f : _interiorClass.getDeclaredFields()) {
			Property anno = f.getAnnotation(Property.class);
			if (anno == null) {
				continue;
			}

			String name = f.getName();
			String label = getString(name + ".label");
			String value = anno.value();

			Class<?> type = f.getType();
			if (PIAnimatableDouble.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableDoublePropertyDescriptor(name, label, value));
			} else if (PIAnimatableInteger.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableIntegerPropertyDescriptor(name, label, value));
			} else if (PIAnimatableVec2d.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableVec2dPropertyDescriptor(name, label, value));
			} else if (PIAnimatableVec3d.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableVec3dPropertyDescriptor(name, label, value));
			} else if (PIAnimatableColor.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableColorPropertyDescriptor(name, label, value));
			} else if (PIAnimatableString.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableStringPropertyDescriptor(name, label, value));
			} else if (PIAnimatableBoolean.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableBooleanPropertyDescriptor(name, label, value));
			} else if (PIAnimatableEnum.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableEnumPropertyDescriptor(name, label, value));
			} else if (String.class.equals(type)) {
				map.put(name, new StringPropertyDescriptor(name, label, value));
			} else if (Enum.class.isAssignableFrom(type)) {
				@SuppressWarnings("unchecked")
				Class<Enum> enumType = (Class<Enum>) type;
				map.put(name, new EnumPropertyDescriptor(name, label, value, enumType));
			} else if (type.isPrimitive()) {
				map.put(name, new PrimitivePropertyDescriptor(name, label, value, type));
			} else {
				// 非サポートの型
				// EffectRegistryImpl#processProperties で警告を出しているはずなので無視
			}
		}

		return map;
	}

	public String getType() {
		return _type;
	}

	public Class<Effect> getEffectClass() {
		return _exteriorClass;
	}

	public Class<?> getExpressionInterface() {
		return _exprInterface;
	}

	public String getCategory() {
		return _category;
	}

	public String getLabel() {
		return _label;
	}

	public List<PropertyDescriptor> getPropertyDescriptors() {
		return _propertyDescriptorList;
	}

	public PropertyDescriptor getPropertyDescriptor(String name) {
		return _propertyDescriptorMap.get(name);
	}


	private <A extends AnimatableValue<?>> Class<A> getPropertyType(String propertyName) {
		try {
			String upperCamelCase = Character.toUpperCase(propertyName.charAt(0)) + propertyName.substring(1);
			@SuppressWarnings("unchecked")
			Class<A> propertyType = (Class<A>) _exteriorClass.getMethod("get" + upperCamelCase).getReturnType();
			return propertyType;
		} catch (NoSuchMethodException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private abstract class AbstractPropertyDescriptor<T> implements PropertyDescriptor {

		protected final Class<T> type;

		protected final String name;

		protected final String label;

		protected final Method getter;

		protected final Method setter;

		private AbstractPropertyDescriptor(Class<T> type, String name, String label) {
			super();
			this.type = type;
			this.name = name;
			this.label = label;
			try {
				String upperCamelCase = Character.toUpperCase(name.charAt(0)) + name.substring(1);
				getter = _exteriorClass.getDeclaredMethod("get" + upperCamelCase);
				setter = _exteriorClass.getDeclaredMethod("set" + upperCamelCase, type);
			} catch (NoSuchMethodException e) {
				throw new JavieRuntimeException(e);
			}
		}

		public Class<T> getPropertyClass() {
			return type;
		}

		public String getName() {
			return name;
		}

		public String getLabel() {
			return label;
		}

		public void afterDecode(Project p, Effect effect) throws ProjectDecodeException {
			// nothing to do
		}

		public T get(Effect effect) {
			try {
				@SuppressWarnings("unchecked")
				T value = (T) getter.invoke(effect);
				return value;
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}

		protected void set(Effect effect, T value) {
			try {
				setter.invoke(effect, value);
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}
	}

	private class AnimatableDoublePropertyDescriptor extends AbstractPropertyDescriptor<AnimatableDouble> {

		private final Double value;

		private AnimatableDoublePropertyDescriptor(String name, String label, String value) {
			super(AnimatableDouble.class, name, label);
			this.value = toDouble(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableDouble(value));
		}
	}

	private class AnimatableIntegerPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableInteger> {

		private final Integer value;

		private AnimatableIntegerPropertyDescriptor(String name, String label, String value) {
			super(AnimatableInteger.class, name, label);
			this.value = toInteger(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableInteger(value));
		}
	}

	private class AnimatableVec2dPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableVec2d> {

		private final Vec2d value;

		private AnimatableVec2dPropertyDescriptor(String name, String label, String value) {
			super(AnimatableVec2d.class, name, label);
			this.value = toVec2d(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableVec2d(value));
		}
	}

	private class AnimatableVec3dPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableVec3d> {

		private final Vec3d value;

		private AnimatableVec3dPropertyDescriptor(String name, String label, String value) {
			super(AnimatableVec3d.class, name, label);
			this.value = toVec3d(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableVec3d(value));
		}
	}

	private class AnimatableColorPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableColor> {

		private final Color value;

		private AnimatableColorPropertyDescriptor(String name, String label, String value) {
			super(AnimatableColor.class, name, label);
			this.value = toColor(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableColor(value));
		}
	}

	private class AnimatableStringPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableString> {

		private final String value;

		private AnimatableStringPropertyDescriptor(String name, String label, String value) {
			super(AnimatableString.class, name, label);
			this.value = value;
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableString(value));
		}
	}

	private class AnimatableBooleanPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableBoolean> {

		private final Boolean value;

		private AnimatableBooleanPropertyDescriptor(String name, String label, String value) {
			super(AnimatableBoolean.class, name, label);
			this.value = toBoolean(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableBoolean(value));
		}
	}

	@SuppressWarnings("unchecked")
	private class AnimatableEnumPropertyDescriptor extends AbstractPropertyDescriptor {

		private final Class<Enum> enumType;

		private final Constructor<?> constructor;

		private final Enum value;

		private AnimatableEnumPropertyDescriptor(String name, String label, String value) {
			super(getPropertyType(name), name, label);
			try {
				ParameterizedType genericSuper = (ParameterizedType) type.getGenericSuperclass();
				enumType = (Class<Enum>) genericSuper.getActualTypeArguments()[0];
				constructor = type.getConstructor(enumType);
				this.value = toEnum(enumType, value);
			} catch (NoSuchMethodException e) {
				throw new JavieRuntimeException(e);
			}
		}

		public void initValue(Effect effect) {
			try {
				set(effect, constructor.newInstance(value));
			} catch (InstantiationException e) {
				throw new JavieRuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}
	}

	private class StringPropertyDescriptor extends AbstractPropertyDescriptor<String> {

		private final String value;

		private StringPropertyDescriptor(String name, String label, String value) {
			super(String.class, name, label);
			this.value = value;
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	@SuppressWarnings("unchecked")
	private class EnumPropertyDescriptor extends AbstractPropertyDescriptor<Enum> {

		private final Enum value;

		private EnumPropertyDescriptor(String name, String label, String value, Class<Enum> enumType) {
			super(enumType, name, label);
			this.value = toEnum(enumType, value);
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	@SuppressWarnings("unchecked")
	private class PrimitivePropertyDescriptor extends AbstractPropertyDescriptor {

		private final Object value;

		private PrimitivePropertyDescriptor(String name, String label, String value, Class<?> primitiveType) {
			super(primitiveType, name, label);
			this.value = toPrimitive(primitiveType, value);
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	private static Boolean toBoolean(String s) {
		return Boolean.valueOf(s.trim());
	}

	private static Double toDouble(String s) {
		try {
			return Double.valueOf(s.trim());
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Double.valueOf(0);
		}
	}

	private static Float toFloat(String s) {
		try {
			return Float.valueOf(s.trim());
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Float.valueOf(0);
		}
	}

	private static Long toLong(String s) {
		try {
			return Long.valueOf(s.trim());
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Long.valueOf(0);
		}
	}

	private static Integer toInteger(String s) {
		try {
			return Integer.valueOf(s.trim());
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Integer.valueOf(0);
		}
	}

	private static Short toShort(String s) {
		try {
			return Short.valueOf(s.trim());
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Short.valueOf((short) 0);
		}
	}

	private static Character toCharacter(String s) {
		try {
			return Character.valueOf((char) Integer.parseInt(s.trim()));
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Character.valueOf((char) 0);
		}
	}

	private static Byte toByte(String s) {
		try {
			return Byte.valueOf((byte) Integer.parseInt(s.trim()));
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			return Byte.valueOf((byte) 0);
		}
	}

	private static Vec2d toVec2d(String s) {
		double x, y;
		try {
			String[] array = s.trim().split("\\s*,\\s*");
			switch (array.length) {
				case 1:
					x = y = Double.valueOf(array[0]);
					break;
				default:
					x = Double.valueOf(array[0]);
					y = Double.valueOf(array[1]);
					break;
			}
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			x = y = 0;
		}
		return new Vec2d(x, y);
	}


	private static Vec3d toVec3d(String s) {
		double x, y, z;
		try {
			String[] array = s.trim().split("\\s*,\\s*");
			switch (array.length) {
				case 1:
					x = y = z = Double.valueOf(array[0]);
					break;
				case 2:
					x = Double.valueOf(array[0]);
					y = Double.valueOf(array[1]);
					z = 0;
					break;
				default:
					x = Double.valueOf(array[0]);
					y = Double.valueOf(array[1]);
					z = Double.valueOf(array[2]);
					break;
			}
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			x = y = z = 0;
		}
		return new Vec3d(x, y, z);
	}

	private static Color toColor(String s) {
		double r, g, b, a;
		try {
			String[] array = s.trim().split("\\s*,\\s*");
			switch (array.length) {
				case 1:
					r = g = b = Double.valueOf(array[0]);
					a = 1;
					break;
				case 2:
					r = g = b = Double.valueOf(array[0]);
					a = Double.valueOf(array[1]);
					break;
				case 3:
					r = Double.valueOf(array[0]);
					g = Double.valueOf(array[1]);
					b = Double.valueOf(array[2]);
					a = 1;
					break;
				default:
					r = Double.valueOf(array[0]);
					g = Double.valueOf(array[1]);
					b = Double.valueOf(array[2]);
					a = Double.valueOf(array[3]);
					break;
			}
		} catch (NumberFormatException e) {
			// TODO ログに警告を出す
			r = g = b = 0;
			a = 1;
		}
		return new Color(r, g, b, a);
	}

	private static <E extends Enum<E>> E toEnum(Class<E> enumType, String s) {
		try {
			return Enum.valueOf(enumType, s.trim());
		} catch (IllegalArgumentException e) {
			// TODO ログに警告を出す
		}
		try {
			@SuppressWarnings("unchecked")
			E first = (E) Array.get(enumType.getMethod("values").invoke(null), 0);
			return first;
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (NoSuchMethodException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private static Object toPrimitive(Class<?> primitiveType, String s) {
		if (primitiveType == boolean.class) {
			return toBoolean(s);
		} else if (primitiveType == double.class) {
			return toDouble(s);
		} else if (primitiveType == float.class) {
			return toFloat(s);
		} else if (primitiveType == long.class) {
			return toLong(s);
		} else if (primitiveType == int.class) {
			return toInteger(s);
		} else if (primitiveType == short.class) {
			return toShort(s);
		} else if (primitiveType == char.class) {
			return toCharacter(s);
		} else if (primitiveType == byte.class) {
			return toByte(s);
		} else {
			throw new JavieRuntimeException("unsupported type: " + primitiveType.getName());
		}
	}

}
