/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

public final class Time implements Comparable<Time> {

	public final long timeValue;

	public final int timeScale;

	private final double _second;

	private final int _hashCode;


	public Time(long timeValue, int timeScale) {
		super();
		this.timeValue = timeValue;
		this.timeScale = timeScale;

		_second = (double) timeValue / timeScale;
		_hashCode = (int) (timeValue * 1000 / timeScale);
	}

	public static Time fromFrameNumber(long frameNumber, Time frameDuration) {
		return new Time(frameNumber * frameDuration.timeValue, frameDuration.timeScale);
	}

	public long toFrameNumber(Time frameDuration) {
		if (this.timeScale == frameDuration.timeScale) {
			return this.timeValue / frameDuration.timeValue;
		} else {
			return this.timeValue * frameDuration.timeScale
						/ (this.timeScale * frameDuration.timeValue);
		}
	}

	public double toSecond() {
		return _second;
	}

	public boolean before(Time o) {
		return (compareTo(o) < 0);
	}

	public boolean after(Time o) {
		return (compareTo(o) > 0);
	}

	public int compareTo(Time o) {
		long diff;
		if (this.timeScale == o.timeScale) {
			diff = this.timeValue - o.timeValue;
		} else {
			diff = this.timeValue * o.timeScale - this.timeScale * o.timeValue;
		}
		return (diff > 0) ? 1 : (diff < 0) ? -1 : 0;
	}

	public Time add(Time addend) {
		if (addend.timeScale == timeScale) {
			return new Time(timeValue + addend.timeValue, timeScale);
		} else {
			return new Time(timeValue + addend.timeValue * timeScale / addend.timeScale, timeScale);
		}
	}

	public Time subtract(Time subtrahend) {
		if (subtrahend.timeScale == timeScale) {
			return new Time(timeValue - subtrahend.timeValue, timeScale);
		} else {
			return new Time(timeValue - subtrahend.timeValue * timeScale / subtrahend.timeScale, timeScale);
		}
	}

	@Override
	public int hashCode() {
		return _hashCode;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		return (compareTo((Time) obj) == 0);
	}

	@Override
	public String toString() {
		return "Time [" + timeValue + "/" + timeScale + "]";
	}

}
