package ch.kuramo.javie.app.actions;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.output.ProgressMonitor;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput.VideoCompressorDescriptor;
import ch.kuramo.javie.core.output.WindowsDirectShowOutput.VideoCompressorSettings;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class AVIOutputAction extends Action {

	private static VideoCompressorSettings defaultVCSettings;

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public AVIOutputAction(LayerCompositionView view) {
		super("AVI...");
		InjectorHolder.getInjector().injectMembers(this);

		this.view = view;

		setId(CommandIds.AVI_OUTPUT);
		setActionDefinitionId(CommandIds.AVI_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/avi_output.png"));
	}

	public void run() {
		IWorkbenchWindow window = getWindow();
		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
		if (pm == null) {
			return;
		}

		CompositionItem compItem = view.getCompositionItem();

		AVICompressorWizard wizard = new AVICompressorWizard(compItem.getComposition(), defaultVCSettings);
		WizardDialog dialog = new WizardDialog(window.getShell(), wizard);
		if (dialog.open() != WizardDialog.OK) {
			return;
		}

		VideoCompressorSettings vcompSettings = wizard.getVideoCompressorSettings();
		defaultVCSettings = vcompSettings;

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		File file = showSaveDialog(defaultFolder, compItem.getName() + ".avi");
		if (file != null) {
			if (file.exists() && !file.delete()) {
				// TODO エラーメッセージを表示する。
				//      あるいは、ユニークな名前の一時ファイルを作ってそこへ書き出し、後でリネームする。
				return;
			}
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), vcompSettings, file);
		}
	}

	private IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	private File showSaveDialog(File folder, String name) {
		String[] filterNames = new String[] { "AVI Files", "All Files (*)" };
		String[] filterExtensions = new String[] { "*.avi", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "AVI Files", "All Files (*.*)" };
			filterExtensions = new String[] { "*.avi", "*.*" };
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(name);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, VideoCompressorSettings vcompSettings, File file) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new OutputWithProgress((CompositionItem) copy.getItem(compItemId), vcompSettings, file));

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private class OutputWithProgress implements  IRunnableWithProgress {
		
		private final CompositionItem compItem;

		private final VideoCompressorSettings vcompSettings;

		private final File file;


		private OutputWithProgress(CompositionItem compItem, VideoCompressorSettings vcompSettings, File file) {
			this.compItem = compItem;
			this.vcompSettings = vcompSettings;
			this.file = file;
		}

		public void run(final IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			WindowsDirectShowOutput dsOut = InjectorHolder.getInjector().getInstance(WindowsDirectShowOutput.class);
			dsOut.setVideoCompressorSettings(vcompSettings);
			dsOut.doOutput(compItem.getComposition(), AudioMode.STEREO_48KHZ_INT16, file,
					new ProgressMonitor() {
						private int totalWork;
						private int currentWork;

						public void beginTask(int totalWork) {
							monitor.beginTask(String.format("書き出し: %s", compItem.getName()), totalWork);
							this.totalWork = totalWork;
						}

						public void currentWork(int currentWork) {
							if (currentWork > this.currentWork) {
								monitor.worked(currentWork - this.currentWork);
								monitor.subTask(String.format("フレーム: %d/%d", currentWork, totalWork));
								this.currentWork = currentWork;
							}
						}

						public void done() {
							monitor.done();
						}

						public boolean isCanceled() {
							return monitor.isCanceled();
						}
					});

			if (monitor.isCanceled()) {
				throw new InterruptedException();
			}
		}

	}

	public static boolean isAvailable() {
		try {
			Class.forName(WindowsDirectShowOutput.class.getName());
			return true;
		} catch (Throwable t) {
			return false;
		}
	}

}

class AVICompressorWizard extends Wizard {

	private final AVIVideoCompressorWizardPage _page;


	AVICompressorWizard(Composition composition, VideoCompressorSettings defaultVCSettings) {
		super();
		setWindowTitle("書き出し");
		_page = new AVIVideoCompressorWizardPage(composition, defaultVCSettings);
	}

	public void addPages() {
		addPage(_page);
	}

	public boolean performFinish() {
		return true;
	}

	VideoCompressorSettings getVideoCompressorSettings() {
		return _page.getVideoCompressorSettings();
	}

}

class AVIVideoCompressorWizardPage extends WizardPage {

	private final Composition _composition;

	private final List<VideoCompressorDescriptor> _compressors;

	private final Map<Integer, VideoCompressorSettings> _settingsCache = Util.newMap();

	private int _selectedIndex;

	private Combo _compressorCombo;

	private Text _qualityText;

	private Text _keyFrameText;

	private Text _PFramesText;

	private Text _bitRateText;

	private Text _windowSizeText;

	private Button _configButton;

	private Button _resetButton;

	private Button _aboutButton;

	private boolean _updatingControls;


	AVIVideoCompressorWizardPage(Composition composition, VideoCompressorSettings defaultVCSettings) {
		super("AVIVideoCompressorWizardPage", "ビデオの圧縮", null);
		setDescription("圧縮プログラムの選択と設定をします。");

		_composition = composition;

		_compressors = WindowsDirectShowOutput.listVideoCompressors(_composition);
		Collections.sort(_compressors, new Comparator<VideoCompressorDescriptor>() {
			public int compare(VideoCompressorDescriptor o1, VideoCompressorDescriptor o2) {
				return o1.friendlyName.compareTo(o2.friendlyName);
			}
		});
		_compressors.add(new VideoCompressorDescriptor(null, "非圧縮", 0, false, false, -1, -1, -1, 0));
		_selectedIndex = _compressors.size()-1;

		if (defaultVCSettings != null) {
			for (int i = 0, n = _compressors.size()-1; i < n; ++i) {
				VideoCompressorDescriptor desc = _compressors.get(i);
				if (desc.moniker.equals(defaultVCSettings.moniker)) {
					_settingsCache.put(i, defaultVCSettings.clone());
					_selectedIndex = i;
					break;
				}
			}
		}
	}

	VideoCompressorSettings getVideoCompressorSettings() {
		return _settingsCache.get(_selectedIndex);
	}

	public void createControl(Composite parent) {
		List<String> compressorNames = Util.newList();
		for (VideoCompressorDescriptor desc : _compressors) {
			compressorNames.add(desc.friendlyName);
		}

		GridBuilder gb = new GridBuilder(parent, 10, true);

							gb.hSpan(2).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "圧縮プログラム:");
		_compressorCombo = 	gb.hSpan(8).hAlign(SWT.FILL).hGrab().combo(SWT.READ_ONLY,
									compressorNames.toArray(new String[compressorNames.size()]));

							gb.hSpan(10).size(10, 10).composite(SWT.NULL);

							gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "圧縮の品質 (0～100):");
		_qualityText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(_compressorCombo).text(SWT.BORDER, "");
							gb.hSpan(2).size(10, 10).composite(SWT.NULL);
		_configButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "設定");

							gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "キーフレーム (0:無し):");
		_keyFrameText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(_qualityText).text(SWT.BORDER, "");
							gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレームごと");
		_resetButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "標準に戻す");

							gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "ｷｰﾌﾚｰﾑ毎のPﾌﾚｰﾑ:");
		_PFramesText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(_keyFrameText).text(SWT.BORDER, "");
							gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレーム");
		_aboutButton =		gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "バージョン情報");

							gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "データレート (0:無し):");
		_bitRateText =		gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(_PFramesText).text(SWT.BORDER, "");
							gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "KB/秒");
							gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "").setVisible(false);

							gb.hSpan(3).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "ウインドウサイズ:");
		_windowSizeText =	gb.hSpan(2).hAlign(SWT.FILL).hGrab().tabAfter(_bitRateText).text(SWT.BORDER, "");
							gb.hSpan(2).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "フレーム");
							gb.hSpan(3).hAlign(SWT.FILL).hGrab().button(SWT.PUSH, "").setVisible(false);

		ModifyListener modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (!_updatingControls) {
					updateSettings();
				}
			}
		};
		_qualityText.addModifyListener(modifyListener);
		_keyFrameText.addModifyListener(modifyListener);
		_PFramesText.addModifyListener(modifyListener);
		_bitRateText.addModifyListener(modifyListener);
		_windowSizeText.addModifyListener(modifyListener);

		_configButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				config();
			}
		});

		_resetButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				reset();
			}
		});

		_aboutButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				about();
			}
		});

		_compressorCombo.setVisibleItemCount(20);
		_compressorCombo.select(_selectedIndex);
		_compressorCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				_selectedIndex = _compressorCombo.getSelectionIndex();
				updateControls();
			}
		});
		updateControls();

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
	}

	private void updateText(Text control, boolean enabled, String text) {
		control.setEnabled(enabled);
		control.setText(enabled ? text : "");
	}

	private void updateControls() {
		_updatingControls = true;
		try {
			VideoCompressorDescriptor desc = _compressors.get(_selectedIndex);
			VideoCompressorSettings settings = _settingsCache.get(_selectedIndex);
			if (settings == null) {
				settings = new VideoCompressorSettings(desc);
				_settingsCache.put(_selectedIndex, settings);
			}

			updateText(_qualityText, desc.canQuality, String.valueOf((int)(settings.quality*100)));
			updateText(_keyFrameText, desc.canKeyFrame, String.valueOf(settings.keyFrameRate));
			updateText(_PFramesText, desc.canBFrame, String.valueOf(settings.PFramesPerKey));
			updateText(_bitRateText, desc.canCrunch, String.valueOf(settings.bitRate >> 13));
			updateText(_windowSizeText, desc.canWindow, String.valueOf(settings.windowSize));

			boolean enableReset = desc.canQuality | desc.canKeyFrame | desc.canBFrame
								| desc.canCrunch | desc.canWindow | desc.canConfigDialog;

			_configButton.setEnabled(desc.canConfigDialog);
			_resetButton.setEnabled(enableReset);
			_aboutButton.setEnabled(desc.canAboutDialog);

			setErrorMessage(null);
			setPageComplete(true);

		} finally {
			_updatingControls = false;
		}
	}

	private void updateSettings() {
		setPageComplete(false);

		VideoCompressorDescriptor desc = _compressors.get(_selectedIndex);
		VideoCompressorSettings settings = _settingsCache.get(_selectedIndex);

		double quality = settings.quality;
		if (desc.canQuality) {
			try {
				quality = Integer.parseInt(_qualityText.getText().trim());
			} catch (NumberFormatException e) {
				quality = -1;
			}
			if (quality < 0 || quality > 100) {
				setErrorMessage("圧縮の品質には0から100までの整数を入力してください。");
				return;
			} else {
				quality /= 100;
			}
		}

		int keyFrameRate = settings.keyFrameRate;
		if (desc.canKeyFrame) {
			try {
				keyFrameRate = Integer.parseInt(_keyFrameText.getText().trim());
			} catch (NumberFormatException e) {
				keyFrameRate = -1;
			}
			if (keyFrameRate < 0) {
				setErrorMessage("キーフレームには0以上の整数を入力してください。");
				return;
			}
		}

		int PFramesPerKey = settings.PFramesPerKey;
		if (desc.canBFrame) {
			try {
				PFramesPerKey = Integer.parseInt(_PFramesText.getText().trim());
			} catch (NumberFormatException e) {
				PFramesPerKey = -1;
			}
			if (PFramesPerKey < 0) {
				setErrorMessage("ｷｰﾌﾚｰﾑ毎のPﾌﾚｰﾑには0以上の整数を入力してください。");
				return;
			}
		}

		long bitRate = settings.bitRate;
		if (desc.canCrunch) {
			try {
				bitRate = Integer.parseInt(_bitRateText.getText().trim());
			} catch (NumberFormatException e) {
				bitRate = -1;
			}
			if (bitRate < 0) {
				setErrorMessage("データレートには0以上の整数を入力してください。");
				return;
			} else {
				bitRate = Math.min(bitRate << 13, Integer.MAX_VALUE);
			}
		}

		long windowSize = settings.windowSize;
		if (desc.canWindow) {
			try {
				windowSize = Long.parseLong(_windowSizeText.getText().trim());
			} catch (NumberFormatException e) {
				windowSize = 0;
			}
			if (windowSize <= 0) {
				setErrorMessage("ウインドウサイズには1以上の整数を入力してください。");
				return;
			}
		}

		settings.quality = quality;
		settings.keyFrameRate = keyFrameRate;
		settings.PFramesPerKey = PFramesPerKey;
		settings.bitRate = bitRate;
		settings.windowSize = windowSize;

		setErrorMessage(null);
		setPageComplete(true);
	}

	private void reset() {
		VideoCompressorDescriptor desc = _compressors.get(_selectedIndex);
		_settingsCache.put(_selectedIndex, new VideoCompressorSettings(desc));

		updateControls();
	}

	private void config() {
		VideoCompressorDescriptor desc = _compressors.get(_selectedIndex);
		VideoCompressorSettings settings = _settingsCache.get(_selectedIndex);

		WindowsDirectShowOutput.openVideoCompressorConfigDialog(
				_composition, desc, settings, getHWND());
	}

	private void about() {
		VideoCompressorDescriptor desc = _compressors.get(_selectedIndex);

		WindowsDirectShowOutput.openVideoCompressorAboutDialog(desc, getHWND());
	}

	private long getHWND() {
		try {
			Shell shell = getShell();
			Class<?> clazz = shell.getClass();
			Field handleField = clazz.getField("handle");
			return (Integer) /*(Long)*/ handleField.get(shell);
		} catch (NoSuchFieldException e) {
			throw new JavieRuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		}
	}

}
