/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.concurrent.atomic.AtomicReference;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;

import org.eclipse.core.runtime.ListenerList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.services.AudioRenderContext;

import com.google.inject.Inject;

public class AudioThread extends Thread {

	private static final Logger _logger = LoggerFactory.getLogger(AudioThread.class);


	private final SourceDataLine _line;

	private final TimeKeeper _timeKeeper;

	private final MediaItem _mediaItem;

	private final AtomicReference<Time> _newTime = new AtomicReference<Time>();

	private final ListenerList _playerThreadListeners = new ListenerList();

	private final Object _monitor = new Object();

	private volatile boolean _playing;

	private volatile boolean _finished;

	private volatile boolean _muted;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private IArrayPools _arrayPools;


	public AudioThread(MediaItem mediaItem) throws LineUnavailableException {
		super("AudioThread: " + mediaItem.getName());
		setPriority(getThreadGroup().getMaxPriority());

		MediaInput input = mediaItem.getMediaInput();
		if (input == null) {
			throw new IllegalArgumentException("no MediaInput is available");
		}
		if (!input.isAudioAvailable()) {
			throw new IllegalArgumentException("no audio is available");
		}

		AudioFormat lineFormat = new AudioFormat(48000, 16, 2, true, ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN));
		DataLine.Info lineInfo = new DataLine.Info(SourceDataLine.class, lineFormat);
		_line = (SourceDataLine) AudioSystem.getLine(lineInfo);
		_line.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 10));	// 0.1秒分のバッファサイズ
//		_line.open(lineFormat, lineFormat.getFrameSize() * (int) (lineFormat.getFrameRate() / 4));	// 0.25秒分のバッファサイズ

		_timeKeeper = TimeKeeper.fromAudioLine(_line);

		InjectorHolder.getInjector().injectMembers(this);
		_mediaItem = mediaItem;
	}

	public TimeKeeper getTimeKeeper() {
		return _timeKeeper;
	}

	public boolean isMuted() {
		return _muted;
	}

	public void setMuted(boolean muted) {
		_muted = muted;
	}

	public void run() {
		try {
			MediaInput input = _mediaItem.getMediaInput();
			Time duration = input.getDuration();

			// TODO プロジェクト設定などから取得
			AudioMode audioMode = AudioMode.STEREO_48KHZ_INT16;

			int frameCount = audioMode.sampleRate / 10;	// 1回あたり0.1秒分のフレームを処理する
//			int frameCount = audioMode.sampleRate / 4;	// 1回あたり0.25秒分のフレームを処理する


			_line.start();
			_arContext.activate();

			Time nextTime = Time.fromFrameNumber(0, audioMode.sampleDuration);

			while (waitForPlaying()) {

				Time time = _newTime.getAndSet(null);
				if (time != null) {
					_line.flush();
					_timeKeeper.setTime(time);
				} else {
					time = nextTime;
				}

				if (!time.before(duration)) {
					_line.drain();
					synchronized (_monitor) {
						_playing = false;
					}
					fireThreadEndOfDuration(time);
					continue;
				}

				AudioBuffer ab = null;

				if (!_muted) {
					_arContext.reset();
					_arContext.setAudioMode(audioMode);
					_arContext.setFrameCount(frameCount);
					_arContext.setEvaluationResolution(audioMode.sampleRate/100);

					if (_mediaItem instanceof CompositionItem) {
						// TODO 上の input.getDuration() も厳密にはロックする必要がある。
						PlayerLock.readLock().lock();
						try {
							// TODO prepareExpressionはプロジェクトに構造的な変更があった場合のみ行えばよい。
							Composition comp = ((CompositionItem) _mediaItem).getComposition();
							comp.prepareExpression(_arContext.createInitialExpressionScope(comp));

							ab = input.getAudioChunk(time);
						} catch (Exception e) {
							_logger.error("error getting audio chunk", e);
						} finally {
							PlayerLock.readLock().unlock();
						}
					} else {
						try {
							ab = input.getAudioChunk(time);
						} catch (Exception e) {
							_logger.error("error getting audio chunk", e);
						}
					}
				}

				if (ab != null) {
					write(ab);
					ab.dispose();
				} else {
					writeZero(frameCount);
				}

				// 現在のところ、要求したフレーム数分だけ必ず返す事になっているので、
				// frameCountの値の分だけ時間を進めればよいが、一応 AudioBuffer のフレーム数を見て時間を進める。
				nextTime = time.add(Time.fromFrameNumber(
						(ab != null) ? ab.getFrameCount() : frameCount, audioMode.sampleDuration));

				fireThreadRender(time);
			}
		} finally {
			_arContext.deactivate();
			_line.close();
		}
	}

	public void setTime(Time time) {
		_newTime.set(time);
	}

	public void play(boolean play) {
		if (_playing == play) {
			return;
		}

		synchronized (_monitor) {
			_playing = play;
			_monitor.notify();
		}
	}

	public void finish() {
		if (_finished) {
			return;
		}

		synchronized (_monitor) {
			_finished = true;
			_monitor.notify();
		}

		try {
			join();
		} catch (InterruptedException e) {
			// ignore
		}
	}

	private boolean waitForPlaying() {
		if (!_playing) {
			_line.flush();

			synchronized (_monitor) {
				while (!_finished && !_playing) {
					try {
						_monitor.wait();
					} catch (InterruptedException e) {
						// ignore
					}
				}
			}
		}
		return !_finished;
	}

	private void write(AudioBuffer ab) {
		AudioMode audioMode = ab.getAudioMode();
		int dataLen = ab.getDataLength();
		int lineDataLen = _line.getFormat().getFrameSize() * ab.getFrameCount();
		IArray<byte[]> pa = _arrayPools.getByteArray(lineDataLen);
		byte[] lineBuf = pa.getArray();

		switch (audioMode.dataType) {
			case SHORT:
				ByteBuffer.wrap(lineBuf, 0, lineDataLen).order(ByteOrder.nativeOrder())
						.asShortBuffer().put((short[]) ab.getData(), 0, dataLen);
				break;
			case INT: {
				int[] data = (int[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					lineBuf[i*2  ] = (byte) ((data[i] >>> 16) & 0xff);
					lineBuf[i*2+1] = (byte) ((data[i] >>> 24) & 0xff);
				}
				break;
			}
			case FLOAT: {
				float[] data = (float[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					short shortVal = (short) (Math.min(Math.max(data[i], -1.0), 1.0) * Short.MAX_VALUE);
					lineBuf[i*2  ] = (byte) ((shortVal      ) & 0xff);
					lineBuf[i*2+1] = (byte) ((shortVal >>> 8) & 0xff);
				}
				break;
			}
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + audioMode.dataType);
		}

		if (audioMode.sampleRate == 96000) {
			for (int i = 0, n = dataLen/2; i < n; ++i) {
				short s1 = (short) ((lineBuf[i*2*2  ] & 0xff) | ((lineBuf[i*2*2+1] & 0xff) << 8));
				short s2 = (short) ((lineBuf[i*2*2+2] & 0xff) | ((lineBuf[i*2*2+3] & 0xff) << 8));
				short s = (short) ((s1 + s2) / 2);
				lineBuf[i*2  ] = (byte) ((s      ) & 0xff);
				lineBuf[i*2+1] = (byte) ((s >>> 8) & 0xff);
			}
			lineDataLen /= 2;
		}

		_line.write(lineBuf, 0, lineDataLen);
		pa.release();
	}

	private void writeZero(int frameCount) {
		IArray<byte[]> pa = _arrayPools.getByteArray(_line.getFormat().getFrameSize() * frameCount);
		pa.clear();
		_line.write(pa.getArray(), 0, pa.getLength());
		pa.release();
	}

	public void addPlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.add(listener);
	}

	public void removePlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.remove(listener);
	}

	private void fireThreadRender(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadRender(event);
		}
	}

	private void fireThreadEndOfDuration(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadEndOfDuration(event);
		}
	}

}
