/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;
import java.util.SortedMap;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ModifyLayerKeyframesOperation<V> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final String _property;

	private final List<Keyframe<V>> _oldKeyframes = Util.newList();

	private final List<Keyframe<V>> _newKeyframes = Util.newList();


	public ModifyLayerKeyframesOperation(
			ProjectManager projectManager, Layer layer,
			String property, Time[] times, V[] values, String label) {

		super(projectManager, label);

		if (times.length != values.length) {
			throw new IllegalArgumentException();
		}

		LayerComposition comp = projectManager.checkLayer(layer);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(layer, property);

		_compId = comp.getId();
		_layerId = layer.getId();
		_property = property;

		for (int i = 0; i < times.length; ++i) {
			Time t = times[i];
			V v = values[i];
			Keyframe<V> oldKf = avalue.getKeyframe(t);
			Keyframe<V> newKf = new Keyframe<V>(t, v, defaultInterpolation(avalue, oldKf, t));
			_oldKeyframes.add(oldKf);
			_newKeyframes.add(newKf);
		}
	}

	// TODO ModifyEffectKeyframeOperation などにも同じメソッドがあるので整理する。
	private Interpolation defaultInterpolation(AnimatableValue<V> avalue, Keyframe<V> oldKeyframe, Time time) {
		if (oldKeyframe != null) {
			return oldKeyframe.interpolation;
		} else {
			SortedMap<Time, Keyframe<V>> head = avalue.headKeyframeMap(time);
			if (!head.isEmpty()) {
				return head.get(head.lastKey()).interpolation;
			} else {
				return avalue.getDefaultInterpolation();
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframe(false, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframe(true, project, pm);
		return Status.OK_STATUS;
	}

	private void replaceKeyframe(boolean undo, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Layer layer = comp.getLayer(_layerId);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(layer, _property);

		Object[][] data = new Object[_oldKeyframes.size()][];

		for (int i = 0, n = _oldKeyframes.size(); i < n; ++i) {
			Keyframe<V> target = undo ? _newKeyframes.get(i) : _oldKeyframes.get(i);
			Keyframe<V> replacement = undo ? _oldKeyframes.get(i) : _newKeyframes.get(i);

			if (replacement != null) {
				avalue.putKeyframe(replacement);
				data[i] = new Object[] { layer, -1, _property, replacement.time };
			} else {
				avalue.removeKeyframe(target.time);
				data[i] = new Object[] { layer, -1, _property, target.time };
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data);
		}
	}

}
