/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class NewLayerFromItemOperation extends ProjectOperation {

	private final String _compId;

	private final int _insertIndex;

	private final List<String> _layerData = Util.newList();

	private final List<String> _layerIds = Util.newList();

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public NewLayerFromItemOperation(
			ProjectManager projectManager, LayerComposition comp, int insertIndex, List<Item> items) {

		super(projectManager, "レイヤーの追加");
		InjectorHolder.getInjector().injectMembers(this);

		projectManager.checkComposition(comp);

		if (insertIndex < 0 || insertIndex > comp.getLayers().size()) {
			throw new IllegalArgumentException("insertIndex: " + insertIndex);
		}

		_compId = comp.getId();
		_insertIndex = insertIndex;

		Size2i compSize = comp.getSize();
		Vec3d compCenter = new Vec3d(compSize.width/2d, compSize.height/2d);

		for (Item item : items) {
			projectManager.checkItem(item);

			if (item instanceof CompositionItem
					&& checkCyclicComposition(comp, (CompositionItem) item)) {
				// TODO 循環したネストはできないことをユーザーに知らせる。
				continue;
			}

			if (item instanceof MediaItem) {
				MediaItem mediaItem = (MediaItem) item;
				MediaItemLayer layer = _elementFactory.newMediaItemLayer(mediaItem);
				if (layer == null) {
					// TODO MediaItemLayerの作成ができなかった場合どうする？
					//throw new JavieRuntimeException("can't create layer from item: " + item.getName());
					continue;
				}

				MediaInput input = mediaItem.getMediaInput();
				Time duration = input.getDuration();
				if (duration == null) {
					duration = comp.getDuration();
				}

				Time time0 = Time.fromFrameNumber(0, comp.getFrameDuration());
				layer.setStartTime(time0);
				layer.setInPoint(time0);
				layer.setOutPoint(duration);

				if (LayerNature.isVideoNature(layer)) {
					VideoBounds bounds = input.getVideoFrameBounds();
					layer.getAnchorPoint().reset(
							new Vec3d(bounds.width/2d + bounds.x, bounds.height/2d + bounds.y));
					layer.getPosition().reset(compCenter);
				}

				_layerData.add(_encoder.encodeElement(layer));
				_layerIds.add(layer.getId());

			} else if (item instanceof Folder) {
				// TODO フォルダの場合、PrecompositionLayerとして追加するか個別に追加するか。
			}
		}

		_noEffect = _layerData.isEmpty();
	}

	private boolean checkCyclicComposition(LayerComposition comp, CompositionItem compItem) {
		Composition otherComp = compItem.getComposition();

		if (otherComp == comp) {
			return true;
		}

		if (otherComp instanceof LayerComposition) {
			for (Layer layer : ((LayerComposition) otherComp).getLayers()) {
				if (layer instanceof MediaItemLayer) {
					MediaItem mi = ((MediaItemLayer) layer).getItem();
					if (mi instanceof CompositionItem
							&& checkCyclicComposition(comp, (CompositionItem) mi)) {
						return true;
					}
				}
			}
		}

		// TODO LayerComposition以外のCompositionを実装したときには
		//      そのCompositionがLayerCompositionを含んでいるケースをチェックする必要がある。
		//if (otherComp instanceof TrackComposition) {
		//}

		return false;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		List<Layer> newLayers = Util.newList();
		try {
			for (String data : _layerData) {
				newLayers.add(_decoder.decodeElement(data, Layer.class));
			}
			for (Layer layer : newLayers) {
				layer.afterDecode(project, comp);
			}
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding layer data", e);
		}

		comp.getLayers().addAll(_insertIndex, newLayers);

		if (pm != null) {
			pm.fireLayersAdd(comp, newLayers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		List<Layer> layers = comp.getLayers().subList(_insertIndex, _insertIndex + _layerIds.size());
		for (int i = 0; i < layers.size(); ++i) {
			if (!layers.get(i).getId().equals(_layerIds.get(i))) {
				throw new ExecutionException("composition structure is corrupted");
			}
		}

		List<Layer> removedLayers = Util.newList(layers);
		layers.clear();

		if (pm != null) {
			pm.fireLayersRemove(comp, removedLayers);
		}

		return Status.OK_STATUS;
	}

}
