/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ReparentItemsOperation extends ProjectOperation {

	private final Map<String, String> _items;

	private final String _destFolderId;


	public ReparentItemsOperation(ProjectManager projectManager, Collection<Item> items, Folder destFolder) {
		super(projectManager, "アイテムの移動");
		projectManager.checkItem(destFolder);

		items = Util.newSet(items);

		// 親子ともに選択されているものは親だけ残して子は除去する。
		for (Iterator<Item> it = items.iterator(); it.hasNext(); ) {
			Item i = it.next();
			while ((i = i.getParent()) != null) {
				if (items.contains(i)) {
					it.remove();
					break;
				}
			}
		}

		_items = Util.newMap();
		_destFolderId = (destFolder != null) ? destFolder.getId() : null;

		_noEffect = true;

		for (Item i : items) {
			Folder parent = i.getParent();
			String parentId = (parent != null) ? parent.getId() : null;
			_items.put(i.getId(), parentId);

			// 対象のアイテム全てが元々destFolder内にあった場合、このオペレーションを実行しない。
			_noEffect &= (parent == destFolder);
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Set<Item> items = Util.newSet();
		Set<Item> related = Util.newSet();
		Folder destFolder = project.getItem(_destFolderId);

		for (Map.Entry<String, String> entry : _items.entrySet()) {
			Item item = project.getItem(entry.getKey());
			Folder oldParent = item.getParent();
			item.setParent(destFolder);
			items.add(item);
			related.add(oldParent);
		}

		if (pm != null) {
			if (destFolder == null) {
				pm.fireItemsReparent(items, null);
			} else {
				related.add(destFolder);
				pm.fireItemsReparent(items, related);
			}
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Set<Item> items = Util.newSet();
		Set<Item> related = Util.newSet();
		Folder destFolder = project.getItem(_destFolderId);

		for (Map.Entry<String, String> entry : _items.entrySet()) {
			Item item = project.getItem(entry.getKey());
			Folder oldParent = project.getItem(entry.getValue());
			item.setParent(oldParent);
			items.add(item);
			related.add(oldParent);
		}

		if (pm != null) {
			if (destFolder == null) {
				pm.fireItemsReparent(items, null);
			} else {
				related.add(destFolder);
				pm.fireItemsReparent(items, related);
			}
		}

		return Status.OK_STATUS;
	}

}
