/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.lang.reflect.InvocationTargetException;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.widgets.Combo;

import ch.kuramo.javie.app.views.ComboBoxViewerCellEditor;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableEnum;
import ch.kuramo.javie.core.JavieRuntimeException;

public class AnimatableEnumElementDelegate<E extends Enum<E>> extends AnimatableValueElementDelegate<E> {

	private final Class<E> enumType;

	private final E[] values;

	private final String[] texts;

	private ComboBoxViewerCellEditor editor;


	public AnimatableEnumElementDelegate(
			AnimatableValueElement element, String name, AnimatableEnum<E> avalue) {

		super(element, name, avalue);
		enumType = avalue.getEnumType();
		values = values();
		texts = texts();
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		drawValue(gc, x, y, height, focused, texts[value.ordinal()]);
	}

	protected boolean canDragGestureEdit() {
		return false;
	}

	protected E dragGesture(double dx, double dy) {
		throw new UnsupportedOperationException("drag gesture edit is not supported");
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return canInlineEdit;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new ComboBoxViewerCellEditor(element.viewer.getTree(), SWT.READ_ONLY);
				editor.setContenProvider(ArrayContentProvider.getInstance());
				editor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

				((Combo) editor.getControl()).addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						setCellEditorValue(LayerCompositionView.VALUE_COL, editor.getValue());
					}
				});
			}

			editor.setInput(null);
			editor.setLabelProvider(new EnumLabelProvider());
			editor.setInput(values());

			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return value;
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			modifyValue(enumType.cast(value));
		}
	}

	private E[] values() {
		try {
			@SuppressWarnings("unchecked")
			E[] values = (E[]) enumType.getMethod("values").invoke(null);
			return values;
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (NoSuchMethodException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private String[] texts() {
		String[] texts = new String[values.length];
		for (int i = 0; i < values.length; ++i) {
			texts[i] = '!' + values[i].name() + '!';
		}

		ResourceBundle bundle;
		try {
			bundle = ResourceBundle.getBundle(
					enumType.getPackage().getName() + ".messages",
					Locale.getDefault(), enumType.getClassLoader());
		} catch (MissingResourceException e) {
			return texts;
		}

		for (int i = 0; i < values.length; ++i) {
			try {
				texts[i] = bundle.getString(String.format(
						"%s.%s", enumType.getSimpleName(), values[i].name()));
			} catch (MissingResourceException e) {
				// ignore
			}
		}
		return texts;
	}

	private class EnumLabelProvider extends LabelProvider {
		public String getText(Object element) {
			return texts[enumType.cast(element).ordinal()];
		}
	}

}
