/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import net.arnx.jsonic.JSONHint;

import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.core.services.EffectRegistry;

import com.google.inject.Injector;

public abstract class AbstractEffect implements Effect {

	protected final Object _interior;

	protected final EffectDescriptor _descriptor;

	protected boolean _enabled;

	protected String _name;


	public AbstractEffect(Injector injector, EffectRegistry registry) {
		_interior = injector.getInstance(getInteriorClass());
		_descriptor = registry.getEffectDescriptor(getType());
		_enabled = true;
		_name = _descriptor.getLabel();

		for (PropertyDescriptor pd : _descriptor.getPropertyDescriptors()) {
			pd.initValue(this);
		}
	}

	protected abstract Class<?> getInteriorClass();

	public abstract String getType();


	@JSONHint(ignore=true)
	public EffectDescriptor getEffectDescriptor() {
		return _descriptor;
	}

	public boolean isEnabled() {
		return _enabled;
	}

	public void setEnabled(boolean enabled) {
		_enabled = enabled;
	}

	public String getName() {
		return _name;
	}

	public void setName(String name) {
		_name = name;
	}

	public void afterDecode(Project p) throws ProjectDecodeException {
		for (PropertyDescriptor pd : _descriptor.getPropertyDescriptors()) {
			pd.afterDecode(p, this);
		}
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.putExpressionElement("thisEffect", this);

		for (PropertyDescriptor pd : _descriptor.getPropertyDescriptors()) {
			if (Expressioner.class.isAssignableFrom(pd.getPropertyClass())) {
				scope.assignTo((Expressioner<?>) pd.get(this));
			}
		}
	}

	public Object createExpressionElement(final RenderContext renderContext) {
		InvocationHandler handler = new InvocationHandler() {
			public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
				String name = method.getName();
				if (!name.startsWith("get")) {
					return Scriptable.NOT_FOUND;
				}
				name = Character.toLowerCase(name.charAt(3)) + name.substring(4);

				PropertyDescriptor pd = _descriptor.getPropertyDescriptor(name);
				if (pd == null) {
					return Scriptable.NOT_FOUND;
				}

				Object value = pd.get(AbstractEffect.this);
				if (value instanceof Expressionee) {
					return renderContext.getExpressionElement((Expressionee) value);
				} else {
					return value;
				}
			}
		};

		Class<?> exprInterface = _descriptor.getExpressionInterface();
		return Proxy.newProxyInstance(exprInterface.getClassLoader(), new Class[] { exprInterface }, handler);
	}

}
