/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.Map.Entry;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Matrix4d;
import javax.vecmath.Quat4d;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.NativeJavaArray;
import org.mozilla.javascript.ScriptRuntime;
import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.IAnimatableVec3d;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.exprelems.DoubleProperty;

public class AnimatableVec3d extends ArithmeticalAnimatableValue<Vec3d> implements IAnimatableVec3d {

	private boolean _slerp;


	public AnimatableVec3d(Vec3d staticValue, Collection<Keyframe<Vec3d>> keyframes, String expression) {
		super(staticValue, keyframes, expression, Vec3d.NEGATIVE_INFINITY, Vec3d.POSITIVE_INFINITY);
	}

	public AnimatableVec3d(Vec3d defaultValue, boolean slerp) {
		super(defaultValue, Vec3d.NEGATIVE_INFINITY, Vec3d.POSITIVE_INFINITY);
		_slerp = slerp;
	}

	public AnimatableVec3d(Vec3d defaultValue) {
		this(defaultValue, false);
	}

	public AnimatableVec3d(Vec3d defaultValue, Vec3d minValue, Vec3d maxValue) {
		super(defaultValue, minValue, maxValue);
	}

	@Override
	public void copyConfigurationFrom(AbstractAnimatableValue<Vec3d> src) {
		super.copyConfigurationFrom(src);
		_slerp = ((AnimatableVec3d) src)._slerp;
	}

	@Override
	public Vec3d clamp(Vec3d value) {
		if (_slerp) {
			return new Vec3d(
					((value.x % 360) + 360) % 360,
					((value.y % 360) + 360) % 360,
					((value.z % 360) + 360) % 360);
		} else {
			return Vec3d.min(Vec3d.max(value, _minValue), _maxValue);
		}
	}

	@Override
	protected Vec3d interpolate(
			Time time, Keyframe<Vec3d> k1, Keyframe<Vec3d> k2, Iterator<Entry<Time, Keyframe<Vec3d>>> tail) {

		if (!_slerp) {
			return super.interpolate(time, k1, k2, tail);
		}

		switch (k1.interpolation) {
			case LINEAR:
				return valueOf(slerp(toArray(k1.value), toArray(k2.value), k1.time, k2.time, time));

			case CATMULL_ROM: {
				SortedMap<Time, Keyframe<Vec3d>> headMap = _keyframes.headMap(k1.time);
				Keyframe<Vec3d> k0 = !headMap.isEmpty() ? headMap.get(headMap.lastKey()) : k1;
				Keyframe<Vec3d> k3 = tail.hasNext() ? tail.next().getValue() : k2;

				double[] p0 = toArray(k0.value);
				double[] p1 = toArray(k1.value);
				double[] p2 = toArray(k2.value);
				double[] p3 = toArray(k3.value);

				// 遠回りな方の回転にならないようにする。
				prepareSlerpModeCatmullRom(p1, p0);
				prepareSlerpModeCatmullRom(p1, p2);
				prepareSlerpModeCatmullRom(p2, p3);

				return valueOf(catmullRom(p0, p1, p2, p3, k1.time, k2.time, time));
			}

			default:
				return super.interpolate(time, k1, k2, tail);
		}
	}

	private void prepareSlerpModeCatmullRom(double[] a, double[] b) {
		for (int i = 0; i < 3; ++i) {
			double d = (((b[i] - a[i]) % 360) + 360) % 360;
			if (d > 180) {
				d -= 360;
			}
			b[i] = a[i] + d;
		}
	}

	protected double[] slerp(double[] p1, double[] p2, Time time1, Time time2, Time time) {
		if (time.equals(time1)) return p1;
		if (time.equals(time2)) return p2;

		double s1 = time1.toSecond();
		double s2 = time2.toSecond();
		double s = time.toSecond();
		double t = (s - s1) / (s2 - s1);

		Matrix4d m1 = new Matrix4d();
		Matrix4d m2 = new Matrix4d();
		Matrix4d m = new Matrix4d();
		AxisAngle4d aa = new AxisAngle4d();

		m1.setIdentity();
		aa.set(1, 0, 0, Math.toRadians(p1[0]));
		m.set(aa);
		m1.mul(m);
		aa.set(0, 1, 0, Math.toRadians(p1[1]));
		m.set(aa);
		m1.mul(m);
		aa.set(0, 0, 1, Math.toRadians(p1[2]));
		m.set(aa);
		m1.mul(m);

		m2.setIdentity();
		aa.set(1, 0, 0, Math.toRadians(p2[0]));
		m.set(aa);
		m2.mul(m);
		aa.set(0, 1, 0, Math.toRadians(p2[1]));
		m.set(aa);
		m2.mul(m);
		aa.set(0, 0, 1, Math.toRadians(p2[2]));
		m.set(aa);
		m2.mul(m);

		Quat4d q1 = new Quat4d();
		Quat4d q2 = new Quat4d();
		q1.set(m1);
		q2.set(m2);

		q1.interpolate(q2, t);

		m.set(q1);

		double x = Math.toDegrees(Math.atan2(m.m12, m.m22));
		double y = Math.toDegrees(Math.asin(-m.m02));
		double z = Math.toDegrees(Math.atan2(m.m01, m.m00));

		// TODO なぜか正負を逆にしないとダメみたい？
		x = -x;
		y = -y;
		z = -z;

		// clampメソッドで同じことをやっているので、ここでやる必要はない。
		//if (x < 0) x += 360;
		//if (y < 0) y += 360;
		//if (z < 0) z += 360;

		return new double[] { x, y, z };
	}

	public Vec3d jsToJava(Object jsValue) {
		if (jsValue instanceof NativeJavaArray) {
			// anchorPoint や position などをそのまま参照した場合 (例: thisComp.layer(3).position)

			Object[] value = (Object[]) ((NativeJavaArray) jsValue).unwrap();
			double[] array = new double[3];
			for (int i = 0, n = Math.min(3, value.length); i < n; ++i) {
				array[i] = (Double) Context.jsToJava(value[i], double.class);
			}
			return valueOf(array);

		} else /*if (jsValue instanceof NativeArray)*/ {
			// 配列表現を用いた場合 (例: [time*10, 200, 0])

			double[] array = (double[]) Context.jsToJava(jsValue, double[].class);
			if (array.length < 3) {
				double[] tmp = array;
				array = new double[3];
				for (int i = 0; i < tmp.length; ++i) {
					array[i] = tmp[i];
				}
			}
			return valueOf(array);
		}
	}

	@Override
	protected double[] toArray(Vec3d value) {
		return new double[] { value.x, value.y, value.z };
	}

	@Override
	protected Vec3d valueOf(double[] d) {
		return new Vec3d(d[0], d[1], d[2]);
	}

	public Scriptable[] createExpressionElement(final RenderContext renderContext) {
		DoubleProperty x = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).x; }
			public double valueAtTime(double t)	{ return AnimatableVec3d.this.valueAtTime(t, renderContext).x; }
		};

		DoubleProperty y = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).y; }
			public double valueAtTime(double t)	{ return AnimatableVec3d.this.valueAtTime(t, renderContext).y; }
		};

		DoubleProperty z = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).z; }
			public double valueAtTime(double t)	{ return AnimatableVec3d.this.valueAtTime(t, renderContext).z; }
		};

		return new Scriptable[] {
				renderContext.toNativeJavaObject(x, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(y, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(z, ScriptRuntime.NumberClass)
		};
	}

}
