/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractItem;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.SolidColorItem;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("solidColor")
public class SolidColorItemImpl extends AbstractItem implements SolidColorItem {

	private Color _color;

	private Size2i _size;

	private VideoBounds _bounds;

	private final SolidColorInput _input = new SolidColorInput();

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;


	public SolidColorItemImpl() {
		super();
	}

	public void initialize(Color color, Size2i size) {
		super.initialize();
		setColor(color);
		setSize(size);
	}

	public void dispose() {
		// nothing to do
	}

	public Color getColor() {
		return _color;
	}

	public void setColor(Color color) {
		_color = color;
	}

	public Size2i getSize() {
		return _size;
	}

	public void setSize(Size2i size) {
		_size = size;
		_bounds = new VideoBounds(size.width, size.height);
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _input;
	}

	@JSONHint(ignore=true)
	public String getKind() {
		return "平面";
	}

	@Override
	public String toString() {
		return "color:" + _color + " size:" + _size;
	}


	private class SolidColorInput implements VectorMediaInput {

		public void dispose() {
			// nothing to do
		}

		public boolean isVideoAvailable() {
			return true;
		}
		
		public boolean isAudioAvailable() {
			return false;
		}

		public Time getDuration() {
			return null;
		}

		public Time getVideoFrameDuration() {
			return null;
		}

		public VideoBounds getVideoFrameBounds() {
			return _bounds;
		}

		public VideoBuffer getVideoFrameImage(Time mediaTime) {
			ColorMode colorMode = _vrContext.getColorMode();
			Size2i size = _vrContext.getRenderResolution().scale(_size);

			VideoBuffer vb = _vrSupport.createVideoBuffer(colorMode, size);
			vb.allocateAsTexture();

			_vrSupport.fill(vb, _color);

			return vb;
		}

		public void rasterize(VideoBuffer resultBuffer, double[] mvMatrix, double[] prjMatrix, Time mediaTime) {
			Size2i size = _vrContext.getRenderResolution().scale(_size);
			_vrSupport.fillRectCR(resultBuffer, mvMatrix, prjMatrix, size, _color);
		}

		public AudioBuffer getAudioChunk(Time mediaTime) {
			throw new UnsupportedOperationException("audio is not available");
		}

	}

}
