/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.SynchronousTaskThread;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;
import ch.kuramo.javie.core.services.SynchronousTaskThread.Task;
import ch.kuramo.javie.core.services.SynchronousTaskThread.TaskWithoutResult;

import com.google.inject.Inject;

public class WindowsDirectShowInput implements MediaFileInput {

	private static final Logger _logger = LoggerFactory.getLogger(WindowsDirectShowInput.class);


	private long _videoInput;

	private long _audioInput;

	private VideoBounds _bounds;

	private Time _duration;

	private Time _videoFrameDuration;

	private long _lastVideoFrameTime = Long.MAX_VALUE;

	private AudioFormat _audioFormat;

	private boolean _audioFloat;

	private AudioMode _audioMode;

	private AudioInputStream _audioStream;

	private long _audioStreamPosition;

	@Inject
	private SynchronousTaskThread _videoThread;

	@Inject
	private SynchronousTaskThread _audioThread;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private IArrayPools _arrayPools;


	public WindowsDirectShowInput() {
		super();
	}

	public boolean initialize(final File file) {
		_videoThread.start();
		_videoThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeVideo(file);
			}
		});
		if (_videoInput == 0) {
			disposeVideo();
		}

		_audioThread.start();
		_audioThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeAudio(file);
			}
		});
		if (_audioInput == 0) {
			disposeAudio();
		}

		return (_videoInput != 0 || _audioInput != 0);
	}

	private void initializeVideo(File file) {
		long[] result = openVideo(file.getAbsolutePath());
		if (result != null) {
			_videoInput = result[0];
			_bounds = new VideoBounds((int) result[1], (int) result[2]);
			_duration = new Time(result[4], 10000000);
			_videoFrameDuration = new Time(result[4] / result[3], 10000000);
			_lastVideoFrameTime = (result[3]-1)*result[4] / result[3];
			_videoThread.setName(getClass().getSimpleName() + " (Video): " + file.getName());
		}
	}

	private void initializeAudio(File file) {
		long[] result = openAudio(file.getAbsolutePath());
		if (result != null) {
			_audioInput = result[0];
			if (_duration == null) {
				_duration = new Time(result[1], 10000000);
			}
			_audioFormat = new AudioFormat(result[2], (int)result[3], (int)result[4], true, false);
			_audioFloat = (result[5] != 0);
			_audioThread.setName(getClass().getSimpleName() + " (Audio): " + file.getName());
		}
	}

	public void dispose() {
		disposeVideo();
		disposeAudio();
	}

	private void disposeVideo() {
		_videoThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_videoInput != 0) {
					closeVideo(_videoInput);
					_videoInput = 0;
					_bounds = null;
				}
			}
		});
	}

	private void disposeAudio() {
		closeAudioStream();

		_audioThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_audioInput != 0) {
					closeAudio(_audioInput);
					_audioInput = 0;
					_audioMode = null;
				}
			}
		});
	}

	private void closeAudioStream() {
		if (_audioStream != null) {
			try {
				_audioStream.close();
			} catch (IOException e) {
				// ログだけ書き出して無視
				_logger.warn("failed to close AudioInputStream", e);
			}
			_audioStream = null;
		}
		_audioStreamPosition = 0;
	}

	public boolean isVideoAvailable() {
		return (_videoInput != 0);
	}

	public boolean isAudioAvailable() {
		return (_audioInput != 0);
	}

	public Time getDuration() {
		return _duration;
	}

	public Time getVideoFrameDuration() {
		return _videoFrameDuration;
	}

	public VideoBounds getVideoFrameBounds() {
		return _bounds;
	}

	public VideoBuffer getVideoFrameImage(Time mediaTime) {
		if (_videoInput == 0) {
			return null;
		}

		VideoBuffer vb = _vrSupport.createVideoBuffer(ColorMode.RGBA8, _bounds);

		if (mediaTime.timeValue < 0 || !mediaTime.before(_duration)) {
			vb.allocateAsTexture();
			_vrSupport.clear(vb);
			return vb;
		}

		try {
			vb.allocateAsArray();

			Time time = Time.fromFrameNumber(mediaTime.toFrameNumber(_videoFrameDuration), _videoFrameDuration);
			long timeValue = time.timeValue + _videoFrameDuration.timeValue/2;
			int timeScale = time.timeScale;
			final long dsTime = timeValue * 10000000 / timeScale;

			final byte[] array = (byte[]) vb.getArray();

			int error = _videoThread.invoke(new Task<Integer>() {
				public Integer run() throws Exception {
					int error = frameImageAtTime(_videoInput, dsTime, array);
					if (error != 0) {
						_logger.error("frameImageAtTime: error " + error);
					}
					return error;
				}
			});

			if (error != 0) {
				return null;
			}

			RenderResolution resolution = _vrContext.getRenderResolution();
			VideoBuffer vb2 = _vrSupport.createVideoBuffer(
					_vrContext.getColorMode(), resolution.scale(_bounds));

			_vrSupport.scaleAndFlipVertical(vb, vb2, resolution.scale(1));

			return vb2;

		} finally {
			vb.dispose();
		}
	}

	public synchronized AudioBuffer getAudioChunk(Time mediaTime) {
		try {
			AudioMode audioMode = _arContext.getAudioMode();
			long frameNumber = mediaTime.toFrameNumber(audioMode.sampleDuration);

			if (-frameNumber >= _arContext.getFrameCount() || !mediaTime.before(_duration)) {
				AudioBuffer ab = _arSupport.createAudioBuffer();
				_arSupport.clear(ab);
				return ab;
			}

			if (audioMode != _audioMode || audioMode.frameSize*frameNumber != _audioStreamPosition) {
				closeAudioStream();
			}

			if (_audioStream == null) {
				_audioStream = createAudioInputStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioInputStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		_audioMode = _arContext.getAudioMode();
		int bits = _audioMode.sampleSize * 8;
		int ch = _audioMode.channels;

		long length = _duration.toFrameNumber(new Time(1, (int)_audioFormat.getFrameRate()));
		AudioInputStream s1 = AudioSystem.getAudioInputStream(_audioFormat, new AudioInputStream(
				new DSAudioInputStream(_audioFormat), _audioFormat, length));

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, false);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(_audioMode.sampleRate, bits, ch, true, false);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private AudioBuffer readAudioData(long frameNumber) throws IOException {
		AudioBuffer ab = _arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataOffset = 0;
		int dataLength = ab.getDataLength();
		int lengthInBytes = ab.getDataLengthInBytes();

		if (_audioStreamPosition == 0) {
			if (frameNumber > 0) {
				long skip = audioMode.frameSize * frameNumber;
				long skipped = _audioStream.skip((skip / audioMode.sampleRate) * audioMode.sampleRate);
				_audioStreamPosition = skipped;
				skip -= skipped;
				if (skip > 0) {
					IArray<byte[]> array = _arrayPools.getByteArray((int)skip);
					try {
						do {
							skipped = _audioStream.read(array.getArray(), 0, (int)skip);
							if (skipped != -1) {
								_audioStreamPosition += skipped;
								skip -= skipped;
							}
						} while (skipped != -1 && skip > 0);
					} finally {
						array.release();
					}
				}
				if (skip != 0) {
					_logger.warn(String.format("skip failed: frameNumber=%d, skip=%d", frameNumber, skip));
					closeAudioStream();
					_arSupport.clear(ab);
					return ab;
				}
			} else if (frameNumber < 0) {
				_arSupport.clear(ab, 0, (int)-frameNumber);
				dataOffset = (int)-frameNumber * audioMode.channels;
				dataLength -= dataOffset;
				lengthInBytes -= dataOffset * audioMode.sampleSize;
			}
		}

		IArray<byte[]> array = _arrayPools.getByteArray(lengthInBytes);
		try {
			byte[] buffer = array.getArray();

			int readBytes = 0;
			do {
				int n = _audioStream.read(buffer, readBytes, lengthInBytes - readBytes);
				if (n == -1) {
					Arrays.fill(buffer, readBytes, lengthInBytes, (byte)0);
					break;
				}
				readBytes += n;
				_audioStreamPosition += n;
			} while (readBytes < lengthInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asShortBuffer().get((short[]) data, dataOffset, dataLength);
					break;
				case INT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asIntBuffer().get((int[]) data, dataOffset, dataLength);
					break;
				case FLOAT:
					for (int i = dataOffset; i < dataLength; ++i) {
						int value =  (buffer[i*4  ] & 0xff)
								  | ((buffer[i*4+1] & 0xff) <<  8)
								  | ((buffer[i*4+2] & 0xff) << 16)
								  | ((buffer[i*4+3] & 0xff) << 24);
						((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
					}
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}
		} finally {
			array.release();
		}

		return ab;
	}


	static { System.loadLibrary("DirectShowInput"); }

	private native long[] openVideo(String filename);

	private native long[] openAudio(String filename);

	private native void closeVideo(long videoInput);

	private native void closeAudio(long audioInput);

	private native int frameImageAtTime(
			long videoInput, long dsTime, byte[] buffer);

	private native int audioChunkFromTime(
			long audioInput, long dsTime, Object buffer, int offset, int size);


	private class DSAudioInputStream extends InputStream {

		private final AudioFormat format;

		private long positionInBytes;


		private DSAudioInputStream(AudioFormat format) {
			super();
			this.format = format;
		}

		@Override
		public int read() throws IOException {
			byte[] b = new byte[1];
			return (read(b, 0, 0) != -1) ? b[0] : -1;
		}

		@Override
		public int read(final byte[] b, final int off, final int len) throws IOException {
			final int frameSize = format.getFrameSize();
			final float frameRate = format.getFrameRate();

			if (len % format.getFrameSize() != 0) {
				throw new IOException("can't read the length bytes that is not multiples of frameSize");
			}

			final long dsTime = (long)(positionInBytes * 10000000 / (frameSize * frameRate));

			// IMediaSeeking::SetPositionsで終端にかなり近いところ（おそらく最終フレームよりも後の時刻）
			// を指定するとメソッドから帰ってこなくなるようで、これはその対処。
			if (dsTime > _lastVideoFrameTime) {
				Arrays.fill(b, off, off+len, (byte)0);

			} else {
				int error = _audioThread.invoke(new Task<Integer>() {
					public Integer run() throws Exception {
						int error;

						// 読み込む量が少なすぎるとDirectShowから帰ってくるのに何故かものすごく時間がかかるので、
						// 必ず一定量以上を読み込むようにする。
						int leastLen = frameSize*(int)(frameRate/10);
						if (len < leastLen) {
							IArray<byte[]> array = _arrayPools.getByteArray(leastLen);
							error = audioChunkFromTime(_audioInput, dsTime, array.getArray(), 0, leastLen);
							System.arraycopy(array.getArray(), 0, b, off, len);
							array.release();
						} else {
							error = audioChunkFromTime(_audioInput, dsTime, b, off, len);
						}

						if (error != 0) {
							_logger.error("audioChunkFromTime: error " + error);
						}
						return error;
					}
				});
				if (error != 0) {
					throw new IOException("error=" + error);
				}

				if (_audioFloat) {
					for (int i = off; i < len; i += 4) {
						int bits =  (b[i  ] & 0xff)
								 | ((b[i+1] & 0xff) <<  8)
								 | ((b[i+2] & 0xff) << 16)
								 | ((b[i+3] & 0xff) << 24);
						int intValue = (int)(Float.intBitsToFloat(bits) * Integer.MAX_VALUE);
	
						b[i  ] = (byte)( intValue         & 0xff);
						b[i+1] = (byte)((intValue >>>  8) & 0xff);
						b[i+2] = (byte)((intValue >>> 16) & 0xff);
						b[i+3] = (byte)((intValue >>> 24) & 0xff);
					}
				}
			}

			positionInBytes += len;
			return len;
		}

		@Override
		public long skip(long n) throws IOException {
			long actualSkip = 0;
			if (n > 0) {
				int frameSize = format.getFrameSize();
				actualSkip = (n / frameSize) * frameSize; 
				positionInBytes += actualSkip;
			}
			return actualSkip;
		}

	}

}
