/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL;
import javax.media.opengl.GL2;
import javax.media.opengl.GLCapabilities;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLDrawableFactory;
import javax.media.opengl.GLException;
import javax.media.opengl.GLPbuffer;
import javax.media.opengl.GLProfile;
import javax.media.opengl.Threading;
import javax.media.opengl.awt.GLCanvas;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.services.GLGlobal;

public class GLGlobalImpl implements GLGlobal {

	private final ReentrantLock _lock = new ReentrantLock();

	private final String _initError;

	private final GLPbuffer _pbuffer;

	private final GLContext _sharedContext;

	private final String _vendor;

	private final boolean _intel;

	private final boolean _amdAti;

	private final boolean _nvidia;

	private final int _maxSamples;

	private final int _maxTextureImageUnits;

	private Queue<GLContext> _pool;

	private GLCapabilities _canvasCapabilities;


	public GLGlobalImpl() {
		GLProfile profile = GLProfile.get(GLProfile.GL2);

		GLCapabilities glcap = new GLCapabilities(profile);
		glcap.setHardwareAccelerated(true);
		glcap.setDoubleBuffered(false);
		glcap.setOnscreen(true);
		glcap.setPBuffer(false);

		setCanvasCapabilities(glcap);

		// 以下の２つ（とdoubleBuffered）は、Windowsでは createGLPbuffer の中で設定されるが
		// 他のプラットフォームでどうなっているか未確認なので設定しておく。
		glcap.setOnscreen(false);
		glcap.setPBuffer(true);

		GLPbuffer pbuf = null;
		GLException exception = null;
		try {
			pbuf = GLDrawableFactory.getFactory(profile).createGLPbuffer(glcap, null, 1, 1, null);
		} catch (GLException e) {
			exception = e;
		}

		final String[] strings = new String[1];
		final int[] integers = new int[2];

		if (pbuf != null) {
			_initError = null;
			_pbuffer = pbuf;
			_sharedContext = _pbuffer.getContext();

			Threading.invokeOnOpenGLThread(new Runnable() {
				public void run() {
					_sharedContext.makeCurrent();
					try {
						GL gl = _sharedContext.getGL();
						strings[0] = gl.glGetString(GL.GL_VENDOR);
						gl.glGetIntegerv(GL2.GL_MAX_SAMPLES, integers, 0);
						gl.glGetIntegerv(GL2.GL_MAX_TEXTURE_IMAGE_UNITS, integers, 1);
					} finally {
						_sharedContext.release();
					}
				}
			});
		} else {
			_initError = exception.getMessage();
			_pbuffer = null;
			_sharedContext = null;
			Arrays.fill(strings, "");
			Arrays.fill(integers, 0);
		}

		_vendor = strings[0];

		String lcVendor = _vendor.toLowerCase();
		_intel = lcVendor.contains("intel");
		_amdAti = lcVendor.contains("amd") || lcVendor.contains("ati");		// ベンダー名に"AMD"を返すものがあるのかどうか知らないが念のため。
		_nvidia = lcVendor.contains("nvidia");

		_maxSamples = integers[0];
		_maxTextureImageUnits = integers[1];
	}

	public ReentrantLock getGlobalLock() {
		return _lock;
	}

	public String getInitError() {
		return _initError;
	}

	public GLContext getSharedContext() {
		return _sharedContext;
	}

	public String getVendor() {
		return _vendor;
	}

	public boolean isIntel() {
		return _intel;
	}

	public boolean isAmdAti() {
		return _amdAti;
	}

	public boolean isNvidia() {
		return _nvidia;
	}

	public int getMaxSamples() {
		return _maxSamples;
	}

	public int getMaxTextureImageUnits() {
		return _maxTextureImageUnits;
	}

	public void enterPoolMode(final int poolSize) {
		if (_pool != null) {
			throw new IllegalStateException("already in pool mode");
		}
		if (poolSize <= 0) {
			throw new IllegalArgumentException("poolSize must be one or greater");
		}

		_pool = new LinkedList<GLContext>();

		Threading.invokeOnOpenGLThread(new Runnable() {
			public void run() {
				for (int i = 0; i < poolSize; ++i) {
					GLContext context = _pbuffer.createContext(_sharedContext);
					context.makeCurrent();
					context.release();
					_pool.offer(context);
				}
			}
		});
	}

	public GLContext createContext() {
		if (_pool != null) {
			GLContext context = _pool.poll();
			if (context == null) {
				throw new JavieRuntimeException("no GLContext is available");
			}
			return context;
		}
		return _pbuffer.createContext(_sharedContext);
	}

	public void destroyContext(GLContext context) {
		if (_pool != null) {
			_pool.offer(context);
		} else {
			context.destroy();
		}
	}

	public GLCapabilities getCanvasCapabilities() {
		return (GLCapabilities) _canvasCapabilities.clone();
	}

	public void setCanvasCapabilities(GLCapabilities capabilities) {
		_canvasCapabilities = (GLCapabilities) capabilities.clone();
	}

	public GLCanvas createCanvas() {
		return new GLCanvas(_canvasCapabilities, null, _sharedContext, null);
	}

}
