/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Set;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.NotFoundException;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;

class ModifiedClassLoader {

	private final Set<CtClass> _modifiedClasses;


	ModifiedClassLoader() {
		super();
		_modifiedClasses = Util.newSet();
	}

	void add(CtClass cc) {
		_modifiedClasses.add(cc);
	}

	void loadAll(ClassLoader cl) {
		try {
			while (!_modifiedClasses.isEmpty()) {
				// load の中で _modifiedClasses を変更しているので、毎回 Iterator の取得からやり直す。
				load(_modifiedClasses.iterator().next(), cl);
			}
		} catch (NotFoundException e) {
			throw new JavieRuntimeException(e);
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void load(CtClass cc, ClassLoader cl) throws NotFoundException, CannotCompileException {
		// 親クラスも修正されている場合は、親から先に toClass する必要がある。
		CtClass sc = cc.getSuperclass();
		if (sc != null) {
			load(sc, cl);
		}
		if (_modifiedClasses.contains(cc)) {
			cc.toClass(cl, null);
			_modifiedClasses.remove(cc);
		}
	}

}
