/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

public abstract class BlueGreenScreenKeyBase {

	private final IVideoEffectContext context;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;


	protected BlueGreenScreenKeyBase(
			IVideoEffectContext context, IShaderRegistry shaders, boolean blue) {

		this.context = context;

		if (blue) {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "BLUE_SCREEN_KEY_MASK_ONLY");
		} else {
			keyingProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY");
			maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "GREEN_SCREEN_KEY_MASK_ONLY");
		}

		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_HIGH");
	}

	protected IVideoBuffer doKeying(
			IAnimatableDouble threshold,
			IAnimatableDouble cutoff,
			IAnimatableBoolean maskOnly,
			IAnimatableEnum<Smoothing> smoothing) {

		double t = context.value(threshold) / 100;
		double c = context.value(cutoff) / 100;
		boolean mask = context.value(maskOnly);
		Smoothing sm = mask ? Smoothing.NONE : context.value(smoothing);

		double t_minus_c = t - c;
		if (t_minus_c == 0) {
			if (t == 0) {
				t_minus_c = Double.POSITIVE_INFINITY;
			} else if (t == 1) {
				t = Double.POSITIVE_INFINITY;
			}
		}

		final float ft = (float) t;
		final float ftc = (float) t_minus_c;

		IVideoBuffer input = null;
		IVideoBuffer buf = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}

			final int w = bounds.width;
			final int h = bounds.height;

			buf = context.createVideoBuffer(bounds);

			final GL2 gl = context.getGL().getGL2();

			VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buf.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

			final IShaderProgram program1 = mask ? maskOnlyProgram : keyingProgram;
			program1.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program1.getUniformLocation("texture"), 0);
					gl.glUniform1f(program1.getUniformLocation("threshold"), ft);
					gl.glUniform1f(program1.getUniformLocation("t_minus_c"), ftc);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			final IShaderProgram program2;
			switch (sm) {
				case LOW:  program2 = smoothingLowProgram; break;
				case HIGH: program2 = smoothingHighProgram; break;
				default:   program2 = null; break;
			}

			if (program2 != null) {
				IVideoBuffer tmp = input;
				input = buf;
				buf = tmp;

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buf.getTexture(), 0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

				program2.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program2.getUniformLocation("texture"), 0);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});
			}

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer output = buf;
			buf = null;
			return output;

		} finally {
			if (input != null) input.dispose();
			if (buf != null) buf.dispose();
		}
	}

}
