/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import java.nio.FloatBuffer;
import java.util.Arrays;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;
import ch.kuramo.javie.effects.blurSharpen.BlurUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.RadialWipe", category=Effect.TRANSITION)
public class RadialWipe {

	public enum TypeOfRadialWipe { CLOCKWISE, COUNTERCLOCKWISE, BOTH }


	private final IVideoEffectContext context;

	private final IArrayPools arrayPools;

	private final IShaderRegistry shaders;

	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property("0")
	private IAnimatableDouble startAngle;

	@Property
	private IAnimatableVec2d wipeCenter;

	@Property
	private IAnimatableEnum<TypeOfRadialWipe> typeOfWipe;

	@Property(min="0", max="500")
	private IAnimatableDouble feather;


	@Inject
	public RadialWipe(IVideoEffectContext context, IArrayPools arrayPools, IShaderRegistry shaders) {
		this.context = context;
		this.arrayPools = arrayPools;
		this.shaders = shaders;
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion)/100;
		if (completion == 0) {
			return null;
		} else if (completion == 1) {
			// TODO context.doPreviousEffect()をしなくてもVideoBoundsを取得できる方法が必要。
			//      context.getPreviousBounds()をここで呼ぶとIllegalStateExceptionが発生する。
			IVideoBuffer vb = context.doPreviousEffect();
			VideoEffectUtil.clearTexture(vb, context.getGL().getGL2());
			return vb;
		}

		double startAngle = context.value(this.startAngle);
		Vec2d wipeCenter = context.value(this.wipeCenter);
		TypeOfRadialWipe typeOfWipe = context.value(this.typeOfWipe);
		double feather = context.value(this.feather);

		RenderResolution resolution = context.getRenderResolution();
		wipeCenter = resolution.scale(wipeCenter);
		feather = resolution.scale(feather);


		// 時計回り／半時計回りどちらの場合でも startAngle < endAngle となるようにする。
		double endAngle;
		switch (typeOfWipe) {
			case COUNTERCLOCKWISE:
				startAngle -= 360*completion;
				break;
			case BOTH:
				startAngle -= 180*completion;
				break;
		}
		startAngle %= 360;
		if (startAngle < 0) startAngle += 360;
		endAngle = startAngle + 360*completion;


		IVideoBuffer inputBuffer = null;
		IVideoBuffer wipeBuffer = null;
		try {
			inputBuffer = context.doPreviousEffect();
			VideoBounds inputBounds = inputBuffer.getBounds();
			if (inputBounds.isEmpty()) {
				IVideoBuffer result = inputBuffer;
				inputBuffer = null;
				return result;
			}

			int wipeBoundsExpand = (int)Math.ceil(feather);
			VideoBounds wipeBounds = VideoEffectUtil.expandBounds(inputBounds, wipeBoundsExpand, true, true);
			wipeCenter = new Vec2d(wipeCenter.x - wipeBounds.x, wipeCenter.y - wipeBounds.y);


			// ワイプのマスクを作成し、元の画像とブレンドする。
			GL2 gl = context.getGL().getGL2();
			GLU glu = context.getGLU();

			wipeBuffer = createWipeBuffer(wipeBounds, wipeCenter, feather, startAngle, endAngle, gl, glu);

			int w = inputBounds.width;
			int h = inputBounds.height;

			VideoEffectUtil.ortho2D(gl, glu, w, h);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, inputBuffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, wipeBuffer.getTexture());
			gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

			gl.glEnable(GL2.GL_BLEND);
			gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_ALPHA, GL2.GL_ZERO, GL2.GL_SRC_ALPHA);

			gl.glBegin(GL2.GL_QUADS);
			gl.glColor4f(1, 1, 1, 1);
			gl.glTexCoord2f(wipeBoundsExpand, wipeBoundsExpand);
			gl.glVertex2f(0, 0);
			gl.glTexCoord2f(wipeBoundsExpand+w, wipeBoundsExpand);
			gl.glVertex2f(w, 0);
			gl.glTexCoord2f(wipeBoundsExpand+w, wipeBoundsExpand+h);
			gl.glVertex2f(w, h);
			gl.glTexCoord2f(wipeBoundsExpand, wipeBoundsExpand+h);
			gl.glVertex2f(0, h);
			gl.glEnd();

			gl.glDisable(GL2.GL_BLEND);
			gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer result = inputBuffer;
			inputBuffer = null;
			return result;

		} finally {
			if (wipeBuffer != null) wipeBuffer.dispose();
			if (inputBuffer != null) inputBuffer.dispose();
		}
	}

	private IVideoBuffer createWipeBuffer(
			VideoBounds wipeBounds, Vec2d wipeCenter, double feather,
			double startAngle, double endAngle, GL2 gl, GLU glu) {

		// TODO アルファチャンネルだけを持ったテクスチャがあれば良い。
		//      しかし、internalFormatを GL_ALPHA* にするとフレームバッファが動作しない？ (エラー1286となる)

		IVideoBuffer wipeBuffer = null;
		int[] wipeTex = null;
		try {
			wipeBuffer = context.createVideoBuffer(wipeBounds);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, wipeBuffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			wipeTex = new int[1];
			gl.glGenTextures(1, wipeTex, 0);
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, wipeTex[0]);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);
			gl.glEnable(GL2.GL_TEXTURE_1D);

			drawWipe(wipeBounds, wipeCenter, feather, startAngle, endAngle, false, gl, glu);

			gl.glPushAttrib(GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT);
			gl.glEnable(GL2.GL_BLEND);
			gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
			gl.glBlendEquation(endAngle - startAngle < 180 ? GL2.GL_MAX : GL2.GL_MIN);

			drawWipe(wipeBounds, wipeCenter, feather, startAngle, endAngle, true, gl, glu);

			gl.glPopAttrib();

			gl.glDisable(GL2.GL_TEXTURE_1D);
			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			if (feather == 0) {
				IVideoBuffer result = wipeBuffer;
				wipeBuffer = null;
				return result;
			} else {
				return blur(wipeBuffer, feather, gl, glu);
			}

		} finally {
			if (wipeTex != null) gl.glDeleteTextures(1, wipeTex, 0);
			if (wipeBuffer != null) wipeBuffer.dispose();
		}
	}

	private void drawWipe(
			VideoBounds wipeBounds, Vec2d wipeCenter, double feather,
			double startAngle, double endAngle, boolean secondPath, GL2 gl, GLU glu) {

		int w = wipeBounds.width;
		int h = wipeBounds.height;

		double angle = secondPath ? endAngle : startAngle;
		double radian = Math.toRadians(angle);
		double wipeLen = w*Math.abs(Math.cos(radian))+h*Math.abs(Math.sin(radian));
		double wipeWidth = h*Math.abs(Math.cos(radian))+w*Math.abs(Math.sin(radian));

		double x0, y0;
		switch (((int)angle / 90) % 4) {
			case 0: x0 = 0; y0 = 0; break;
			case 1: x0 = w; y0 = 0; break;
			case 2: x0 = w; y0 = h; break;
			case 3: x0 = 0; y0 = h; break;
			default: throw new Error();	// never reached here.
		}

		double sig = -Math.signum(Math.cos(radian));
		if (sig == 0) sig = 1;

		double tan = Math.tan(radian);
		double d = sig*(x0+tan*y0-wipeCenter.x-tan*wipeCenter.y)/Math.sqrt(1+tan*tan);

		// startAngleとendAngleの差が小さいとき、
		// ワイプの中心から反対側にマスク境界が突き出てしまうため、
		// 以下ではその対処をしている。
		if (secondPath) {
			double angleDiff = endAngle - startAngle;
			if (angleDiff < 5) {
				d -= 2*(1-angleDiff/5);
			} else if (angleDiff > 355) {
				d += 2*(1-(360-angleDiff)/5);
			}
		}

		IArray<float[]> wipeArray = arrayPools.getFloatArray((int)Math.ceil(wipeLen + 2*feather));
		try {
			float[] array = wipeArray.getArray();
			int arrayLen = wipeArray.getLength();

			double border = Math.min(Math.max(feather+d, 0), arrayLen);
			int borderInt = (int)border;

			Arrays.fill(array, 0, borderInt, secondPath ? 0 : 1);
			if (borderInt < arrayLen) {
				array[borderInt] = (float)(secondPath ? 1 - (border - borderInt) : border - borderInt);
			}
			if (borderInt+1 < arrayLen) {
				Arrays.fill(array, borderInt+1, arrayLen, secondPath ? 1 : 0);
			}

			FloatBuffer buffer = FloatBuffer.wrap(array, 0, arrayLen);

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA8, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA32F, arrayLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}

			VideoEffectUtil.ortho2D(gl, glu, w, h);
			gl.glMatrixMode(GL2.GL_MODELVIEW);
			gl.glTranslatef(w/2f, h/2f, 0);
			gl.glRotatef((float)(angle+90), 0, 0, 1);
			gl.glScalef((float)(wipeWidth/w), (float)((wipeLen + 2*feather)/h), 1);
			gl.glTranslatef(-w/2f, -h/2f, 0);

			float texCoord = (float)((wipeLen + 2*feather)/arrayLen);

			gl.glBegin(GL2.GL_QUADS);
			gl.glTexCoord1f(texCoord);
			gl.glVertex2f(0, 0);
			gl.glTexCoord1f(texCoord);
			gl.glVertex2f(w, 0);
			gl.glTexCoord1f(0);
			gl.glVertex2f(w, h);
			gl.glTexCoord1f(0);
			gl.glVertex2f(0, h);
			gl.glEnd();

		} finally {
			wipeArray.release();
		}
	}

	private IVideoBuffer blur(IVideoBuffer input, double blur, GL2 gl, GLU glu) {
		return blur(input, blur, true, true, true, true, gl, glu);
	}

	private IVideoBuffer blur(
			IVideoBuffer input, double blur, boolean repeatEdgePixels,
			boolean horz, boolean vert, boolean fast, GL2 gl, GLU glu) {

		// TODO 以下は GaussianBlur から丸々コピペしたもの。整理する。
		// TODO ぼかすのはアルファチャンネルだけでよい。
		// TODO ぼかすのは境界線付近だけでよい。

		VideoBounds inputBounds = input.getBounds();

		int sampleRatio = 1;
		if (fast) {
			VideoBounds bounds = inputBounds;
			for (int factor : BlurUtil.getDownSampleFactors(blur)) {
				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				}

				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);

				IVideoBuffer buf = context.createVideoBuffer(bounds);
				BlurUtil.doDownSample(input, buf, hFactor, vFactor, gl, glu, shaders);
				input.dispose();
				input = buf;

				sampleRatio *= factor;
			}
			blur /= sampleRatio;
		}

		int radius = (int) Math.ceil(blur);


		// 標準偏差の2.5倍の位置がぼかしの端となるようにする（これでだいたいAEと同じになる）
		double sigma = blur / 2.5;
		double sigmaSquare = sigma * sigma;

		float[] kernel = new float[radius*2+1];
		float sum = 0;
		for (int i = 1; i <= radius; ++i) {
			sum += 2 * (kernel[radius-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
		}
		kernel[radius] = 1 / (++sum);
		for (int i = 1; i <= radius; ++i) {
			kernel[radius+i] = (kernel[radius-i] /= sum);
		}


		IVideoBuffer buf1 = null, buf2 = null, buf3 = null;
		try {
			if (horz) {
				VideoBounds bounds = input.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, true, false);
				}

				buf1 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(input, buf1, true, radius, kernel, gl, glu, shaders);
			} else {
				buf1 = input;
				input = null;
			}

			if (vert) {
				VideoBounds bounds = buf1.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf1, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, false, true);
				}

				buf2 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(buf1, buf2, false, radius, kernel, gl, glu, shaders);
			} else {
				buf2 = buf1;
				buf1 = null;
			}

			if (sampleRatio != 1) {
				int hRatio = horz ? sampleRatio : 1;
				int vRatio = vert ? sampleRatio : 1;
				VideoBounds bounds;

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf2, gl);
					bounds = inputBounds;
				} else {
					bounds = buf2.getBounds();
					bounds = new VideoBounds(bounds.x*hRatio, bounds.y*vRatio, bounds.width*hRatio, bounds.height*vRatio);
				}

				buf3 = context.createVideoBuffer(bounds);
				BlurUtil.doUpSample(buf2, buf3, hRatio, vRatio, gl, glu);
			} else {
				buf3 = buf2;
				buf2 = null;
			}

			return buf3;

		} finally {
			if (input != null) {
				input.dispose();
			}
			if (buf1 != null) {
				buf1.dispose();
			}
			if (buf2 != null) {
				buf2.dispose();
			}
		}
	}

}
