/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ReorderLayersOperation extends ProjectOperation {

	private final String _compId;

	private final List<String> _oldOrder = Util.newList();

	private final List<String> _newOrder = Util.newList();

	private final List<String> _layerIds = Util.newList();


	public ReorderLayersOperation(ProjectManager projectManager, List<Layer> newOrder, List<Layer> layers) {
		super(projectManager, "レイヤーの移動");

		if (Util.newSet(newOrder).size() != newOrder.size()) {
			throw new IllegalArgumentException("dupulicated elements found in the order list");
		}
		if (Util.newSet(layers).size() != layers.size()) {
			throw new IllegalArgumentException("dupulicated elements found in the layer list");
		}

		LayerComposition comp = projectManager.checkLayer(newOrder.get(0));
		_compId = comp.getId();

		for (Layer l : newOrder) {
			if (projectManager.checkLayer(l) != comp) {
				throw new IllegalArgumentException("all layers must belong to same composition");
			}
			_newOrder.add(l.getId());
		}

		for (Layer l : comp.getLayers()) {
			_oldOrder.add(l.getId());
		}

		for (Layer l : layers) {
			if (projectManager.checkLayer(l) != comp) {
				throw new IllegalArgumentException("all layers must belong to same composition");
			}
			_layerIds.add(l.getId());
		}

		_noEffect = _newOrder.equals(_oldOrder);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		reorder(_newOrder, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		reorder(_oldOrder, project, pm);
		return Status.OK_STATUS;
	}

	private void reorder(List<String> ids, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);

		List<Layer> newOrder = Util.newList();
		for (String id : ids) {
			Layer layer = comp.getLayer(id);
			if (layer == null) {
				throw new ExecutionException("no layer found: id=" + id);
			}
			newOrder.add(layer);
		}

		List<Layer> layers = Util.newList();
		for (String id : _layerIds) {
			Layer layer = comp.getLayer(id);
			if (layer == null) {
				throw new ExecutionException("no layer found: id=" + id);
			}
			layers.add(layer);
		}

		comp.getLayers().clear();
		comp.getLayers().addAll(newOrder);

		if (pm != null) {
			pm.fireLayersReorder(comp, layers);
		}
	}

}
