/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.project.LayerSwitchOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ThreeDSwitchOperation;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.FrameBlend;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.TransformableLayer;

abstract class LayerSwitch implements Switch {

	protected final ProjectManager projectManager;

	protected final Layer layer;


	public LayerSwitch(ProjectManager projectManager, Layer layer) {
		super();
		this.projectManager = projectManager;
		this.layer = layer;
	}

	public boolean hasBorder() {
		return true;
	}

	protected Image getImage(String key) {
		return Activator.getDefault().getImageRegistry().get(key);
	}

}

class VideoSwitch extends LayerSwitch {

	public VideoSwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return LayerNature.isVideoEnabled(layer)
				? getImage(Activator.IMG_SWITCH_VIDEO) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new LayerSwitchOperation.Video(projectManager, layer));
	}

}

class AudioSwitch extends LayerSwitch {

	public AudioSwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return LayerNature.isAudioEnabled(layer)
				? getImage(Activator.IMG_SWITCH_AUDIO) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new LayerSwitchOperation.Audio(projectManager, layer));
	}

}

class ShySwitch extends LayerSwitch {

	private boolean aboutToSwitch;


	public ShySwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return getImage(layer.isShy() ^ aboutToSwitch
				? Activator.IMG_SWITCH_SHY : Activator.IMG_SWITCH_UNSHY);
	}

	public void mouseDown(MouseEvent event) {
		final Display display = event.display;

		Listener filter = new Listener() {
			public void handleEvent(Event event) {
				aboutToSwitch = false;
				display.removeFilter(SWT.MouseUp, this);
				display.removeFilter(SWT.Deactivate, this);

				projectManager.postOperation(
						new LayerSwitchOperation.Shy(projectManager, layer));
			}
		};

		aboutToSwitch = true;
		display.addFilter(SWT.MouseUp, filter);
		display.addFilter(SWT.Deactivate, filter);

		Rectangle r = (Rectangle) event.data;
		((Control) event.widget).redraw(r.x, r.y, r.width, r.height, false);
	}

}

class EffectsSwitch extends LayerSwitch {

	public EffectsSwitch(ProjectManager projectManager, EffectableLayer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return ((EffectableLayer) layer).isEffectsEnabled()
				? getImage(Activator.IMG_SWITCH_EFFECT) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new LayerSwitchOperation.Effects(projectManager, (EffectableLayer) layer));
	}

}

class CTCRSwitch extends LayerSwitch {

	public CTCRSwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return LayerNature.isCTCR(layer)
				? getImage(Activator.IMG_SWITCH_CTCR) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new LayerSwitchOperation.CTCR(projectManager, layer));
	}

}

class FrameBlendSwitch extends LayerSwitch {

	public FrameBlendSwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
//		switch (LayerNature.getFrameBlend(layer)) {
//			case FRAME_MIX: return getImage(Activator.IMG_SWITCH_FBLEND_FMIX);
//			case PIXEL_MOTION: return getImage(Activator.IMG_SWITCH_FBLEND_PMOTION);
//			default: return null;
//		}

		return (LayerNature.getFrameBlend(layer) == FrameBlend.FRAME_MIX)
				? getImage(Activator.IMG_SWITCH_FBLEND) : null;
	}

	public void mouseDown(MouseEvent event) {
		FrameBlend[] values = FrameBlend.values();
		FrameBlend oldValue = LayerNature.getFrameBlend(layer);
		FrameBlend newValue = values[(oldValue.ordinal() + 1) % values.length];

		projectManager.postOperation(
				new LayerSwitchOperation.FrameBlend(projectManager, layer, newValue));
	}

}

class MotionBlurSwitch extends LayerSwitch {

	public MotionBlurSwitch(ProjectManager projectManager, Layer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return LayerNature.isMotionBlurEnabled(layer)
				? getImage(Activator.IMG_SWITCH_MBLUR) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new LayerSwitchOperation.MotionBlur(projectManager, layer));
	}

}

class ThreeDSwitch extends LayerSwitch {

	public ThreeDSwitch(ProjectManager projectManager, TransformableLayer layer) {
		super(projectManager, layer);
	}

	public Image getImage() {
		return LayerNature.isThreeD(layer)
				? getImage(Activator.IMG_SWITCH_3D) : null;
	}

	public void mouseDown(MouseEvent event) {
		projectManager.postOperation(
				new ThreeDSwitchOperation(projectManager, (TransformableLayer) layer));
	}

}
